"""Module containing base configurations for metro products
"""
import logging
import time
from rest_api.products import product_specific_base

from rest_api.products.cpservices import access_anywhere
from rest_api.products.cpservices import instrument_time
from rest_api.products.cpservices import instrument_location
from rest_api.products.cpservices import update_stratasync_info
from rest_api.products.usc import instrument_sync_files
from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync

from rest_api.products import process_cdm_job_metro
from rest_api.products.config_schema_common import SubTypeRefInfoFormatter
from clientmgr import utils
from clientmgr.utils import ClientFactory

log = logging.getLogger(__name__)


class ProductSpecific(product_specific_base.ProductSpecific, SubTypeRefInfoFormatter):
    """Class for SmartClass 4800 product specific functions
    """
    JOB_MANAGER_DIR = '/user/job-manager/'
    JOB_FILE_EXT = '.job.json'
    TEMPLATE_FILE_EXT = '.tpl.json'
    instrument_info = None

    def __init__(self):
        super().__init__()
        
    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        file_sync.register_report_path(report_dir)
        file_sync.register_procedures_path(procedures_dir)
        ProductSpecific.JOB_MANAGER_DIR = jobs_dir
        cdm_sync.register_cdm_filepath(jobs_dir)        

    @staticmethod
    def get_access_anywhere_code(servername):
        return access_anywhere.create_code(servername)

    @staticmethod
    def make_instrument_info():
        if ProductSpecific.instrument_info != None:
            return ProductSpecific.instrument_info

        info_version = {'infoVersion':1}

        __version__ = '1.3.0'
        __files_api_version__ = 1
        __job_manager_api_version__ = 2
        __workflow_api_version__ = [1,2]
        __location_api_version__ = 1
        __datetime_api_version__ = 1

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [1]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'syncfiles',
                    'versions': [1]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'jobmanager',
                    'versions': [__job_manager_api_version__]
                },
                {
                   'name': 'workflow', 
                   'versions': __workflow_api_version__
                },
                {
                    'name': 'saa',
                    'versions': [1]
                }
            ]
        }
        #Handled attachment of these types from VMT
        supported_file_extensions = {
          'supportedJobFileExtensions': [ 'png', 'jpg', 'json' ]
        }

        ProductSpecific.instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()
        ProductSpecific.instrument_info['assetInfo'] = asset_info.make_asset_info()

        ProductSpecific.instrument_info.update(info_version)
        ProductSpecific.instrument_info.update(api_support)
        ProductSpecific.instrument_info.update(supported_file_extensions)

        return ProductSpecific.instrument_info

    @staticmethod
    def update_time(time_value, location):
        return instrument_time.update_local_time(time_value, location)

    @staticmethod
    def update_location(geolocation):
        latitude = geolocation['latitude']
        longitude = geolocation['longitude']
        altitude = geolocation['altitude']
        return instrument_location.update_location(latitude, longitude, altitude)

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        return update_stratasync_info.update_stratasync_info(tech_id, account_id, url)

    @staticmethod
    def get_stratasync_info():
        return update_stratasync_info.get_stratasync_info()

    @staticmethod
    def get_sync_list(sync_all):
        return instrument_sync_files.get_sync_list(sync_all)
        
    @staticmethod
    def get_sync_file_type(full_path):
        return instrument_sync_files.get_sync_file_type(full_path)
        
    @staticmethod
    def set_sync_list_ack(full_path):
        instrument_sync_files.set_sync_list_ack(full_path)

    @staticmethod
    def call_process_cdm_job():
        process_cdm_job_metro.main()
   
    @staticmethod
    def get_sync_file_info(filepath, base_dir):
        return instrument_sync_files.get_sync_file_info(filepath, base_dir)

    @staticmethod
    def get_instrument_login_config():
        return update_stratasync_info.get_stratasync_info()

    @staticmethod
    def update_test_data_side_effect(test_plan_index,job_status_ratio):
        try:
            if test_plan_index != -1:
                # This is either a 'Skip' or a 'Re-do', do not show the prompt
                utils.cmsend('127.0.0.1', 'jobstatus', 'showPrompt', test_plan_index, 10)
            utils.cmsend('127.0.0.1',
                'ctrlpanelsvc',
                'set',
                'JOBMANAGER_STATUS,{}/{}'.format(job_status_ratio['complete_count'],job_status_ratio['total_count']))
        except ClientFactory.ConnectionError:
            pass

    @staticmethod
    def update_loaded_job_side_effect(is_job_active,job_status_ratio):
        try:
            if is_job_active:
                utils.cmsend('127.0.0.1',
                    'ctrlpanelsvc',
                    'set',
                    'JOBMANAGER_STATUS,{complete_count}/{total_count}'.format(**job_status_ratio))
            else:
                utils.cmsend('127.0.0.1', 'ctrlpanelsvc', 'set', 'JOBMANAGER_STATUS,""')
        except ClientFactory.ConnectionError:
            pass

    @staticmethod
    def get_job_dir():
        return ProductSpecific.JOB_MANAGER_DIR
    @staticmethod
    def get_job_extension():
        return ProductSpecific.JOB_FILE_EXT

    @staticmethod
    def get_template_extension():
        return ProductSpecific.TEMPLATE_FILE_EXT

