"""
Module to interface with 5800v2 ISU via SCPI interface
"""
import sys
import time
import logging

try:
    from scpi import client
except ImportError:
    client = None

log = logging.getLogger(__name__)

class ScpiAdapter(object):
    """
    Class to connect to and manage the interface with the ISU
    """
    def __init__(self, host='127.0.0.1', port=8002): # Not sure this port number should be hard-wired
        self.host = host
        self.port = port
        self.isu = client.Client(self.host, self.port)

    @classmethod
    def initialize_adapter(cls, host='127.0.0.1', port=8002):
        """
        Factory method that will attempt to initialize the isu.  If the isu
        is successfully initialized a ScpiAdapter instance will be returned
        otherwise None will be returned

        Returns:
            adapter (ScpiAdapter): the intialized class or None
        """
        adapter = cls(host=host, port=port)
        if (adapter.wait_for_remote_control() and
                adapter.wait_for_isu_ready()):
            return adapter
        return None

    def wait_for_remote_control(self):
        """
        Method to wait 60 seconds to see the isu will repsond to remote
        control commands
        """
        start = current = time.time()
        while current - start < 60:
            if self.send_remote_command():
                return True
            time.sleep(2)
            current = time.time()
        return False

    def wait_for_isu_ready(self):
        """
        Method to wait 60 secons to see if isu is in the ready state
        """
        start = current = time.time()
        while current - start < 60:
            if self.isu_is_ready():
                return True
            time.sleep(2)
            current = time.time()
        return False

    def update_job_info(self, job_info):
        """
        Method to update the ISU with the four parameters in the job info dictionary

        Args:
            job_info (dict): dictionary with the job info
        """
        self.send_remote_command()
        customerName = ""
        try:
            customerName = job_info['workflow']['customerInfo']['company']
        except:
            pass
        workOrderId = ""
        try:
            workOrderId = job_info['workflow']['workOrderId']
        except:
            pass
        technicianId = ""
        try:
            technicianId = job_info['workflow']['techInfo']['techId']
        except:
            pass
        testLocation = ""
        workflowId = ""
        try:
            workflowId = job_info['workflow']['workflowId']
        except:
            pass

        self.isu.command(':JOB:CUST', [customerName])
        self.isu.command(':JOB:NUMB', [workOrderId])
        self.isu.command(':JOB:TECH', [technicianId])
        self.isu.command(':JOB:LOC',  [testLocation])
        self.isu.command(':JOB:WKFL', [str(workflowId)])

    def send_remote_command(self):
        """
        Method to attempt to send the remote control command to the ISU

        Returns:
            True if the command is successful, False otherwise
        """
        try:
            self.isu.command('*REM')
            return True
        except: #pylint: disable=bare-except
            log.info(sys.exc_info()) #log the error so narrow the exception
            return False

    def isu_is_ready(self):
        """
        Method to determine if the ISU is ready to accept commands

        Returns:
            True if the ISU is ready, False otherwise
        """
        self.send_remote_command()
        ready = self.isu.query('ISU:READy?')
        return ready == 'READY\n'
