"""
Module to interface with the ISU via a Scpi Adapter
"""
import logging
from threading import Thread

from rest_api.products.isu import job_manager_scpi
from rest_api.products.isu.job_manager_scpi import ScpiAdapter
from rest_api.api.job_manager.job_manager import JobManager
from rest_api.api.job_manager.plugin import JobManagerPlugin
log = logging.getLogger(__name__)

HAS_SCPI_MODULE = bool(job_manager_scpi.client)


def update_isu_in_thread_if_active(db_file, jobs_dir, port):
    """
    Method to update the isu in a thread if the job manager is in the active
    state.  Used at startup.
    """
    job_manager = JobManagerPlugin.job_manager
    cdm_job = job_manager.get_active_job()
    threaded_send_job_info_to_isu(cdm_job, port)


def send_job_info_to_isu(job_info, port):
    """
    Function to send job information to the ISU.  If the SCPI adapter cannot
    be initilized the function will do nothing.

    Args:
        job_info (dict): dictionary of the job information to send to the isu
    """
    if HAS_SCPI_MODULE:
        adapter = ScpiAdapter.initialize_adapter(port=port)
        if adapter:
            adapter.update_job_info(job_info)
        else:
            log.warning('could not initialize scpi adapter')

def threaded_send_job_info_to_isu(cdm_job, port):
    """
    Method to send informaion to the ISU in a thread at startup so that the initialization
    does not interfear with the main startup script
    """
    thread = Thread(target=send_job_info_to_isu, args=[cdm_job], kwargs={'port': port})
    thread.start()
