"""Module containing base REST API configurations for Fiber products
"""
import logging
import threading
import traceback
import json

from rest_api.products import product_specific_base

from rest_api.products.mts_services import access_anywhere
from rest_api.products.mts_services import instrument_time
from rest_api.products.mts_services import instrument_location
from rest_api.products.mts_services import update_stratasync_info
from rest_api.products.mts_services import instrument_sync_files
from rest_api.products.mts_services import instrument_upgrade
from rest_api.products.mts_services import update_asset_info
from rest_api.products.mts_services import instrument_jobs
from rest_api.products.config_schemas_fiber import CONFIG_SCHEMAS
from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync

#from rest_api.products.usc import sync_files

from rest_api.products import process_cdm_job_fiber
from rest_api.products.config_schema_common import SubTypeRefInfoFormatter


log = logging.getLogger(__name__)

timer_apply = True

WAIT_SECONDS = 30

class ProductSpecific(product_specific_base.ProductSpecific, SubTypeRefInfoFormatter):
    """Class for all Fiber product specific functions
    """
    def __init__(self):
        super().__init__()
        self.test_definitions = CONFIG_SCHEMAS
    
    def start_timer_refresh():
        def myTimer():
            global timer_apply
            if timer_apply:
                instrument_sync_files.get_sync_list(False)
                
            timer_apply = True
            threading.Timer(WAIT_SECONDS, myTimer).start()
        
        timer = threading.Timer(WAIT_SECONDS, myTimer)
        timer.start()
        
    def cancel_next_refresh():
        global timer_apply
        timer_apply = False
        
    
    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        file_sync.register_report_path(report_dir)
        file_sync.register_procedures_path(procedures_dir)
        cdm_sync.register_cdm_filepath(jobs_dir)
        
        ProductSpecific.start_timer_refresh()
        log.debug('## startup_side_effect (fiber_base): Jobs in %s', jobs_dir)


    @staticmethod
    def get_access_anywhere_code(servername):
        return access_anywhere.create_code(servername)

    @staticmethod
    def make_instrument_info():
        info_version = {'infoVersion':1}

        __version__ = '1.11.0' # Should we align this with latest Yocto project tag?
        __files_api_version__ = 1
        __job_manager_api_version__ = 2
        __location_api_version__ = 1
        __datetime_api_version__ = 1

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [1]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'syncfiles',
                    'versions': [1]
                },
                {
                    'name': 'signature',
                    'versions': [1]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
#                 {
#                     'name': 'jobmanager',
#                     'versions': [__job_manager_api_version__]
#                 },
                {
                    'name': 'saa',
                    'versions': [1]
                },
                {
                    'name': 'instrumentlogin',
                    'versions': [1]
                },
                {
                    'name': 'firmware',
                    'versions': [1]
                #},
                #{
                    #'name': 'options',
                    #'versions': [1] 
                #},
                #{
                #    'name': 'workflow', #TBD vs job_manager (ATT???)
                #    'versions': [1]
                }
            ]
        }


        instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()

        asset_info_usc = asset_info.make_asset_info()
        instrument_info['assetInfo'] = update_asset_info.update_asset_info(asset_info_usc)

        instrument_info.update(info_version)
        instrument_info.update(api_support)

        log.debug('## make_instrument_info: version                 = %s', __version__)
        log.debug('## make_instrument_info: files_api_version       = %s', __files_api_version__)
        log.debug('## make_instrument_info: location_api_version    = %s', __location_api_version__)
        log.debug('## make_instrument_info: datetime_api_version    = %s', __datetime_api_version__)
        log.debug('## make_instrument_info: api_support: %s', api_support)
        return instrument_info

    @staticmethod
    def update_time(time_value, location):
        return instrument_time.update_local_time(time_value, location)

    @staticmethod
    def update_location(geolocation):
        return instrument_location.update_location(geolocation)

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        return update_stratasync_info.update_stratasync_info(tech_id, account_id, url)
        
    @staticmethod
    def get_sync_list(sync_all):
        ProductSpecific.cancel_next_refresh()
        return instrument_sync_files.get_sync_list(sync_all)
        
    @staticmethod
    def set_sync_list_ack(full_path):
        instrument_sync_files.set_sync_list_ack(full_path)
        

    @staticmethod
    def get_sync_file_info(filepath, base_dir):
        return instrument_sync_files.get_sync_file_info(filepath, base_dir)
    
    @staticmethod
    def get_instrument_login_config():
        return update_stratasync_info.get_instrument_login_config()
    
    @staticmethod
    def get_instrument_login_credentials(loginConfig):
        asset_info_usc = asset_info.make_asset_info()
        
        fullLoginConfig = {
            'assetType': asset_info_usc.get('assetType', ''),
            'uniqueId': asset_info_usc.get('uniqueId', ''),
            'techId': loginConfig.get('techId', ''),
            'accountId': loginConfig.get('accountId', '')
        }
        return update_stratasync_info.get_instrument_login_credentials(fullLoginConfig)
        
    @staticmethod
    def reset_firmware_upgrade_status():
        instrument_upgrade.reset_firmware_upgrade_status()
        
    @staticmethod
    def process_firmware_upgrade_status():
        return instrument_upgrade.process_firmware_upgrade_status()
        
    @staticmethod
    def process_firmware_upgrade_post(request, Filename):
        return instrument_upgrade.process_firmware_upgrade_post(request, Filename)
        
    @staticmethod
    def process_firmware_upgrade_put(request):
        return instrument_upgrade.process_firmware_upgrade_put(request)

    @staticmethod
    def get_job_dir():
        return instrument_jobs.get_job_dir()

    @staticmethod
    def get_job_extension():
        return instrument_jobs.get_job_extension()

    @staticmethod
    def get_template_extension():
        return instrument_jobs.get_template_extension()

