"""Module containing code to connect a 5800 to the smart access anywhere server
The main function is create_code, which returns an ACCESS_ANYWHERE namedtupple
"""
import time
import clientmgr.utils
from clientmgr.utils import ClientFactory
import clientmgr.ctrlpanelsvc

from rest_api.products.access_anywhere import AccessAnywhere


NO_IP_ERROR_MESSAGE = ('Unable to connect to Smart Access Anywhere because '
                       'the instrument has no IP address. You may be able to '
                       'put this smartphone into wifi hotspot mode and '
                       'connect this instrument to it.')

NOT_CONNECTED_ERROR_MESSAGE = ('Unable to connect to Smart Access Anywhere '
                               'Server please check instrument IP connection '
                               'and try again.')


def create_code(servername):
    """Function to send a request to smart access anywhere ctrlpanelsvc to
       establish a connection to the smart access anywhere server.

    args:
        servername: hostname/address of clientmgr (usually '127.0.0.1')
    returns:
        named tuple:
            connectStatus: True if connected, False otherwise
            accessCode: If connection established returns code for SAA access,
                otherwise None is returned
            hasIpAddress: True if the instrument has an IP address or is in static mode,
                False otherwise
    """
    try:
        with clientmgr.utils.ClientFactory.build(
                servername,
                name='saaclient',
                auto_rename=True,
                timeout=30
        ) as mts_client:
            ctrlpanelsvc = clientmgr.ctrlpanelsvc.Ctrlpanelsvc(mts_client)

            has_ip_addr = instrument_has_ip_addr(ctrlpanelsvc)
            if not has_ip_addr:
                return_value = AccessAnywhere(
                    accessCode=None,
                    connectStatus=False,
                    hasIpAddress=has_ip_addr,
                    errorMessage=NO_IP_ERROR_MESSAGE,
                )
                return return_value

            ctrlpanelsvc.set_config('ACCESS_ANYWHERE', 'CONNECT')
            connect_status = _wait_for_connect(ctrlpanelsvc)
            magic_number = ctrlpanelsvc.get_config('ACCESS_ANYWHERE_MAGIC_NUMBER')

            if not connect_status:
                error_message = NOT_CONNECTED_ERROR_MESSAGE
            else:
                error_message = ''
    except ClientFactory.ConnectionError:
        magic_number = '0000000'
        connect_status = 'Failed'
        has_ip_addr = False
        error_message = 'Invalid access code'

    return_value = AccessAnywhere(
        accessCode=magic_number,
        connectStatus=connect_status,
        hasIpAddress=has_ip_addr,
        errorMessage=error_message
    )
    return return_value

def _wait_for_connect(ctrlpanelsvc):
    start = time.time()
    while not _connected(ctrlpanelsvc):
        if time.time() > start + 120:
            break
        time.sleep(1)

    return _connected(ctrlpanelsvc)

def _connected(ctrlpanelsvc):
    return ctrlpanelsvc.get_config('ACCESS_ANYWHERE_STATE') == "CONNECTED"


def instrument_has_ip_addr(ctrlpanelsvc):
    """Function to determine if the instrument has an ip address

    args:
        ctrlpanelsvc (Controlpanelsvc): instance that is used to determine if
            the instrument has an ip address or notify
    returns:
        bool: True if the insturment has an ip address or is in static mode
            False otherwise
    """
    ip_mode = ctrlpanelsvc.get_config("ETH_IPMODE")
    if ip_mode == 'STATIC':
        return True

    lan_ip_addr = ctrlpanelsvc.get_config('ETH_DHCP_IPADDR')
    wifi_ip_addr = ctrlpanelsvc.get_config('WIFI_DHCP_IPADDR')
    if lan_ip_addr != '?.?.?.?' or wifi_ip_addr is not None:
        return True

    return False


if __name__ == "__main__":
    create_code('10.12.59.118')
