"""
Module that contains schemas for the config information of different tests to
launch on the T-BERD/MTS-4000v2/2000v2/SmartOTDR product family
"""
import logging

from marshmallow import Schema, fields, post_load
from marshmallow.validate import OneOf

from .config_schema_common import ConfigBase, test_type_factory, config_factory

#pylint: disable=no-self-use
log = logging.getLogger(__name__)


class FiberInspectionSchema(Schema):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC 61300-3-35 2.0)",
        "SM APC (IEC 61300-3-35 2.0)",
        "SM PC&FiberStub(IEC 61300-3-35 2.0)",
        "MM (IEC 61300-3-35 2.0)",
        "Ribbon, SM APC (IEC 61300-3-35 2.0)",
        ""
    )

    test_type = test_type_factory('Fiber Inspection')
    profile = fields.Str(
        load_from='profile',
        dump_to='profile',
        description='The Inspection Probe profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )


class FiberInspectionCdmSchema(Schema):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC 61300-3-35 2.0)",
        "SM APC (IEC 61300-3-35 2.0)",
        "SM PC&FiberStub(IEC 61300-3-35 2.0)",
        "MM (IEC 61300-3-35 2.0)",
        "Ribbon, SM APC (IEC 61300-3-35 2.0)",
        ""
    )

    test_type = test_type_factory('fiberInspection')
    profileName = fields.Str(
        load_from='profileName',
        dump_to='profileName',
        description='The Inspection Probe profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )


class OTDRSchema(Schema):
    """Schema for the configuration of an OTDR test
    """
    test_type = test_type_factory('OTDR')
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
        'bidirOtdr',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Str(
        load_from='alarms',
        dump_to='alarms',
        description='The name of the alarms standard to be used for pass fail criteria'
    )
    """TBD: Does this really need to exist as well as the more general .cfg/.prj file definition ?"""
    labelFile = fields.Str(
        load_from='labelFile',
        dump_to='labelFile',
        description='The path to the project (.prj) file containing Fiber labels'
    )
    fiber_type = fields.Str(
        required=False,
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )
    config_file = config_factory('/user/disk/')



class ManualSchema(Schema):
    """
    Schema for Generic Manual Steps in a Test Plan.
    """
    test_type = test_type_factory('manualStep')



class CdmOTDRSchema(Schema):
    """Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
        'bidirOtdr',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        required=False,
        allow_none=True,
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='alarmThresholds',
        dump_to='alarmThresholds',
        description='The alarm thresholds'
    )
    config_file = config_factory('/user/disk/')
    setups = fields.Nested(
        Schema,
        missing=[],
        required=False,
        allow_none=True,
        load_from='setup',
        dump_to='setup',
        description='The setup(s)'
    )
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )
    fiber_type = fields.Str(
        required=False,
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )



class OTDRSettingsSchema(Schema):#CdmOTDRSchema):
    """Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    test_type = 'OTDR'

    otdr_settings = fields.Nested(
        CdmOTDRSchema,
        required=True,
        missing={},
        load_from='otdrSettings',
        dump_to='otdrSettings',
        description='The CDM configuration structure for OTDRs'
    )

class OpenSchema(Schema):
    """
    Schema to leave any config schema checks outside of this scope
    """
    test_type = test_type_factory('fromCdm')

    @post_load(pass_original=True)
    def _add_unknown(self, data, original):
        """Preserve unknown keys during deserialization."""
        for key, val in original.items():
            if key not in self.fields:
                data[key] = val
        return data

class FcompProIlOrlSchema(Schema):
    """
    Cdm Schema for FCompPRO bidir Il+Orl measurements
    """
    meas_seq_choices = (
        'IL+ORL',
        'IL',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    test_type = test_type_factory('bidirIlOrl')
    meas_sequence = fields.Str(
        validate=OneOf(meas_seq_choices),
        load_from='measurementSequence',
        dump_to='measurementSequence',
        description='The IL/ORL measurement sequence to be used'
    ) 
    fiber_type = fields.Str(
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )
    config_file = config_factory('/user/disk/')


class FcompProMainSchema(Schema):
    """
    Cdm Schema for full FCompPRO sequence measurements
    """
    tpro_meas_seq_choices = (
        'IL+ORL',
        'IL',
        'bidirOtdr',
        'loopbackOtdr',
    )
    fiber_types = (
        'Simplex',
        'Duplex',
        'MPO',
    )
    test_type = test_type_factory('fcompPro') # Don't want this here but need it for native Job Mgr format

    # This is what SHOULD work, but doesn't because lib-jobmanager can't handle anything other than basic strings!!!
    #meas_sequence = fields.List(
    #        fields.Str(
    #        validate=OneOf(tpro_meas_seq_choices)
    #        ),
    #        load_from='measurementSequence',
    #        dump_to='measurementSequence',
    #        description='The FCompPRO measurement sequence to be used'
    #
    meas_sequence = fields.Str(
       # Impossible to validate content at this level with current lib jobmanager - do it in solution sw
       load_from='measurementSequence',
       dump_to='measurementSequence',
       description='The FCompPRO measurement sequence to be used'
    )
    fiber_type = fields.Str(
        validate=OneOf(fiber_types),
        load_from='fiberType',
        dump_to='fiberType',
        description='The type of fiber to be tested'
    )
    config_file = config_factory('/user/disk/')


class PwmSchema(Schema):
    """
    Cdm Schema for Powermeter
    """
    test_type = test_type_factory('powermeter')

class TrueFiberSchema(Schema):
    """
    Cdm Schema for ATT TRUE FIBER measurements
    """
    test_type = test_type_factory('trueFiber')
#     certification_settings can specify a : config_file = config_factory('/user/disk/')
#     fiber = fields.Dict(
#         required=False,
#         missing={},
#         allow_none=True,
#         load_from='fiber',
#         dump_to='fiber',
#         description='The info on the fiber'
#     )
#     setups = fields.Nested(
#         Schema,
#         missing=[],
#         required=False,
#         allow_none=True,
#         load_from='setup',
#         dump_to='setup',
#         description='The setup(s)'
#     )

CONFIG_SCHEMAS = {
                 #'otdrSettings': CdmOTDRSchema, # CDM for OTDRs
                 'CDMOTDR': CdmOTDRSchema, # CDM for OTDRs
                 'OTDR': OTDRSchema, # old, for native job manager backwards compatibility
                 'manualStep': Schema,  ## ManualSchema for native
                 'Fiber Inspection': FiberInspectionSchema, # Allows something to happen if generated from Web tool rather than rejecting the Job
                 'fiberInspection': FiberInspectionCdmSchema, # FIT schema for Northbound CDM ?any differences to require this? CdmFiberInspectionSchema,
                 'fromCdm': OpenSchema,
                 'powermeter': PwmSchema,
                 'bidirIlOrl': FcompProIlOrlSchema,
                 'trueFiber': TrueFiberSchema,  # Specific test type for AT&T combining OTDR & PON Powermeter
                 'fcompPro': FcompProMainSchema # full FCompPRO sequence (can combine subtests)
                 }

class ConfigSchema(ConfigBase):
    """
    Polymorphic mapping for different CDM ConfigSchemas
    """

    # Need to override the _load method of OneOfSchema
    # because for some CDM schemas we need to get the config test type expected by Native Job Manager schemas
    # from the test.type itself
    # e.g. OTDR
    def _load(self, data, partial=None):
        log.debug('## ConfigSchema._load: data = %s', data )

        if 'otdrSettings' in data:
            log.debug('##        ConfigSchema is for an OTDR' )
            data = data.get("otdrSettings")
            data['testType'] = 'CDMOTDR' # 'otdrSettings' #'CDMOTDR'
        elif 'testType' not in data: # need the CDM test type to feed in here somehow TODO
            log.debug('##        ConfigSchema config doesnt have test type -> get it fromCdm' )
            data['testType'] = 'fromCdm'
        elif 'testType' in data and data['testType'] == 'fcompPro':
            # Need to jump through some hoops to allow use of a list for CDM measurementSequence
            # whilst still keeping it formatted as a string to avoid segfaulting in the (rather limited) jobmanager library
            # Strip content 'back to basics' before reconstructing as a list inside a string (!)
            adjust_value = str( data['measurementSequence'] ).replace("[","").replace("]","").replace('"',"").replace("'","")
            #log.debug('##        adjust_value ---> %s', adjust_value )
            data['measurementSequence'] = str( list( adjust_value.split( ", ") ))
            log.debug('##        ConfigSchema for fcompPro: Sequence is ---> %s', data['measurementSequence'] )

        return super(ConfigSchema, self)._load(data)

    type_schemas = CONFIG_SCHEMAS

