"""
Module that contains schemas for the config information of different tests to
launch on the T-BERD/MTS-5800 product family
"""
from marshmallow import Schema, fields, validates_schema, ValidationError
from marshmallow.validate import OneOf

from .config_schema_common import ConfigBase, test_type_factory, SubTypeRefInfoFormatter

#pylint: disable=no-self-use

def validate_ip_version(data):
    """
    Validates that a layer 3 or layer 4 test has a IP version
    """
    if (data['layer'] == 'L3') or (data['layer'] == 'L4'):
        if not data.get('ipversion'):
            raise ValidationError('Layer 3/4 test requires a IP version',
                                  field_names='ipVersion')

def validate_traffic_mode(data):
    """
    Validates that a layer 2 or layer 3 test has a traffic mode
    """
    if (data['layer'] == 'L2') or (data['layer'] == 'L3'):
        if not data.get('mode'):
            raise ValidationError('Layer 2/3 test requires a traffic mode',
                                  field_names='mode')


def config_factory(ui_path_hint):
    """Function to create a config file field containing

    Args:
        ui_path_hint (str): a path hint for the job manager web app to use as the base path
    """
    return fields.Str(
        required=False,
        load_from='configFile',
        dump_to='configFile',
        x_uiPathHint=ui_path_hint,
        description=(
            'the config or project file that needs to be loaded for the test at launch'
        )
    )


IP_VERSIONS = ('IPv4',
               'IPv6',
               None)

IP_VERSION_FIELD = fields.Str(required=False,
                              allow_none=True,
                              validate=OneOf(IP_VERSIONS,
                                             error="Not a valid IP version"),
                              load_from='ipVersion',
                              dump_to='ipVersion',
                              description='version of IP')


class CpriCheckConfigSchema(Schema):
    """
    Schema for the configuration information of a CPRI
    Check test
    """
    rates = ('Rates 1-7',
             '614.4M',
             '1228.8M',
             '2457.6M',
             '3072.0M',
             '4915.2M',
             '6144.4M',
             '9830.4M',
             '10137.6M')

    test_type = test_type_factory('CPRI Check')
    rate = fields.Str(required=True,
                      validate=OneOf(rates, error='Not a valid rate'),
                      dump_to='rate',
                      load_from='rate',
                      description='the rate of CPRI Check test to be launched')
    config_file = config_factory('/user/bert/tests/')

class CpriConfigSchema(Schema):
    """Schema for the configuration of a CPRI test"""
    rates = ("Rates 1-7", "Rates 8-9")
    modes = (
        "Layer 2 BERT Term",
        "Layer 2 BERT Mon/Thru",
        "Spectrum-High-Refresh Term",
        "Spectrum-High-Refresh Mon/Thru"
    )

    test_type = test_type_factory("CPRI")
    rate = fields.Str(
        required=True,
        validate=OneOf(rates, error="Not a valid rate"),
        dump_to="rate",
        load_from="rate",
        description="the rate of the CPRI test to be launched"
    )
    mode = fields.Str(
        required=True,
        validate=OneOf(modes, error="Not a valid mode"),
        dump_to="mode",
        load_from="mode",
        description="the mode of the CPRI test to be launched"
    )

class QuickCheckConfigSchema(Schema):
    """
    Schema for the configuration information of the QuickCheck test
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '200GigE',
                  '400GigE')
    layers = ('L2',
              'L3')
    test_type = test_type_factory('QuickCheck')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           load_from='interface',
                           dump_to='interface',
                           description='interface to launch the test')
    layer = fields.Str(required=True,
                       validate=OneOf(layers, error='Not a valid layer'),
                       load_from='layer',
                       dump_to='layer',
                       description='layer to launch the test')
    config_file = config_factory('/user/bert/tests/')

class PtpCheckConfigSchema(Schema):
    """
    Schema for the configuration information of the PTP Check test.
    """
    interfaces = ('1GigE', '10GigE', '25GigE')
    profiles = ('G.8275.1',
                'L4')
    test_type = test_type_factory('PTP Check')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the PTP Check test to be launched')
    profile = fields.Str(required=True,
                         validate=OneOf(profiles, error='Not a valid profile'),
                         load_from='profile',
                         dump_to='profile',
                         missing='L4',
                         description='PTP Check profile',
                         config_file=config_factory('/user/bert/tests/'))
    config_file = config_factory('/user/bert/tests/')

class DarkFiberQualConfigSchema(Schema):
    """
    Schema for the Dark Fiber Qual test.
    """
    test_type = test_type_factory('Dark Fiber Qual')

class OpticsSelfTestConfigSchema(Schema):
    """
    Schema for the configuration information of the Optics Self-test.
    """
    interfaces = ('OTU3',
                  'OTU4',
                  'OC-768',
                  'STM-256',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '200GigE',
                  '400GigE')
    test_type = test_type_factory('Optics Self-Test')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the Optics Self-Test to be launched')
    config_file = config_factory('/user/bert/tests/')

class CableTestConfigSchema(Schema):
    """
    Schema for the configuration information of Cable Test
    """
    interfaces = ('10GigE',
                  '25GigE',
                  '40GigE',
                  '100GigE',
                  '100GigE RS-FEC')
    test_type = test_type_factory('Cable Test')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the Optics Self-Test to be launched')
    config_file = config_factory('/user/bert/tests/')

class TrueSpeedConfigSchema(Schema):
    """
    Schema for the configuration information of the Truespeed test.
    """
    interfaces = ('10/100/1000',
                  '1GigE',
                  '10GigE',
                  '100GigE')
    test_type = test_type_factory('RFC 6349 TrueSpeed')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the TrueSpeed test to be launched')
    ipversion = IP_VERSION_FIELD
    config_file = config_factory('/user/bert/tests/')

class TrueSpeedVNFConfigSchema(Schema):
    """Schema for the configuration of the TrueSpeed VNF test
    """
    interfaces = ('10/100/1000',
                  '1GigE',
                 )
    test_type = test_type_factory('RFC 6349 TrueSpeed VNF')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description=('the interface that the TrueSpeed VNF'
                                        'test will be launched on'),
                          )
    config_file = config_factory('/user/bert/tests/')

class OtnCheckConfigSchema(Schema):
    """
    Schema for the configuration information of the OTN Check test
    """
    rates = ('2.7G',
             '10.7G',
             '11.05G',
             '11.1G',
             '43.02G',
             '111.8G')
    test_type = test_type_factory('OTN Check')
    rate = fields.Str(required=True,
                      validate=OneOf(rates, error='Not a valid rate'),
                      dump_to='rate',
                      load_from='rate',
                      description='the rate of CPRI Check test to be launched')
    config_file = config_factory('/user/bert/tests/')

class RFC2544ConfigSchema(Schema):
    """
    Schema for the configuration information of the RFC 2544 Test.
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '200GigE',
                  '400GigE')
    layers = ('L2',
              'L3',
              'L4')

    test_type = test_type_factory('RFC 2544')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the RFC 2544 test to be launched')
    layer = fields.Str(required=True,
                       validate=OneOf(layers, error='Not a valid layer'),
                       load_from='layer',
                       dump_to='layer',
                       description='layer to launch the test')
    ipversion = IP_VERSION_FIELD
    config_file = config_factory('/user/bert/tests/')

    @validates_schema(skip_on_field_errors=True)
    def validate_ip_version(self, data):
        """
        Schema level validator that guarantees proper format and cotent of the fields specified
        """
        validate_ip_version(data)

class SAMCompleteConfigSchema(Schema):
    """
    Schema for the configuration information of the SAMComplete Test.
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '200GigE',
                  '400GigE')
    layers = ('L2',
              'L3',
              'L4')
    modes = ('traffic',
             'streams',
             None)
    test_type = test_type_factory('Y.1564 SAMComplete')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the SAMComplete test to be launched')
    layer = fields.Str(required=True,
                       validate=OneOf(layers, error='Not a valid layer'),
                       load_from='layer',
                       dump_to='layer',
                       description='layer to launch the test')
    mode = fields.Str(required=False,
                      allow_none=True,
                      validate=OneOf(modes, error='Not a valid mode'),
                      load_from='mode',
                      dump_to='mode',
                      description='traffic mode')
    ipversion = IP_VERSION_FIELD
    config_file = config_factory('/user/bert/tests/')

    @validates_schema(skip_on_field_errors=True)
    def validate_ip_and_traffic(self, data):
        """
        Schema level validator that guarantees proper format and cotent of the fields specified
        """
        validate_ip_version(data)
        validate_traffic_mode(data)


class EthernetConfigSchema(Schema):
    """Schema for the configuration of Ethernet BERT tests
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '200GigE',
                  '400GigE')
    layers = ('Layer 2 Traffic',
              'Layer 2 Multiple Streams',
              'Layer 3 Traffic',
              'Layer 3 Multiple Streams',
              'Layer 4 Traffic,',
              'Layer 4 Multiple Streams')

    test_type = test_type_factory('Ethernet')
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces, error='Not a valid interface'),
        dump_to='interface',
        load_from='interface',
        description='the interface of the Ethernet test to be launched'
    )
    layer = fields.Str(
        required=True,
        validate=OneOf(layers, error='Not a valid layer'),
        dump_to='layer',
        load_from='layer',
        description='layer to launch the test'
    )
    ipversion = IP_VERSION_FIELD

    mode = fields.Str(
        requied=True,
        validate=OneOf(('Term', 'Loopback')),
        dump_to='mode',
        load_from='mode',
        description='the mode of the test to launch'
    )

    @validates_schema(skip_on_field_errors=True)
    def validate_ip_version(self, data):
        """
        Validates that a layer 3 or layer 4 test has a IP version
        """
        if ('3' in data['layer']) or ('4' in data['layer']):
            if not data.get('ipversion'):
                raise ValidationError('Layer 3/4 test requires a IP version',
                                      field_names='ipVersion')

class Discover5gNrConfigSchema(Schema):
    """
    Schema for the configuration information of the  5G NR Discovery test.
    """
    interfaces = ('1GigE', '10GigE', '25GigE')
    test_type = test_type_factory('5G NR Discovery')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the 5G NR Discovery test to be launched')

class CableDiagnosticsConfigSchema(Schema):
    """
    Schema for the configuration information of the Cable Diagnostics Test.
    """
    test_type = test_type_factory('Cable Diagnostics')


class VoipConfigSchema(Schema):
    """Schema for the configuration information for VoIP tests
    """
    test_type = test_type_factory('VoIP')
    interfaces = (
        '10/100/1000',
        '100M',
        '1GigE',
        '10GigE',
        )
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='the interface of the VoIP test to be launched'
    )


class SDHConifigSchema(Schema):
    """Schema for the configuration of SDH tests
    """
    test_type = test_type_factory('SDH')
    interfaces = (
        'STM-1e',
        'STM-1e jitter',
        'STM-1',
        'STM-4',
        'STM-16',
        'STM-64',
    )
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='the interface of the SDH test to be launched'
    )
    mapping = fields.Str(
        load_from='mapping',
        dump_to='mapping',
        description='the payload mapping for the test'
    )
    mode = fields.Str(
        load_from='mode',
        dump_to='mode',
        description='the mode for the test'
    )


class SonetConfigSchema(Schema):
    """Schema for configuration of SONET tests
    """
    interfaces = (
        'STS-1',
        'STS-1 jitter',
        'OC-3',
        'OC-12',
        'OC-48',
        'OC-192',
    )
    test_type = test_type_factory('SONET')
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='the interface for the SONET test to be launched'
    )
    mapping = fields.Str(
        load_from='mapping',
        dump_to='mapping',
        description='the payload mapping for the test'
    )
    mode = fields.Str(
        load_from='mode',
        dump_to='mode',
        description='the mode for the test'
    )


class DS1ConfigSchema(Schema):
    """Schema for configuration of DS1 tests
    """
    test_type = test_type_factory('DS1')
    modes = (
        'BERT Term',
        'Wander Mon',
        'Signaling Term',
        'ISDN PRI Term',
        'VF Term',
    )
    mode = fields.Str(
        required=True,
        validate=OneOf(modes),
        load_from='mode',
        dump_to='mode',
        description='The test mode of the test to launch'
    )


class E1ConfigSchema(Schema):
    """Schema for configuration of E1 tests
    """
    test_type = test_type_factory('E1')
    modes = (
        'BERT Term',
        'Wander Mon',
        'ISDN PRI Term'
    )
    mode = fields.Str(
        required=True,
        validate=OneOf(modes),
        load_from='mode',
        dump_to='mode',
        description='The test mode for the test to launch'
    )


class OnePPSConfigSchema(Schema):
    """Schema for configuration of Timing tests
    """
    test_type = test_type_factory('1PPS')


class IEEE1588ConfigSchema(Schema):
    """Schema for configuration of IEEE 1588 PTP tests
    """
    test_type = test_type_factory('PTP/1588')
    interfaces = (
        '10/100/1000',
        '100M',
        '1GigE',
        '10GigE',
        '25GigE',
    )
    layers = (
        'Layer 4',
        'Layer 2',
    )
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='The interface for the IEEE 1588 test to be launched'
    )
    layer = fields.Str(
        required=True,
        validate=OneOf(layers),
        load_from='layer',
        dump_to='layer',
        description='The layer of encapsulation for the PTP traffic'
    )


class FiberInspectionSchema(Schema):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC-61300-3-35 Ed. 2.0)",
        "SM APC (IEC-61300-3-35 Ed. 2.0)",
        "MM  (IEC-61300-3-35 Ed. 2.0)",
        "SM PC & Fiber Stub (IEC-61300-3-35 Ed. 2.0)",
    )

    test_type = test_type_factory('Fiber Inspection')
    profile = fields.Str(
        load_from='profile',
        dump_to='profile',
        description='The Inspection Probe profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )

class FiberTestLaunch():
    launch = {
    "launch_type" : "exe",
    "can_launch" : "/user/job-manager/launch/fiber-can-launch.sh", 
    "launch" : "/user/job-manager/launch/fiber-launch.sh"}

class OTDRSchema(Schema, FiberTestLaunch, SubTypeRefInfoFormatter):
    """Schema for the configuration of an OTDR test
    """
    test_type = test_type_factory('OTDR')
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Str(
        load_from='alarms',
        dump_to='alarms',
        description='The name of the alarms standard to be used for pass fail criteria'
    )
    """TBD: Does this really need to exist as well as the more general .cfg/.prj file definition ?"""
    labelFile = fields.Str(
        load_from='labelFile',
        dump_to='labelFile',
        description='The path to the project (.prj) file containing Fiber labels'
    )

    config_file = config_factory('/user/disk/')

class PowerMeterConfigSchema(Schema):
    """Schema for configuration of Power Meter tests
    """
    test_type = test_type_factory('opticalPower')

class RfVisionAntennaAlignConfigSchema(Schema):
    """Schema for configuration of Rf Vision Antenna Alignment tests
    """
    test_type = test_type_factory('AntennaAlign')

class RfVisionMicrowaveAlignConfigSchema(Schema):
    """Schema for configuration of Rf Vision Microwave Alignment tests
    """
    test_type = test_type_factory('MicrowaveAlign')

class ManualSchema(Schema):
    """
    Schema for Generic Manual Steps in a Test Plan.
    """
    test_type = test_type_factory('manualStep')


class CdmOTDRSchema(Schema):
    """Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='alarmThresholds',
        dump_to='alarmThresholds',
        description='The alarm thresholds'
    )
    config_file = config_factory('/user/disk/')
    schema_version = fields.Str(
        load_from='schemaVersion',
        dump_to='schemaVersion',
        description='version number'
    )
    setups = fields.Nested(
        Schema,
        missing=[],
        required=False,
        allow_none=True,
        load_from='setup',
        dump_to='setup',
        description='The setup(s)'
    )
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )


CONFIG_SCHEMAS = {'CPRI Check': CpriCheckConfigSchema,
                  'CPRI': CpriConfigSchema,
                  'QuickCheck': QuickCheckConfigSchema,
                  'RFC 2544': RFC2544ConfigSchema,
                  'PTP Check': PtpCheckConfigSchema,
                  'Dark Fiber Qual': DarkFiberQualConfigSchema,
                  'Optics Self-Test': OpticsSelfTestConfigSchema,
                  'Cable Test': CableTestConfigSchema,
                  'OTN Check': OtnCheckConfigSchema,
                  'RFC 6349 TrueSpeed': TrueSpeedConfigSchema,
                  'Y.1564 SAMComplete': SAMCompleteConfigSchema,
                  'Fiber Inspection': FiberInspectionSchema,
                  'Cable Diagnostics': CableDiagnosticsConfigSchema,
                  'RFC 6349 TrueSpeed VNF': TrueSpeedVNFConfigSchema,
                  'VoIP': VoipConfigSchema,
                  'SDH': SDHConifigSchema,
                  'SONET': SonetConfigSchema,
                  'Ethernet': EthernetConfigSchema,
                  '5G NR Discovery': Discover5gNrConfigSchema,
                  'DS1': DS1ConfigSchema,
                  'E1': E1ConfigSchema,
                  '1PPS': OnePPSConfigSchema,
                  'PTP/1588': IEEE1588ConfigSchema,
                  'OTDR': OTDRSchema,
                  'manualStep': ManualSchema,
                  'opticalPower': PowerMeterConfigSchema,
                 }



CDM_CONFIG_SCHEMAS = {
                  'otdrSettings': CdmOTDRSchema,
                  'manualStep': Schema
                 }


class ConfigSchema(ConfigBase):
    """
    Polymorphic mapping for different ConfigSchemas
    """
    type_schemas = CONFIG_SCHEMAS
