"""Module containing common components for config schemas

Primarily field factory functions to create a parameritized field and
the ConfigBase class
"""

from marshmallow import fields
from marshmallow.validate import Equal
from marshmallow_oneofschema import OneOfSchema

def test_type_factory(test_type_str):
    """Function to create a test type field based on a string input

    Args:
        test_type_str (string): the string that will be required input for
            this field
    """
    test_type = fields.Str(validate=Equal(test_type_str,
                                          error='test type mismatch error'),
                           required=True,
                           load_from='testType',
                           dump_to='testType',
                           description='the type of the test')
    return test_type

def config_factory(ui_path_hint):
    """Function to create a config file field containing

    Args:
        ui_path_hint (str): a path hint for the job manager web app to use as the base path
    """
    return fields.Str(
        required=False,
        load_from='configFile',
        dump_to='configFile',
        x_uiPathHint=ui_path_hint,
        description=(
            'the config or project file that needs to be loaded for the test at launch'
        )
    )

def get_launch_params(cls):
    launch = {}
    try:
        launch = cls.launch
    except:
        print(cls, " missing launch parameters")
    return launch

#Formats the current ONA test configuration format using sub type and reference info. For example
# "testLocations": [
# 	{
# 		"label": "OTDR",
# 		"attributes": {
# 			"subTypeInfo": {
# 				"testLabel": ""
# 			},
# 			"referenceInfo": {
# 				"fiberNumber": "1",
# 				"fiberId": "fid1"
# 			}
# 		}
# 	}
# ],

class SubTypeRefInfoFormatter():
    #Formatters must implement a static method with signature ui_config_parameters(cdm_test, test_index=0, location_index=0)
    #location_index is the index of the test location of interest within the "testLocations" array
    #location_index of -1 indicates there are no locations associated with the test
    #Expected output is {"Sub Type":"subtype1", "fiberNumber":"1","fiberId":"fid1"}
    @classmethod
    def ui_config_parameters(cls, cdm_test, location_index=0):
        #Formatter for sub type info and reference info 
        ui_params = {}
        test_attrs = None
        try:              
            if location_index == -1:   
                #No locations associated with this test   
                test_attrs = cdm_test["attributes"]        
            else:
                #Use the specified location
                test_attrs = cdm_test["testLocations"][location_index]["attributes"]        
            try:
                #See if there is a subtype and save to a ui parameter called Sub Type
                test_label = test_attrs["subTypeInfo"]["testLabel"]
                if test_label:
                    ui_params["Sub Type"] = test_label
            except:
                pass
            try: 
                for k, v in test_attrs["referenceInfo"].items():
                    ui_params[k] = v
            except:
                pass
        except:
            pass
        return ui_params



class ConfigBase(OneOfSchema):
    """
    Polymorphic mapping for different ConfigSchemas
    """
    type_field = 'testType'
    type_field_remove = False

    def get_obj_type(self, obj):
        if not 'test_type' in obj:
            return None
        return obj['test_type']
