"""
Module containing functions to launch the rest api web server with
desired user configurations
"""

import argparse
import sys
import logging
import os, time

import bottle
from rest_api.api.datetime_v1        import api_node as api_datetime_v1
from rest_api.api.location_v1        import api_node as api_location_v1
from rest_api.api.syncfiles_v1       import api_node as api_syncfiles_v1
from rest_api.api.filemgr_v1         import api_node as api_filemgr_v1
from rest_api.api.filemgr_v2         import api_node as api_filemgr_v2
from rest_api.api.saa_v1             import api_node as api_saa_v1
from rest_api.api.vnc_v1             import api_node as api_vnc_v1
from rest_api.api.firmware_v1        import api_node as api_firmware_v1
from rest_api.api.instrumentlogin_v1 import api_node as api_instrumentlogin_v1
from rest_api.api.techinfo_v1        import api_node as api_techinfo_v1
from rest_api.api.userfiles_v0       import api_node as api_userfiles_v0
from rest_api.api.workflow_v1        import api_node as workflow_v1
from rest_api.api.workflow_v2        import api_node as workflow_v2
from rest_api.api.jobmanager_v1      import api_node as jobmanager_v1
from rest_api.api.jobmanager_v2      import api_node as jobmanager_v2
from rest_api.api.options_v1         import api_node as options_v1
from rest_api.api import instrument_info
from rest_api.api.job_manager.plugin import JobManagerPlugin
from rest_api.api.job_manager.job_manager import JobManager
from rest_api import products
from rest_api.api.job_manager.import_modules import import_test_definitions
from rest_api.utils import viavi_paste_translogger
from paste import httpserver

def main(args=None):
    """
    Main launch point for the rest api
    Parses command line arguments and launches the web server
    """
    if args is None:
        args = parse_args(sys.argv[1:])
    log_level = logging.DEBUG if args.verbose else logging.INFO
    if args.logfile:
        logging.basicConfig(filename=args.logfile,level=log_level)
    else:
        #Log to stdout. If starting with systemd this will use journald.
        logging.basicConfig(level=log_level)

    log = logging.getLogger(__name__)

    if not args.filemgrdir:
        args.filemgrdir = args.userdir
        
    product_specific = products.product_specific(args.model, args.product_definitions_dir)
    config = {
        'rest_api.product_specific': product_specific,
        'user_files.base_dir': args.userdir,
        'user_files.filemgr_dir': args.filemgrdir,
        'user_files.upload_dir': args.uploads,
        'user_files.reports_dir': args.reports,
        'rest_api.db_file': args.job_db_file,
        'rest-api.jobs_dir': args.jobs_dir,
        'rest-api.ss_workflow_file': args.ss_workflow_file,
        'rest-api.ss_template_file': args.ss_template_file,
    }
    #After we create the product specific object which imports test definitions, dynamically look for any test definitions
    imported_test_definitions = import_test_definitions(args.test_definitions_dir)
    product_specific.test_definitions.update(imported_test_definitions)

    app = create_app(config)
    log.debug('## main.create_app done')

    product_specific.startup_side_effect(
        args.job_db_file,
        args.reports,
        args.procedures,
        args.jobs_dir,
    )
    log.debug('## main.product_specific.startup_side_effect done')

    cleanup_northbound_jobs(product_specific, args.jobs_dir)
    log.info('## cleanup_northbound_jobs done')

    app = viavi_paste_translogger.ViaviPasteTransLogger(app)
    httpserver.serve(
        app,
        host='0.0.0.0', # improve security ?? run as 127.0.0.1 ?? ...
        port=args.port,
        protocol_version="HTTP/1.1",
    )


def parse_args(args):
    """
    Parses command line arguments
    """
    parser = get_argparser()
    args = parser.parse_args(args)
    return args


def get_argparser():
    """
    Sets up the commmand line argument parser

    Returns:
        parser (ArgumentParser): initilaized object configured with arguments to parse
    """
    parser = argparse.ArgumentParser()
    parser.add_argument('-d',
                        '--userdir',
                        default='.',
                        help='Base directory to serve files from')
    parser.add_argument('--filemgrdir',
                        default='',
                        help='Base directory for filemgr')
    parser.add_argument('-p',
                        '--port',
                        default=80,
                        type=int,
                        help='HTTP server listen port')
    parser.add_argument('-u',
                        '--uploads',
                        default='.',
                        help='Directory where uploaded files should be saved')
    parser.add_argument('-v',
                        '--verbose',
                        action='store_true',
                        help='Enable verbose debug output')
    parser.add_argument('-l',
                        '--logfile',
                        help='Enable logging to specific file or stdout if empty')
    parser.add_argument('--job_db_file',
                        default='./jobs.db',  # May need to be elsewhere /var/db for example?
                        help='path to the job manager database file')
    parser.add_argument('--model',
                        default='off-target',
                        help='model type for the instrument')
    parser.add_argument('--reports',
                        default='./job-manager/', # May need to be different for STE ?
                        help='path to the default base directory where reports are created')
    parser.add_argument('--procedures',
                        default='./job-manager/test-procedures/',
                        help='path to the base directory where test procedures are stored')
    parser.add_argument('--jobs_dir',
                        default='./job-manager',
                        help='path to the directory where job files are stored')
    parser.add_argument('--ss_workflow_file',
                        default='/tmp/cdm/workflow.cdm.json',
                        help='path to the directory where StrataSync syncs workflow.cdm.json')
    parser.add_argument('--ss_template_file',
                        default='/tmp/workflow/template.cdm.json',
                        help='path to the directory where StrataSync syncs template.cdm.json')
    parser.add_argument('--test_definitions_dir',
                        default='/usr/share/job-manager/test-definitions/',
                        help='path to the base directory where solutions install test definition files')
    parser.add_argument('--product_definitions_dir',
                        default='/usr/share/mts-restapi/products/',
                        help='path to the base directory where instruments install product definition files')

    return parser


def create_app(config):
    """Function to create the bottle application and merge in routes from
    the user files api and the job manager api

    Returns:
        app (Bottle instance): the bottle web application to run
    """

    app = bottle.Bottle()
    JobManagerPlugin.job_manager = JobManager(config.get('rest_api.db_file'), config.get("rest-api.jobs_dir"), config.get("rest_api.product_specific"), config.get('rest-api.ss_workflow_file'), config.get('rest-api.ss_template_file'))
    app.config.update(config)

    def add_api_node(node):
        node.config.update(config)
        app.merge(node)


    # MobileTech API
    add_api_node(api_datetime_v1)
    add_api_node(api_location_v1)
    add_api_node(api_syncfiles_v1)
    add_api_node(api_filemgr_v2)
    add_api_node(api_saa_v1)
    add_api_node(api_vnc_v1)
    add_api_node(api_firmware_v1)
    add_api_node(api_instrumentlogin_v1)
    add_api_node(api_techinfo_v1)
    add_api_node(workflow_v1)
    add_api_node(workflow_v2)
    add_api_node(jobmanager_v1)
    add_api_node(jobmanager_v2)
    add_api_node(options_v1)

    # Legacy API
    add_api_node(api_userfiles_v0)
    add_api_node(api_filemgr_v1)

    instrument_info.api_node.config.update(config)
    app.merge(instrument_info.api_node)

    app.add_hook('after_request', add_connection_header) #TBD Needed by all?

    return app

def cleanup_northbound_jobs(product_specific_obj, nb_jobs_dir):
    cleanup_path = ''
    try:
        file_list_to_sync = product_specific_obj.get_sync_list(False)
        
        if os.path.exists(nb_jobs_dir):
            cleanup_path = os.path.join(nb_jobs_dir, 'northbound_jobs/')
            #clean up any job instance files older than 30 days in addition to the existing get_sync_files strategy
            remove_jobs_older_than_xDays(cleanup_path, 30)
            existing_files = [os.path.join(cleanup_path, f) for f in os.listdir(cleanup_path) if os.path.isfile(os.path.join(cleanup_path, f))]

            for file in existing_files:
                if(file not in file_list_to_sync):
                    os.remove(file)
    except:
        pass

def remove_jobs_older_than_xDays(path, numberOfDays):
    now = time.time()
    for filename in os.listdir(path):
        if os.path.getmtime(os.path.join(path, filename)) < now - numberOfDays * 86400 and os.path.isfile(os.path.join(path, filename)):
            os.remove(os.path.join(path, filename))

def add_connection_header():
    """
    Re-transmits any received 'Connection' headers.
    Only necessary for the mobile app.
    """
    connection = bottle.request.get_header('Connection')
    if connection:
        bottle.response.set_header('Connection', connection)


if __name__ == '__main__':
    main()
