import os
import json
import logging
import traceback
import trace
import datetime

from bottle import Bottle, response, request
from webargs.bottleparser import use_args

from rest_api.api.job_manager.plugin import JobManagerPlugin, SideEffect
from rest_api.utils.marshmallow_compat import marshmallow_data_compat
from rest_api.api.job_manager.cdm.cdm_schemas import CdmJob
from rest_api.api.job_manager.cdm.cdm_schemas import CdmWorkflow
from rest_api.api.job_manager.schemas_viavi import ActiveWorkOrderIdSchema

api_node = Bottle()

log = logging.getLogger(__name__)

@api_node.post("/api/workflow/v1/workorders", apply=JobManagerPlugin())
@use_args(CdmWorkflow(strict=True))
def post_workorders(workflow_input, job_manager):
    """Method to receive workorders from Mobile Tech    
    ---
    post:
      summary: Deploy workorders to instrument
      tags:
        - workflow/v1
      description: Deploys a list of work orders to the instrument in CDM format.
      parameters:
        - in: body
          name: body
          required: true
          schema:
            $ref: "#/definitions/WorkOrders"
      responses:
        200:
          description: Success
        400:
          description: Invalid request. (malformed body)
          schema:
            $ref: "#/definitions/Error"
    """

    log.debug('## Mobile Tech workflow: post_workorders:', workflow_input)

    try:
        job_manager.process_vmt_cdm_workflow(workflow_input)
        response.status = 200
    except:
        log.debug('## Mobile Tech workflow : post_workorders ERROR' )
        print(traceback.format_exc())
        response.status = 400

    if (response.status == 200):
        return {}
    else:
        return {'errorMessage': 'Error'}

@api_node.get("/api/workflow/v1/list", apply=JobManagerPlugin())
def get_workorder_id_list(job_manager):
    """Returns a list of the workorders currently stored on the instrument
    ---
    get:
      summary: Get list of work orders on instrument
      tags:
        - workflow/v1
      description: Returns a list of CDM v2.1 and later work orders with their locally
        referenced ID.This is recommended to be a list of workOrderId fields as assigned
        by StrataSync, but supports any identifier since the work order may not originate
        in StrataSync
      responses:
        200:
          description: Success.
          schema:
            $ref: "#/definitions/WorkOrderListResponseBody"
    """

    log.debug('## Mobile Tech workflow: get_workorder_id_list' )
    try:
        workOrderIds_list = []
        workOrderInfo_list = []

        workorder_info = job_manager.get_all_job_info() 
        for w in workorder_info:
            workOrderIds_list.append(w["workflow"]["workOrderId"])
            w["workflow"]["modifiedTime"]           = w["jobModifiedOn"]
            w["workflow"]["ijm_archived"]           = w["ijm_archived"]
            w["workflow"]["totalNumberOfTests"]     = w["totalNumberOfTests"]
            w["workflow"]["numberOfCompletedTests"] = w["numberOfCompletedTests"]
            managedBy = w["workflow"].get("jobManagement", {}).get("managedBy")
            if "managedBy" in w:
              w["workflow"]["managedBy"] = managedBy
            workOrderInfo_list.append(w["workflow"])

        response_status = 200
        workorder_list = {"workOrderIds": workOrderIds_list, "workOrderInfo": workOrderInfo_list}        

    except:
        log.debug('## Mobile Tech workflow: get_workorder_id_list ERROR' )
        print(traceback.format_exc())
        response_status = 400

    response.status = response_status
    if response_status == 200:        
        return workorder_list
    else:
        return {}

@api_node.get("/api/workflow/v1/templates", apply=JobManagerPlugin())
def get_template_list(job_manager):
    """Returns a list of workorder templates currently stored on the instrument
    ---
    get:
      summary: Get list of templates
      description: Returns a list of CDM v2.1 compliant work order attributes and
        test configurations
      responses:
        200:
          description: Success
          schema:
            $ref: "#/definitions/WorkOrderTemplatesResponseBody"
      tags:
        - workflow/v1
    """
    log.debug('## Mobile Tech workflow: get_template_list' )
    template_list = {}
    try:
        template_list["templates"] = job_manager.get_templates() 
        response.status = 200

    except:
        log.debug('## Mobile Tech workflow: get_workorder_id_list ERROR' )
        print(traceback.format_exc())
        response.status = 400

    return template_list
    

@api_node.get("/api/workflow/v1/workorder/<id>", apply=JobManagerPlugin())
def get_workorder(id, job_manager):
    """Returns the details for a specific workOrder in workflow format
    ---
    get:
      summary: Get specified work order Mobile Tech format. This isn't valid cdm since "cdm" is in an object
      tags:
        - workflow/v1
      description: Returns the CDM/CWM v2.1 and later associated with a work order, based
        on the ID from the list command. 
      parameters:
        - in: path
          name: id
          description: work order ID from the list command
          required: true
          type: string
      responses:
        200:
          description: Workorder found. Return workorder.
          schema:
            $ref: "#/definitions/WorkOrder"
        404:
          description: Workorder not found.
    """
    log.debug('## Mobile Tech workflow: get_workorder' )
    workorder = {}
    try:
        job = job_manager.get_job_clean_view(id) 
        workorder = None
        if job:
            cdm_version = job.get('cdmVersion')
            #MobileTech expects a non-CDM compliant version since "cdm" is not in an array but a singular object
            workorder = {"cdmVersion" : cdm_version}
            workorder["cdm"] = job 
        if workorder:
            response_status = 200
        else:
            response_status = 404

    except:
        log.debug('## Mobile Tech workflow: get_workorder  ERROR' )
        print(traceback.format_exc())
        response_status = 400
    response.status = response_status    
    if (response_status == 200):
        return workorder
    else:
        return {}

@api_node.get("/api/workflow/v1/job/<id>", apply=JobManagerPlugin())
def get_job(id, job_manager):
    """Returns the details for a specific workOrder in job format
    ---
    get:
      summary: Get specified work order in valid CDM format
      tags:
        - workflow/v1
      description: Returns the CDM/CWM v2.1 and later associated with a work order, based
        on the ID from the list command
      parameters:
        - in: path
          name: id
          description: work order ID from the list command
          required: true
          type: string
      responses:
        200:
          description: Workorder found. Return workorder.
          schema:
            $ref: "#/definitions/WorkOrder"
        404:
          description: Workorder not found.
    """
    log.debug('## Mobile Tech workflow: get_workorder' )
    try:
        job = job_manager.get_job_clean_view(id) 
        return job
    except:
        log.debug('## Mobile Tech workflow: get_workorder  ERROR' )
        print(traceback.format_exc())
        response.status = 400

@api_node.get("/api/workflow/v1/active", apply=JobManagerPlugin())
def get_active_workorder(job_manager):
    """
    Returns the id of the currently active workorder (aka Job)
    ---
    get:
      summary: Get active workorder
      tags:
        - workflow/v1
      description: Returns the active workorder ID
      responses:
        200:
          description: Success.
          schema:
            $ref: "#/definitions/WorkOrderActiveResponseBody"
    """
    log.debug('## Mobile Tech : get_active_workorder' )
    # Doc says : string representing the ID of the active workorder, or empty string if no workorder is active
    active_workorder_id = ''
    try:
        active_workorder_id = None
        job = job_manager.get_active_job_object()
        if job:
          active_workorder_id = job.workOrderId()
        if active_workorder_id == None:
            #We don't want to return None if there is no active job so set it to empty string
            active_workorder_id = ""
    except:
        log.debug('## Mobile Tech workflow: get_active_workorder ERROR' )
        print(traceback.format_exc())
        active_workorder_id = ''
    log.debug('get_active_workorder [Returning %s]', active_workorder_id )
    return {"activeWorkOrderId": active_workorder_id}


@api_node.put("/api/workflow/v1/active", apply=JobManagerPlugin())
@use_args(ActiveWorkOrderIdSchema(strict=True))
def set_active_workorder(active_work_order_id_input, job_manager):
    """Sets a specific workorder to be active
    ---
    put:
      summary: Set a workorder to be active
      tags:
        - workflow/v1
      description: Sets a specific workorder to be active on the instrument, using
        the ID returned by the list command
      parameters:
        - in: body
          name: activeWorkOrderId
          description: workorder ID from the list command. None deactivates any active job
          required: true
          schema:
            $ref: "#/definitions/WorkOrderActiveResponseBody"
      responses:
        200:
          description: Success.
          schema:
            $ref: "#/definitions/WorkOrderActiveResponseBody"
        404:
          description: WorkOrder not found.
    """
    log.debug('## Mobile Tech workflow: put_active_workorder' )
    rv = {}
    response.status = 400
    try:
        work_order_id = active_work_order_id_input["activeWorkOrderId"]
        if work_order_id:
          log.debug('## Mobile Tech workflow: put_active_workorder work_order_id {}'.format(work_order_id) )
          job = job_manager.set_active_job(work_order_id)
          if job:              
              rv = active_work_order_id_input
              product_specific = api_node.config['rest_api.product_specific']
              #TODO Fiber does some specific stuff based on ATT or job manager job type
              #TODO New JM UI needs to know when VMT has updated active job
              product_specific.call_notify_job_received(work_order_id, job)
              response.status = 200
          else:
              response_status = 404
        else:
          log.debug('## Mobile Tech workflow: put_active_workorder deactivating' )
          job_manager.deactivate()
          product_specific = api_node.config['rest_api.product_specific']
          product_specific.call_notify_job_received(None, None)
          response.status = 200
    except:
        log.debug('## Mobile Tech workflow: put_active_workorder ERROR' )
        #print(traceback.format_exc())
    return rv

@api_node.post("/api/workflow/v1/workorder/<id>/attachment", apply=JobManagerPlugin())
def post_workorder_attachment(id, job_manager):
    """
    ---
    post:
      summary: Attach a file to workorder
      tags:
        - workflow/v1
      description: Send down a file, such as a photo or signature, to the instrument to
        associate with the specified ID. This should be done via multipart/form-data encoding,
        with the metadata in the first part, with the file and the file contents following.
        Metadata part must come first, shall have the filename of "metadata.txt", and
        Content-Type should be application/json. File part must come second, shall have
        name of "filedata", and Content-Type must match one of StrataSync's accepted MIME types.
      consumes:
        - multipart/form-data
      parameters:
        - in: path
          name: id
          description: workorder ID from the list command
          required: true
          type: string
        - in: formData
          name: metadata.txt
          type: string
          format: json
          description: JSON string with [File](#model-File) type contents
        - in: formData
          name: filedata
          type: file
          required: true
      responses:
        200:
          description: Success
        400:
          description: Invalid request. (malformed body)
          schema:
            $ref: "#/definitions/Error"
        404:
          description: Workorder not found.
    """
    log.debug('## Mobile Tech workflow: post_workorder_attachment' )

    response_status = 200
    try:
        metadata = request.files.get('metadata')
        metadata_data = json.load(metadata.file)
        
        filename = metadata_data['name']
        fileUri = metadata_data['fileUri']

        # The doc says ' shall have name of "filedata"'
        filedata = request.files.get('filedata')
        filedata.save(fileUri, overwrite=True)

        attachment_name = fileUri

        workorder = get_workorder(id)
        active_workorder_id = None
        job = job_manager.get_active_job_object()
        if job:
            active_workorder_id = job.workOrderId()

        if (active_workorder_id == id):

            test_data = {
                'test_type': 'Unknown',
                'verdict': 'NotApplicable',
                'file_path': attachment_name,
                'reference_info': [],
                'sub_type_info': [],
            }

            if not job_manager.add_test_data(test_data):
                response_status = 400
        else:
            # 'id' is not the active workorder
            # the documentation doesn't specified a return status for this case
            response_status = 404

    except:
        log.debug('## Mobile Tech workflow: post_workorder_attachment ERROR' )
        #print(traceback.format_exc())
        response_status = 400
 
    response.status = response_status
    if (response_status == 200):
        return {}
    else:
        return {'errorMessage': 'Error'}

