"""
Module containing functions for user files rest api
"""
import os
import json
import datetime
import logging
import xml.etree.ElementTree as ET
import traceback

try:
    from urllib import quote, unquote  # Python 2.X
except ImportError:
    from urllib.parse import quote, unquote  # Python 3+

from webargs.bottleparser import use_kwargs

from bottle import request, response, static_file, Bottle


log = logging.getLogger(__name__)

api_node = Bottle() # pylint: disable=C0103
__api_version__ = 1
__api_name__ = "syncfiles"
__api_enable__ = True

# Example to dynamically disable an api
# def api_init():
#     global __api_enable__
#     __api_enable__ = False

@api_node.get('/api/syncfiles/v1/pendingfiles/all')
def syncfiles_get_all_pendingfiles():
    """Endpoint to list the files to sync and already sync
    ---
    get:
      tags:
        - user files
      description: Endpoint to list the files to sync and and already sync
      responses:
        200:
          description: success
          schema: {'$ref': '#/definitions/UserDirectory'}
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member
    product_specific = api_node.config['rest_api.product_specific']

    files = product_specific.get_sync_list(True)
    res = []
    for f in files:
        res.append( get_file_info(f, base_dir) )

    response.content_type = 'application/json'
    return json.dumps(res)

@api_node.get('/api/syncfiles/v1/pendingfiles')
def syncfiles_get_pendingfiles():
    """Endpoint to list the files to sync
    ---
    get:
      tags:
        - user filesto sync
      responses:
        200:
          description: success
          schema: {'$ref': '#/definitions/UserDirectory'}
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member
    product_specific = api_node.config['rest_api.product_specific']

    files = product_specific.get_sync_list(False)

    res = []
    for f in files:
        res.append( get_file_info(f, base_dir) )

    response.content_type = 'application/json'
    return json.dumps(res)

@api_node.get('/api/syncfiles/v1/userfiles<file_path:path>')
def syncfiles_get_files(file_path):
    """
    Endpoint to either list the files and directories in the file_path or to
    return the file to the client
    ---
    get:
      tags:
        - user files
      description: >
                     Endpoint to either list the files and directories in the
                     file_path or to return the file to the client
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
        400:
          description: file or directory not found
    """
    base_dir = request.app.config.get('user_files.base_dir')

    unquoted_file_name = unquote(file_path)
    rel_path = strip_basepath(unquoted_file_name, base_dir)
    full_path = os.path.join(base_dir, rel_path)

    if os.path.isfile(full_path):
        static_file_response = static_file(unquoted_file_name, root=base_dir)
        return static_file_response

    response.status = 400
    return {}

@api_node.get('/api/syncfiles/v1/metadata/<file_path:path>')
def syncfiles_get_metadata(file_path):
    """Endpoint for getting the metadata for a file from the instrument

    See doc at https://conf1.ds.jdsu.net/wiki/display/MOBILE/Mobile+Tech+REST+API

    ---
    post:
      tags:
        - user files
      description: Endpoint for getting the metadata for a file from the instrument
      parameters:
        - in: path
          name: file_path
          type: string
    """

    unquoted_file_name = unquote(file_path)

    base_dir = request.app.config.get('user_files.base_dir')

    product_specific = api_node.config['rest_api.product_specific']
    res = product_specific.get_sync_file_info( unquoted_file_name, base_dir )

    return res

@api_node.put('/api/syncfiles/v1/userfiles/<file_path:path>')
def syncfiles_put_acknowledge(file_path):
    """Endpoint for uploading a file to the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload a file to the instrument
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
    """
    unquoted_file_name = unquote(file_path)

    request.body.getvalue().decode('utf-8') #pylint: disable=no-member

    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member

    rel_path = strip_basepath(unquoted_file_name, base_dir)
    full_path = os.path.join(base_dir, rel_path)
    if os.path.isfile(full_path):
        product_specific = api_node.config['rest_api.product_specific']
        product_specific.set_sync_list_ack(full_path)

        response.status = 200
    else:
        response.status = 404

    return {}

from rest_api.products.usc.cdm_sync import CDM_DOWNSTREAM_FILEPATH, CDM_DOWNSTREAM_TEMPLATES_PATH
@api_node.post('/api/syncfiles/v1/userfiles')
def syncfiles_save_file():
    """Endpoint for uploading a file to the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload a file to the instrument
      consumes:
        - multipart/form-data
      parameters:
        - in: formData
          schema: UploadArgs
      responses:
        200:
          description: success
    """
    fileUri = '' #  'except:' needs fileUri
    try:
        metadata = request.files.get('metadata')
        metadata_contents = metadata.file.read()
        if not isinstance(metadata_contents, str):
            metadata_contents = metadata_contents.decode('utf-8') 
        metadata_data = json.loads(metadata_contents)

        filename = metadata_data['name']
        fileUri = metadata_data['fileUri']
        log.debug('## syncfiles_save_file: fileUri = %s, name = %s', fileUri, filename )

        listAllowedDir = []

        listAllowedDir.append(api_node.config.get('user_files.base_dir'))
        listAllowedDir.append(CDM_DOWNSTREAM_FILEPATH)
        listAllowedDir.append(CDM_DOWNSTREAM_TEMPLATES_PATH)

        bHappy = False
        
        for allowedDir in listAllowedDir:
            if fileUri.startswith(allowedDir):
                bHappy = True
                break

        if bHappy == False:
            while (fileUri.startswith('/')):
                # os.path.join will stop joining paths when it comes accross an absolute path
                fileUri = fileUri[1:]
            fileUri = os.path.join(listAllowedDir[0], fileUri)


        # no need to save the file if the path doesn't exist
        pathOnly = os.path.dirname(fileUri)

        if os.path.isdir(pathOnly):

            filedata = request.files.get(filename)

            filedata.save(fileUri, overwrite=True)

            if 'workflow.cdm.json' in filedata.filename:
                product_specific = api_node.config['rest_api.product_specific']
                product_specific.call_process_cdm_job()
                # if we want to preserve ATT functionality over sync files endpoint, then need:
                ##notify_job_received( file_path, cdm )

                log.debug('## syncfiles_save_file: workflow file processed' )

            if 'template.cdm.json' in filedata.filename:
    #             product_specific = user_files.config['rest_api.product_specific']
    #             product_specific.call_process_cdm_job()
                log.debug('## syncfiles_save_file: template file to deal with (TODO)' )

            response.status = 200
        else:
            log.debug('## syncfiles_save_file 2 : failed for %s', fileUri )
            response.status = 400

    except:
        print(traceback.format_exc())
        if fileUri:
          log.debug('## syncfiles_save_file 3 : failed for %s', fileUri )
        response.status = 400

    return {}

def get_file_info(filepath, base_dir):
    """Function to create a dictionary representing the file path
    the swagger schema is below
    ---
    type:
      enum:
        - F
      type: string
      description: the type of this object for files this will be F
    name:
      type: string
      description: the name of the file
    path:
      type: string
      description: the relative path to the file from the base directory
    mtime:
      type: string
      description: time of most recent content modification fo rthis file
    size:
      type: number
      description: 'the size of the file, in bytes'
    """
    fstats = os.stat(filepath)
    file_name = os.path.basename(filepath)
    file_path_rel = os.path.relpath(filepath, base_dir)

    file_info = {
            "name": file_name,
            "fileUri": "/" + file_path_rel,
        }
    return file_info

def strip_basepath(file_path, base_path):
    """Strip any directories matching the base directories from the file path

    This function is so that if the user supplies an absolute path that matches
    the base path the api will return the file
    """
    file_path_dirs = file_path.split('/')
    base_path_dirs = base_path.split('/')

    for dir_name in base_path_dirs:
        if dir_name in file_path_dirs and file_path_dirs.index(dir_name) == 0:
            file_path_dirs.remove(dir_name)

    return '/'.join(file_path_dirs)
