"""
Module containing endpoints for the job manager api
"""
from webargs.bottleparser import use_args
from bottle import request, response, Bottle
from rest_api.schemas import LocationSchema


api_node = Bottle()  # pylint: disable=invalid-name
__api_version__ = 1
__api_name__ = "location"

@api_node.put('/api/v1/location')
@use_args(LocationSchema(strict=True))
def set_geolocation(geolocation):
    """
    Endpoint to update geolocation data on the instrument

    See doc at https://conf1.ds.jdsu.net/wiki/display/MOBILE/Mobile+Tech+REST+API


    GeolocationInfo  : Valid on 2020-12-15
    type: object
    required:
      - latitude
      - longitude
      - timestamp
      - accuracy
    properties:
      latitude:
        type: number
        description: A double representing the position's latitude in decimal degrees.
        example: 12
      longitude:
        type: number
        description: A double representing the position's longitude in decimal degrees.
        example: 234
      timestamp:
        type: number
        description: Timestamp when geolocation was taken in milliseconds since epoch
        example: 1448755200000
      accuracy:
        type: number
        description: A double representing the accuracy of the latitude and longitude properties, expressed in meters.
        example: 78
      altitude:
        type: number
        description: A double representing the position's altitude in metres, relative to sea level. This value can be null if the implementation cannot provide the data.
        example: 99.1
      altitudeAccuracy:
        type: number
        description: A double representing the accuracy of the altitude expressed in meters. This value can be null.
        example: 2


    ---
    put:
        tags:
          - settings
        description: Set location information from mobile tech app
        responses:
            200:
                description: Success
                'schema':
                    title: Instrument Time
                    type: object
            400:
                description: Bad Request
    """
    product_specific = api_node.config['rest_api.product_specific']
    success = product_specific.update_location(geolocation)
    if success:
        response.status = 200
    else:
        response.status = 400

    return {}
