"""
Module containing endpoints for the job manager api
"""
import os
import os.path
import json
import logging
import traceback
import datetime
import time
log = logging.getLogger(__name__)

from webargs.bottleparser import use_args, use_kwargs
from bottle import response, request, Bottle

from rest_api.api.job_manager.save_job_artifacts import create_artifacts
from rest_api.api.job_manager.job import Job
from rest_api.api.job_manager.schemas_viavi import FilePathSchema
from rest_api.api.job_manager.schemas_viavi import ArtifactSchema
from rest_api.api.job_manager.plugin import JobManagerPlugin, SideEffect
from rest_api.api.job_manager import save_job_artifacts
from rest_api.api.schemas import UploadArgs
from rest_api.api.job_manager.cdm.cdm_schemas import CdmJob, ReferenceInfoForTestTypeSchema, CdmTestsSchema, CdmWorkflowSchema, CreateJobSchema, ManualStepSchema
from rest_api.api.job_manager.schemas_viavi import TestIndexSchema
from rest_api.utils.job_test_launcher import TestLauncher, LaunchException, ServerLaunchException
from rest_api.products.config_schema_common import get_launch_params

METADATA_KEY = 'metadata'

api_node = Bottle() #pylint: disable=invalid-name

#Inject job_manger object into the view function for each route
api_node.install(JobManagerPlugin())
api_node.install(SideEffect())

@api_node.get('/api/jobs/v2/current')
def get_current_job(job_manager):
    """
    ---
    get:
        tags:
          - jobs
        description: get details about a particular job
        responses:
            200:
                'schema': CdmJob
                description: Job found. Return job. Empty JSON if no current job.
    """
    job = job_manager.get_active_job()
    if job:
        return job
    else:
        return {}


@api_node.get('/api/jobs/v2/ui_group_state')
def get_ui_job_view(job_manager):
    return {"uiview_group_state" : job_manager.get_uiview_group_state()}

@api_node.post('/api/jobs/v2/ui_group_state/<groupstate>')
def get_ui_job_view(groupstate, job_manager):
    if not job_manager.set_uiview_group_state(groupstate):
        response.status = 400

@api_node.get('/api/jobs/v2/ui/<work_order_id>')
def get_ui_job_view(work_order_id, job_manager):
    """
    ---
    get:
        tags:
          - jobs
        description: details about a particular job
        parameters:
            -   in: path
                name: work_order_id
                required: true
                type: string
        responses:
            200:
                'schema': CdmJob (with ijm specific fields)
                description: Job found. Return job. Empty JSON if not job found with matching work_order_id
    """
    grouping = request.query.get('grouping')
    job_ui = job_manager.get_job_ui_view(work_order_id, grouping)
    if job_ui:
        return job_ui
    else:
        return {}

@api_node.get('/api/jobs/v2/ui')
def get_ui_job_view_nowoid(job_manager):
    return get_ui_job_view(None, job_manager)

@api_node.put('/api/jobs/v2/archive/<work_order_id>')
def archive_job(work_order_id, job_manager):
    """
    ---
    put:
        tags:
          - jobs
        description: Endpoint to archive a job (i.e. move it out of the main job list)
        parameters:
            -   in: path
                name: workOrderId
                required: true
                type: string
        responses:
            200: Archived Job
            400: Could not archive job
    """
    if not job_manager.archive_job(work_order_id):
        response.status = 400


@api_node.put('/api/jobs/v2/restore/<work_order_id>')
def restore_job(work_order_id, job_manager):
    """
    ---
    put:
        tags:
          - jobs
        description: Endpoint to restore a job to main job list (i.e. unarchive)
        parameters:
            -   in: path
                name: workOrderId
                required: true
                type: string
        responses:
            200: Archived Job
            400: Could not restore job
    """
    if not job_manager.restore_job(work_order_id):
        response.status = 400




@api_node.post('/api/jobs/v2/active/testData')
@use_args(CdmTestsSchema(strict=True))
def add_test_data(cdm_test_data, job_manager):
    """
    Endpoint to add test data to the active job
    ---
    post:
        tags:
          - jobs
        description: adds test data to the active job
        parameters:
            -   in: body
                name: Test Data
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    #Remove "ijm_filePath" from cdm_tests and pass this as a parameter. 
    report_filepaths = cdm_test_data.pop("ijm_filePath", [])
    job = job_manager.add_test_data(cdm_test_data, False, report_filepaths, True)
    if job:
        response.status = 201
        return job
    else:
        response.status = 202
        return {}


@api_node.put('/api/jobs/v2/active/clearTestData/<ui_index>')
def clear_test_data_by_uiindex(ui_index, job_manager):
    """
    Endpoint to clear test data of the specified planned test
    ---
    get:
        tags:
          - jobs
        description: clears the test data for the specified planned test
        responses:
            201:
                description: Success.
    """
    #When setting the active test index, also clear any test data associated with this index.     
    test_index, location_index = job_manager.get_test_location_index_from_uiindex(ui_index)
    if test_index >= 0:
        if job_manager.clear_test_data(test_index, location_index):
            response.status = 201
        else:
            response.status = 202
        
    else:
        response.status = 202

@api_node.get('/api/jobs/v2/active')
def get_active_state(job_manager):
    """
    Endpiont for determining if job manager is active
    ---
    get:
        tags:
          - jobs
        description: returns "1" if the job manager is active, "0" otherwise
        responses:
            200:
                description: Success.
    """
    active = "0"
    if job_manager.is_active():        
        active = "1"
    return active


@api_node.put('/api/jobs/v2/active')
def set_active_state(job_manager, side_effect):
    """
    Endpoint for activating/deactivating job manager
    ---
    put:
        tags:
          - jobs
        responses:
            200:
                description: Success.
                schema:
                    title: Is Active
                    type: boolean
    """
    request_body = request.body.getvalue().decode('utf-8') #pylint: disable=no-member
    try:
        activate = int(request_body)
        if activate != 0:
            job_manager.activate()
        else:
            job_manager.deactivate()
    except ValueError:
        response.status = 400


@api_node.get('/api/jobs/v2/active/jobStatusRatio')
def get_job_status_ratio(job_manager):
    """
    Endpoint for obtaining the active job's job status ratio
    ---
    get:
        tags:
          - jobs
        description: returns the job status ratio of the active job
        responses:
            200:
                description: Success.
                schema:
                    title: Job Status Ratio
                    type: string
    """
    status = job_manager.get_job_status()
    if status:
        return  "/".join([str(status['complete_count']),str(status['total_count'])])
    return ""


@api_node.get('/api/jobs/v2/active/jobStatus')
def get_job_status(job_manager):
    """
    Return job status ratio as json. Same as above but JSON response instead of text.
    ---
    get:
        tags:
          - jobs
        description: returns the job status ratio of the active job
        responses:
            200:
                description: Success.
                schema:
                    title: Job Status Ratio
                    type: string
    """
    rv = {} 
    status = job_manager.get_job_status()
    if status:
        rv["totalNumberOfTests"] = status['total_count']
        rv["numberOfCompletedTests"] = status['complete_count']
    return rv


@api_node.get('/api/jobs/v2/active/testDataIndex')
def get_active_test_index(job_manager):
    """
    Endpiont for determining the launched job item of active test plan
    ---
    get:
        tags:
          - jobs
        description: returns an object containing the test and location index of the launched job item of active test plan
        responses:
            200:
                description: Success.
                schema:
                    title: Indices of Launched Job Item in Active Test Plan
                    type: {"testIndex" : int, "testLocationIndex" : int}. Index is -1 when there is no valid entry. 
    """
    #rv = []
    rv = {"testIndex" : -1, "testLocationIndex" : -1}
    index_list = job_manager.get_active_test_plan_index()
    if index_list:
        rv = {"testIndex" : index_list[0], "testLocationIndex" : index_list[1]}    
    return rv

@api_node.get('/api/jobs/v2/active/uiIndex')
def get_active_uiindex(job_manager):
    """
    Endpiont for determining the launched job item of active test plan. Returns the active ui index.
    ---
    get:
        tags:
          - jobs
        description: returns an object containing the ui index of the launched job item of active test plan
        responses:
            200:
                description: Success.
                schema:
                    title: Index of Launched Job Item in Active Test Plan
                    type: {"uiIndex" : string}. Index is empy when there is no active test. 
    """
    return {"ijm_uiindex" : job_manager.get_active_test_plan_uiindex()}    

@api_node.post('/api/jobs/v2/active/uiIndex/<ui_index>')
def set_active_test_index_by_uiindex(ui_index, job_manager):
    """
    Set the active test location of the test to be launched using the ui_index.
    ---
    get:
        tags:
          - jobs
        description: set the index of the launched test/location item of active test plan
        responses:
            201:
                description: Success.
            202:
                description: Failed to set test plan index
    """
    #When setting the active test index, also clear any test data associated with this index.     
    test_index, location_index = job_manager.get_test_location_index_from_uiindex(ui_index)
    if test_index:
        job_manager.set_active_test_plan_index(test_index, location_index)
    else:
        response.status = 202


@api_node.get('/api/jobs/v2/active/matchingTestPlanIndex')
@use_args(ReferenceInfoForTestTypeSchema(strict=True))
def get_test_plan_index(ref_info_input,job_manager):
    """
    Endpiont for determining a job item index from its test type
    and reference info
    ---
    get:
        tags:
          - jobs
        description: returns the index of the job item having the specified test type and reference info
        responses:
            404:
                description: Job Not Found
            200:
                description: Success.
                schema:
                    title: index of matching job item
                    type: integer
    """
    job = job_manager.get_active_job()

    if not job:
        response.status = 404
        return None

    response.status = 200

    ref_info = ref_info_input['reference_info']
    test_type = ref_info_input['test_type']
    index = job_manager.get_test_plan_index_of_ref(test_type, ref_info)

    if (index == 'None'):
        index = "-1"

    return str(index)



@api_node.delete('/api/jobs/v2/<work_order_id>')
def remove_job(work_order_id, job_manager):
    """
    Endpoint for deleting a job
    ---
    delete:
        tags:
          - jobs
        description: deletes the job
        parameters:
            -   in: path
                name: job_id
                required: true
                type: string
        responses:
            200:
                description: Success
            401:
                description: Cannot delete active job
            404:
                description: Job Not Found
    """
    try:
        job_manager.delete_job(work_order_id)
        response.status = 200
    except ReferenceError:
        response.status = 401        
    except FileNotFoundError:
        response.status = 404

@api_node.delete('/api/jobs/v2/jobs')
def delete_all_job(job_manager):
    """
    Endpoint for deleting all jobs. Probably only used for test
    ---
    delete:
        tags:
          - jobs
        description: Endpoint for deleting all jobs. Probably only used for test
        responses:
            200:
                description: Success
    """
    job_manager.delete_all_jobs()

@api_node.get('/api/jobs/v2/template/<type_name>')
def get_template(type_name, job_manager):
    """
    Endpoint for retrieving a template by name
    ---
    delete:
        tags:
          - jobs
        description: get the template
        parameters:
            -   in: path
                name: template type name
                required: true
                type: string
        responses:
            200:
                description: Success
            404:
                description: Template Not Found
    """
    rv = {}
    try:
        rv = job_manager.get_template(type_name)
        response.status = 200
    except FileNotFoundError:
        response.status = 404
    return rv

@api_node.delete('/api/jobs/v2/template/<type_name>')
def delete_template(type_name, job_manager):
    """
    Endpoint for deleting a template
    ---
    delete:
        tags:
          - jobs
        description: deletes the template
        parameters:
            -   in: path
                name: template type name
                required: true
                type: string
        responses:
            200:
                description: Success
            404:
                description: Template Not Found
    """
    try:
        job_manager.delete_template(type_name)
        response.status = 200
    except FileNotFoundError:
        response.status = 404

@api_node.delete('/api/jobs/v2/templates')
def delete_all_templates(job_manager):
    """
    Endpoint for deleting all templates except default. Probably only used for test. 
    ---
    delete:
        tags:
          - jobs
        description: Endpoint for deleting all templates except default
        responses:
            200:
                description: Success
    """
    job_manager.delete_all_templates()

@api_node.put('/api/jobs/v2')
@use_args(CreateJobSchema(strict=True))
def create_job(create_job_schema, job_manager):
    """
    Endpoint for creating a job from a template. 
    ---
    post:
        tags:
          - jobs
        description: Endpoint for creating a job from a template. Expects that at minimum the workOrderId is populated.
        parameters:
            -   in: body
                name: Artifact
                schema: CdmJob
                required: true
        responses:
            200:
                description: Job successfully created
            400:
                description: Failed to create job
    """
    rv = job_manager.create_job(create_job_schema)
    if not rv:
        log.debug('create_job: error')
        response.status = 400
        return rv
    return {}

@api_node.get("/api/jobs/v2/template/list", apply=JobManagerPlugin())
def get_template_list(job_manager):
    """Returns a list of the templates currently stored on the instrument
    ---
    get:
      summary: Get list of templates on instrument
      tags:
        - workflow/v1
      description: Returns a list of CDM v2.1 and later work orders with their locally
        referenced ID.This is recommended to be a list of workOrderId fields as assigned
        by StrataSync, but supports any identifier since the work order may not originate
        in StrataSync
      responses:
        200:
          description: Success.
          schema:
            $ref: "#/definitions/WorkOrderListResponseBody"
    """

    log.debug('## Mobile Tech workflow: get_template_list' )    
    template_names = [t['workflow']['typeName'] for t in job_manager.get_all_template_info()]
    return {"templateNames": template_names}


@api_node.put('/api/jobs/v2/template')
@use_args(CdmJob(strict=True))
def add_template(cdm_template, job_manager):
    """
    Endpoint for adding a template. This may not be used in customer use case but is handy for test
    ---
    post:
        tags:
          - jobs
        description: Endpoint for adding a template. This may not be used in customer use case but is handy for test
        parameters:
            -   in: body
                name: Artifact
                schema: CdmJob
                required: true
        responses:
            200:
                description: Template successfully created
            400:
                description: Failed to add template
    """
    rv = False
    try:
        if cdm_template['workflow']['template']:
            cdm_template_workflow = {"cdm" : [cdm_template]}
            cdm_template_workflow ["cdmVersion"] = cdm_template.get("cdmVersion")
            rv = job_manager.process_cdm_template_workflow(cdm_template_workflow)
    except:
        log.debug(traceback.format_exc())
    if not rv:
        log.debug('add_template: error')
        response.status = 400
        return {'errorMessage': 'Error'}
    return {}


def save_job_to_pdf(artifact_args, work_order_id, job_manager):
    job = None
    file_path_return = None
    #To work around some legacy users of the API that always assume the active/current
    #job ID is 1, we'll assume we ned to use the active job
    if not work_order_id or work_order_id == 1 or work_order_id == "1":
        job = job_manager.get_active_job_object()
    else:
        job = job_manager.get_job_object(work_order_id)
    if job:
        file_path_return = create_artifacts(job=job, **artifact_args)

    if file_path_return:
        response.status = 201
        return {'filePath': file_path_return}
    else:
        response.status = 403
        return {}

@api_node.post('/api/jobs/v2/report')
@use_args(ArtifactSchema(strict=True))
def save_current_job_to_pdf_without_workorder(artifact_args, job_manager):
    return save_job_to_pdf(artifact_args, None, job_manager)

@api_node.post('/api/jobs/v2/report/<work_order_id>')
@use_args(ArtifactSchema(strict=True))
def save_given_job_to_pdf_with_workorder(artifact_args, work_order_id, job_manager):
    return save_job_to_pdf(artifact_args, work_order_id, job_manager)


@api_node.get('/api/jobs/v2/changecount')
def get_change_count(job_manager):
    """
    Endpoint to get the current change count which is used by UI to when any job data has changed
    ---
    get:
        tags:
          - jobs
        description: get the change count
        responses:
            200:
                description: Job found. Return job. Empty JSON if no current job.
    """
    return {"changeCount" : job_manager.get_change_count()}


@api_node.get('/api/jobs/v2/testtypes')
def get_test_types(job_manager):
    test_types = {"test_types" : []}
    try:
        #test_type = request.body.getvalue().decode('utf-8') #pylint: disable=no-member   
        test_types["test_types"] = list(request.app.config["rest_api.product_specific"].test_definitions.keys())
        response.status = 200
    except:
        print(traceback.format_exc())
        response.status = 400
    return test_types



@api_node.post('/api/jobs/v2/jobManagerFile')
@use_args(FilePathSchema(strict=True))
def load_job_from_file(file_path_input, job_manager, side_effect):
    """
    Endpoint for loading a job to the datastore from a file located at a path supplied by the client.
    Used to import new jobs from USB or filesystem. 
    ---
    post:
        tags:
          - jobs
        description: loads job to the datastore from a file located at a path supplied by the client
        parameters:
            -   in: body
                name: FilePathSchema
                schema: FilePathSchema
                required: true
        responses:
            201:
                description: Job loaded successfully
                schema: CdmJob
            422:
                description: Cannot process request for a reason supplied by the response
    """
    file_path = file_path_input['file_path']

    if not os.path.isfile(file_path):
        response.status = 422
        return {"filePath": ["Bad file path"]}

    output_job = job_manager.load_job_from_file(file_path)

    if not output_job:
        response.status = 422
        return{file_path: ["Invalid job file"]}

    if job_manager.is_active():
        side_effect(output_job)

    response.status = 201
    return output_job


@api_node.error(422)
def handle_unprocessable_entity(error):
    """
    Method to make error 422 error responses raised from webargs
    more consumer friendly

    The response body will now contain a representation of the schema
    parsing error in the request.  The message will contain information
    like:

    '{"customerName": ["Missing data for required field."]}'

    Args:
        error (HTTPError): the error raised by webargs due to parsing errors
    """
    log.debug(str("ERROR: %s" % str(error.body)))
    return error.body



@api_node.post('/api/jobs/v2/active/matchingTestAndLocationIndex')
@use_args(CdmTestsSchema(strict=True))
def get_test_and_location_index_for_cdm_test(cdm_test,job_manager):
    """
    Endpiont for finding test and location indices from a given CdmTest object
    ---
    get:
        tags:
          - jobs
        responses:
            404:
                description: Match Not Found
            200:
                description: Success.
                schema:
                    title: indices of matching job item
                    type: {"testIndex" : int, "testLocationIndex" : int}. Index is -1 when there is no valid entry. 
    """
    #rv = []
    rv = {"testIndex" : -1, "testLocationIndex" : -1}
    index_list = job_manager.get_test_index_of_cdm_test(cdm_test)
    test_index = -1
    test_location_index = -1

    if index_list:
        if index_list[0] != None and index_list[0] >= 0:
            test_index = index_list[0]
        if index_list[1] != None and index_list[1] >= 0:
            test_location_index = index_list[1]    
        rv = {"testIndex" : test_index, "testLocationIndex" : test_location_index}
    else:
        response.status = 404
        return
    return rv


@api_node.post('/api/jobs/v2/active/matchingWorkflowidAndTestplanindex')
@use_args(CdmTestsSchema(strict=True))
def get_matching_workflowid_and_testplanindex(cdm_test,job_manager):
    """
    Endpiont for finding test and location indices from a given CdmTest object
    ---
    get:
        tags:
          - jobs
        responses:
            404:
                description: Match Not Found
            200:
                description: Success.
                schema:
                    title: indices of matching job item
                    type: {"workFlowId" : null/int, "testPlanIndex" : null/int} 
    """
    #rv = []
    rv = job_manager.get_matching_workflowid_and_testplanindex(cdm_test)
    if not rv:
        response.status = 404
        return
    return rv


@api_node.post('/api/jobs/v2/active/getMetaDataForTest')
@use_args(CdmTestsSchema(strict=True))
def get_metadata_for_cdmtest(cdm_test,job_manager):
    """
    Endpiont for finding test and location indices from a given CdmTest object
    ---
    get:
        tags:
          - jobs
        responses:
            404:
                description: Match Not Found
            200:
                description: Success.
                schema:
                    title: indices of matching job item
                    type: {"indexType" : string("workflowId" or "testPlanIndex"), "id" : int}. id is -1 when there is no valid entry. 
    """
    #rv = []
    rv = job_manager.get_metadata_for_cdmtest(cdm_test)
    if not rv:
        response.status = 404
        return
    return rv


@api_node.get('/api/jobs/v2/currentTestAndLocation')
def get_current_test_and_location(job_manager):
    """
    Endpoint to get the launched CDM test and location block
    ---
    get:
        tags:
          - jobs
        responses:
            200:
                'schema': CdmTest
                description: If there is an active test/location that has been launched by job manager, the CDM test/location block is returned
            404:
                description: There is no active/launched test
    """
    cdm_test_location = job_manager.get_active_test_and_location()
    if not cdm_test_location:
        response.status = 404
        return {}
    else:
        return cdm_test_location

@api_node.post('/api/jobs/v2/active/addTestLocation')
@use_args(CdmTestsSchema(strict=True))
def add_test_location(cdm_test, job_manager):
    """
    Endpoint to add a test location to an open ended test
    ---
    post:
        tags:
          - jobs
        description: adds test location to the open ended test
        parameters:
            -   in: body
                name: Test 
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    try:
        test_index = cdm_test.pop('index', -1)
        test_plan_index = int(test_index)
        job = job_manager.add_test_location(cdm_test, test_plan_index)
        if job:
            response.status = 201
            return job
        else:
            response.status = 202
            return {}
    except ValueError:
        response.status = 400
        return {}


@api_node.put('/api/jobs/v2/active/closeOpenEndedTest')
@use_args(TestIndexSchema(strict=True))
def close_open_ended_test(index, job_manager):
    """
    Endpoint to close open ended test
    ---
    put:
        tags:
          - jobs
        description: closes open ended status of active test
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Open ended test status closed
            202:
                description: No active job.  Open ended status not closed
    """

    try:
        test_index = index.pop('index', '-1')
        test_plan_index = int(test_index)
        if job_manager.close_open_ended_test(test_plan_index):
            response.status = 201
        else:
            response.status = 202
    except ValueError:
        response.status = 400
        return 


@api_node.post('/api/jobs/v2/active/addTest')
@use_args(CdmTestsSchema(strict=True))
def add_test(cdm_test, job_manager):
    """
    Endpoint to add test to the open ended active job
    ---
    post:
        tags:
          - jobs
        description: adds test to the open ended active job
        parameters:
            -   in: body
                name: Test 
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test added to job
            202:
                description: No active job.  Test not stored
    """
    job = job_manager.add_test(cdm_test)
    if job:
        response.status = 201
        return job
    else:
        response.status = 202
        return {}


@api_node.put('/api/jobs/v2/active/closeOpenEndedJob')
def close_open_ended_job(job_manager):
    """
    Endpoint to close open ended job
    ---
    post:
        tags:
          - jobs
        description: closes open ended status of active job
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Open ended job status closed
            202:
                description: No active job.  Open ended status not closed
    """

    if job_manager.close_open_ended_job():
        response.status = 201
    else:
        response.status = 202


# Start a test launch
# POST /api/jobs/v2/testlaunch/{ui_index}
# ui_index is the UI index from the particular test/location to launch
# response
# 200 Launch started. Returns an integer ID used for checking status Return a launch_uid used for calls to get('/api/jobs/v2/testlaunch/<launch_uid>') {"launch_uid": 1}
# 400 Invalid ijm_uiindex
# 409 Conflict - Another launch in progress
# 500 product specific test definitions not defined
# Any error codes (codes other than 200) will include an error object like this {"ErrorCode": "ERROR_CAN_LAUNCH_FAILED_TO_RUN","ErrorMessage": ""}
# See https://conf1.ds.jdsu.net/wiki/pages/viewpage.action?spaceKey=PSP&title=Format+of+Error+String+for+Test+Launcher+Specification
@api_node.post('/api/jobs/v2/testlaunch/<ui_index>')
def launch_test(ui_index, job_manager):
    rv = {}
    launch_uid = None
    testLauncher = TestLauncher()
    test_index, location_index = job_manager.get_test_location_index_from_uiindex(ui_index)
    if test_index != None and test_index >=0:
        cdm_test = job_manager.get_cdm_test_with_locations(test_index)        
        if cdm_test:
            test_type = cdm_test['type']
            test_definitions = request.app.config["rest_api.product_specific"].test_definitions
            if test_definitions:
                try:
                    launch_params = get_launch_params(test_definitions[test_type])
                    print("launch_params: ", launch_params)
                    launch_uid = testLauncher.create_launcher(launch_params, test_type, cdm_test, test_index, location_index)
                    if launch_uid != None:
                        testLauncher.can_launch(launch_uid)
                        job_manager.set_active_test_plan_index(test_index, location_index)       
                        rv['LaunchUid'] = launch_uid
                except LaunchException as e:
                    response.status = 400
                    log.debug("launch_test launch exception{}".format(e))
                    rv["ErrorCode"] =  str(e)
                    rv["ErrorMessage"] =  e.custom_error_message
                except ServerLaunchException as e:
                    response.status = 500
                    log.debug("launch_test launch exception{}".format(e))
                    rv["ErrorCode"] = str(e)
                    rv["ErrorMessage"] =  e.custom_error_message
                except:
                    response.status = 500
                    log.debug("launch_test launch exception ERROR_INVALID_LAUNCH_PARAMETERS")
                    rv["ErrorCode"] = "ERROR_INVALID_LAUNCH_PARAMETERS"
                    rv["ErrorMessage"] = ""
            else:
                #No test definitions
                response.status = 500
        else:
            #Invalid ui_index
            log.debug("launch_test invalid cdm_test: ui_index={}".format(ui_index))
            response.status = 400
    else:
        #Invalid ui_index
        log.debug("launch_test invalid ui_index: ui_index={}".format(ui_index))
        response.status = 400
    print("launch_test returning: ", rv)
    return rv

# Get launch status - call this every 500 ms until you get a non-202 call. 
# GET /api/jobs/v2/testlaunch/{launch_uid} 
# launch_uid is the ID received in response to the POST call
# response
# 200 Launch complete and successful
# 202 Launch in progress
# 404 Launch ID is invalid
# 400 CanLaunch or Launch error (body contains string to be displayed)
# Any error codes (codes other than 200 or 202) will include an error object like this {"ErrorCode": "ERROR_OPTION","ErrorMessage": "ONA-SP-CPRI8"}
# See https://conf1.ds.jdsu.net/wiki/pages/viewpage.action?spaceKey=PSP&title=Format+of+Error+String+for+Test+Launcher+Specification
@api_node.get('/api/jobs/v2/testlaunch/<launch_uid>')
def get_launch_status(launch_uid, job_manager):
    rv = {}
    launch_uid = int(launch_uid)
    testLauncher = TestLauncher()    
    if testLauncher.is_valid_launch_index(launch_uid):
        try:
            #Returns None if still in progress, True if done and successful
            #Throws Exception with error message if fails
            cl_status = testLauncher.can_launch_status(launch_uid)
            if cl_status:
                #Returns true if launch was successful or already complete
                #Returns false if can launch isn't complete 
                #Throws Exception with error message if fails
                l_status = testLauncher.launch(launch_uid)
                if l_status:
                    response.status = 200
                else:
                    response.status = 202             
            else:            
                response.status = 202
        except LaunchException as e:
            response.status = 400
            log.debug("get_launch_status launch exception{}".format(e))
            rv["ErrorCode"] =  str(e)
            rv["ErrorMessage"] =  e.custom_error_message

    else:
        log.debug("get_launch_status invalid launch_uid: launch_uid={}".format(launch_uid))
        response.status = 404
    return rv

# Cancel launch
# DELETE /api/jobs/v2/testlaunch/{launch_uid} 
# launch_uid is the ID received in response to the POST call
# response
# 200 OK
# 404 Launch ID is invalid
@api_node.delete('/api/jobs/v2/testlaunch/<launch_uid>')
def cancel_launch(launch_uid, job_manager):
    launch_uid = int(launch_uid)
    testLauncher = TestLauncher()    
    if not testLauncher.cancel(launch_uid):
        response.status = 404

@api_node.route('/api/jobs/v2/workflowAttributes', method='PATCH')
@api_node.route('/api/jobs/v2/workflowAttributes/<work_order_id>', method='PATCH')
@use_args(CdmWorkflowSchema(strict=True))
def patch_workflow(cdm_workflow_patch, work_order_id=None, job_manager=None):
    """
    Endpoint for patching the workflow object with updated data.
    ---
    post:
        tags:
          - jobs
        description: Endpoint for patching the workflow object with updated data.
        parameters:
            -   in: body
                name: Minimal populated workflow. Only requires minimal data. 
                schema: CdmWorkflowSchema
                required: true
        responses:
            200:
                description: Update successful
            400:
                description: Failed update
    """
    log.debug('patch_workflow {}\n{}'.format(cdm_workflow_patch, work_order_id))    
    rv = job_manager.patch_workflow(work_order_id, cdm_workflow_patch)
    if not rv:
        log.debug('patch_workflow: error')
        response.status = 400

@api_node.route('/api/jobs/v2/testAttributes/<ijm_uiindex>', method='PATCH')
@use_args(CdmTestsSchema(strict=False))
def patch_test_attributes(cdm_test_patch, ijm_uiindex, job_manager):
    """
    Endpoint for patching the test object with updated attributes
    ---
    post:
        tags:
          - jobs
        description: Endpoint for patching the test object with updated attributes
        parameters:
            -   in: body
                name: Minimal populated test object with updated attributes.
                schema: CdmTestsSchema
                required: true
        responses:
            200:
                description: Update successful
            400:
                description: Failed update
    """
    log.debug('patch_test_attributes {}\n{}'.format(cdm_test_patch, ijm_uiindex))    
    rv = job_manager.patch_test_attributes(ijm_uiindex, cdm_test_patch)
    if not rv:
        log.debug('patch_workflow: error')
        response.status = 400


# POST /api/jobs/v2/tests/manualTest/<ui_index>
# userResponse: required string (examples “yes”, “no”, dropdown selection)
# comment: optional string (not used for TPA 1.0)
@api_node.post('/api/jobs/v2/tests/manualTest/<ijm_uiindex>')
@use_args(ManualStepSchema(strict=True))
def manual_step(manual_step_schema, ijm_uiindex, job_manager):
    """
    Endpoint for setting the response for manual step
    ---
    post:
        tags:
          - jobs
        description: Endpoint for setting the response for manual step
        parameters:
            -   in: body(json)
                name: NB CDM with updated response.
                schema: ManualStepSchema
                required: true
        responses:
            200:
                description: Update successful
            400:
                description: Failed update
    """
    rv = job_manager.set_manual_test_response(ijm_uiindex, manual_step_schema)
    if not rv:
        log.debug('set_manual_test_response: error')
        response.status = 400
        return rv
    return {}
