"""
Module containing endpoints for the job manager api
"""
import os
import os.path
import json
import logging
import traceback
import datetime
import time
log = logging.getLogger(__name__)

from webargs.bottleparser import use_args, use_kwargs
from bottle import response, request, Bottle

from rest_api.api.job_manager.save_job_artifacts import create_artifacts
from rest_api.api.job_manager.job import Job
from rest_api.api.job_manager.schemas_viavi import FilePathSchema
from rest_api.api.job_manager.schemas_viavi import ArtifactSchema
from rest_api.api.job_manager.plugin import JobManagerPlugin, SideEffect
from rest_api.api.job_manager import save_job_artifacts
from rest_api.api.schemas import UploadArgs
from rest_api.utils.marshmallow_compat import marshmallow_data_compat
from rest_api.api.job_manager.cdm.cdm_schemas import CdmJob, ReferenceInfoForTestTypeSchema, CdmTestsSchema
from rest_api.products.config_schema_common import get_launch_params, SubTypeRefInfoFormatter
from rest_api.api.job_manager.helper import get_test_ui_config_format
from rest_api.api.job_manager.schemas_viavi import TestIndexSchema
from rest_api.api.job_manager.helper import get_type_of_file
from rest_api.api.job_manager.save_image_to_pdf import save_image_to_pdf

METADATA_KEY = 'metadata'

####DEPRECATED - DO NOT MODIFY. USE jobmanager_v2.py

api_node = Bottle() #pylint: disable=invalid-name

#Inject job_manger object into the view function for each route
api_node.install(JobManagerPlugin())
api_node.install(SideEffect())

@api_node.get('/api/v1/cdm/workflow')
def cdm_workflow(job_manager):
    job = job_manager.get_active_job()
    if not job:
        response.status = 404
        return {}
    workflow = {}
    workflow['workflow'] = job['workflow']
    return workflow

#Use the first URL. Others are legacy.
@api_node.post('/api/jobs/v1/updateJob')
@api_node.post('/api/v1/jobs/updateJob')
@api_node.post('/api/v1/jobs/postJobUI')
@use_args(CdmJob(strict=True))
def post_job_ui(job_input, job_manager):
    """
    Method for UI to update an existing job in the database
    """
    rv = job_manager.update_job(job_input)
    if not rv:
        log.debug('UI post Job: error')
        response.status = 400
        return {'errorMessage': 'Error'}
    return {}

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/current')
@api_node.get('/api/v1/jobs/current')
def get_current_job(job_manager):
    """
    Endpoint to get details about current job
    ---
    get:
        tags:
          - jobs
        description: details about a particular job
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
        responses:
            200:
                'schema': CdmJob
                description: Job found. Return job. Empty JSON if no current job.
    """
    job = job_manager.get_active_job()
    if job:
        return job
    else:
        return {}

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/ui/<id>')
@api_node.get('/api/v1/jobs/ui/<id>')
def get_job(id, job_manager):
    """
    Endpoint to get details about a job
    ---
    get:
        tags:
          - jobs
        description: details about a particular job
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
        responses:
            200:
                'schema': CdmJob
                description: Job found. Return job. Empty JSON if no current job.
    """
    job = job_manager.get_job(id)
    if job:
        return job
    else:
        return {}

#Use the first URL. Other is legacy.
@api_node.put('/api/jobs/v1/archive/<work_order_id>')
@api_node.put('/api/v1/jobs/archive/<work_order_id>')
def archive_job(work_order_id, job_manager):
    if not job_manager.archive_job(work_order_id):
        response.status = 400

#Use the first URL. Other is legacy.
@api_node.put('/api/jobs/v1/restore/<work_order_id>')
@api_node.put('/api/v1/jobs/restore/<work_order_id>')
def restore_job(work_order_id, job_manager):
    if not job_manager.restore_job(work_order_id):
        response.status = 400


#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testData')
@api_node.post('/api/v1/jobs/active/testData')
@use_args(CdmTestsSchema(strict=True))
def add_test_data(cdm_test_data, job_manager):
    """
    Endpoint to add test data to the active job
    ---
    post:
        tags:
          - jobs
        description: adds test data to the active job
        parameters:
            -   in: body
                name: Test Data
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    #Remove "ijm_filePath" from cdm_tests and pass this as a parameter. 
    report_filepaths = cdm_test_data.pop("ijm_filePath", [])
    job = job_manager.add_test_data(cdm_test_data, False, report_filepaths, True)
    if job:
        response.status = 201
        return job
    else:
        response.status = 202
        return {}

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testDataWaitForAll')
@api_node.post('/api/v1/jobs/active/testDataWaitForAll')
@use_args(CdmTestsSchema(strict=True))
def add_test_data_wait_for_all(cdm_test_data, job_manager):
    """
    Endpoint to add test data to the active job but delay invoking the side effect
    ---
    post:
        tags:
          - jobs
        description: adds test data to the active job but delay invoking the side effect
        parameters:
            -   in: body
                name: Test Data
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    #Remove "ijm_filePath" from cdm_tests and pass this as a parameter. 
    report_filepaths = cdm_test_data.pop("ijm_filePath", [])
    job = job_manager.add_test_data(cdm_test_data, False, report_filepaths, False)
    if job:
        response.status = 201
        return job
    else:
        response.status = 202
        return {}

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testDataLegacy')
@api_node.post('/api/v1/jobs/active/testDataLegacy')
@use_args(CdmTestsSchema(strict=True))
def add_test_data_old(test_data, job_manager):
    """
    Endpoint to add test data to the active job
    ---
    post:
        tags:
          - jobs
        description: adds test data to the active job
        parameters:
            -   in: body
                name: Test Data
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    job = job_manager.add_test_data(test_data, True, [], True)
    if job:
        response.status = 201
        return job
    else:
        response.status = 202
        return {}

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testDataWaitForAllLegacy')
@api_node.post('/api/v1/jobs/active/testDataWaitForAllLegacy')
@use_args(CdmTestsSchema(strict=True))
def add_test_data_wait_for_all_old(test_data, job_manager):
    """
    Endpoint to add test data to the active job but delay invoking the side effect
    ---
    post:
        tags:
          - jobs
        description: adds test data to the active job but delay invoking the side effect
        parameters:
            -   in: body
                name: Test Data
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    job = job_manager.add_test_data(test_data, True, [], False)
    if job:
        response.status = 201
        return job
    else:
        response.status = 202
        return {}

#Use the first URL. Other is legacy.
@api_node.put('/api/jobs/v1/active/clearTestData/<test_index>/<test_location_index>')
@api_node.put('/api/v1/jobs/active/clearTestData/<test_index>/<test_location_index>')
def clear_test_data(test_index, test_location_index, job_manager):
    """
    Endpoint to clear test data of the specified planned test
    ---
    put:
        tags:
          - jobs
        description: clears the test data for the specified planned test
        responses:
            200:
                description: Success.

    """

    try:
        test_plan_index = int(test_index)
        test_location_index = int(test_location_index)
        if job_manager.clear_test_data(test_plan_index, test_location_index):
            response.status = 201
        else:
            response.status = 202
    except ValueError:
        response.status = 400
        return

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/active')
@api_node.get('/api/v1/jobs/active')
def get_active_state(job_manager):
    """
    Endpiont for determining if job manager is active
    ---
    get:
        tags:
          - jobs
        description: returns "1" if the job manager is active, "0" otherwise
        responses:
            200:
                description: Success.
    """
    active = "0"
    if job_manager.is_active():        
        active = "1"
    return active

#Use the first URL. Other is legacy.
@api_node.put('/api/jobs/v1/active')
@api_node.put('/api/v1/jobs/active')
def set_active_state(job_manager, side_effect):
    """
    Endpoint for activating/deactivating job manager
    ---
    put:
        tags:
          - jobs
        responses:
            200:
                description: Success.
                schema:
                    title: Is Active
                    type: boolean
    """
    request_body = request.body.getvalue().decode('utf-8') #pylint: disable=no-member
    try:
        activate = int(request_body)
        if activate != 0:
            job_manager.activate()
        else:
            job_manager.deactivate()
    except ValueError:
        response.status = 400

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/active/jobStatusRatio')
@api_node.get('/api/v1/jobs/active/jobStatusRatio')
def get_job_status_ratio(job_manager):
    """
    Endpoint for obtaining the active job's job status ratio
    ---
    get:
        tags:
          - jobs
        description: returns the job status ratio of the active job
        responses:
            200:
                description: Success.
                schema:
                    title: Job Status Ratio
                    type: string
    """
    status = job_manager.get_job_status()
    if status:
        return  "/".join([str(status['complete_count']),str(status['total_count'])])
    return ""

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/active/jobStatus')
@api_node.get('/api/v1/jobs/active/jobStatus')
def get_job_status(job_manager):
    """
    Return job status ratio as json. Same as above but JSON response instead of text.
    ---
    get:
        tags:
          - jobs
        description: returns the job status ratio of the active job
        responses:
            200:
                description: Success.
                schema:
                    title: Job Status Ratio
                    type: string
    """
    rv = {} 
    status = job_manager.get_job_status()
    if status:
        rv["totalNumberOfTests"] = status['total_count']
        rv["numberOfCompletedTests"] = status['complete_count']
    return rv

#Use the first URL. Other is legacy.
@api_node.put('/api/jobs/v1/active/allTestDataAdded')
@api_node.put('/api/v1/jobs/active/allTestDataAdded')
def set_all_test_data_added(job_manager):
    """
    Endpiont to execute any side effects waiting for all test data to have been added
    ---
    get:
        tags:
          - jobs
        description: execute any side effects waiting for all test data to have been added
        responses:
            200:
                description: Success.
    """
    job_manager.set_all_test_data_added()
    return {}

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/active/testDataIndex')
@api_node.get('/api/v1/jobs/active/testDataIndex')
def get_active_test_index(job_manager):
    """
    Endpiont for determining the launched job item of active test plan
    ---
    get:
        tags:
          - jobs
        description: returns an object containing the test and location index of the launched job item of active test plan
        responses:
            200:
                description: Success.
                schema:
                    title: Indices of Launched Job Item in Active Test Plan
                    type: {"testIndex" : int, "testLocationIndex" : int}. Index is -1 when there is no valid entry. 
    """
    #rv = []
    rv = {"testIndex" : -1, "testLocationIndex" : -1}
    index_list = job_manager.get_active_test_plan_index()
    if index_list:
        rv = {"testIndex" : index_list[0], "testLocationIndex" : index_list[1]}    
    return rv

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testDataIndex/<test_index>/<test_location_index>')
@api_node.post('/api/v1/jobs/active/testDataIndex/<test_index>/<test_location_index>')
def set_active_test_index(test_index, test_location_index, job_manager):
    """
    Endpiont for record the launched job item of active test plan
    ---
    get:
        tags:
          - jobs
        description: set the index of the launched job item of active test plan
        responses:
            201:
                description: Success.
            202:
                description: Failed to set test plan index
    """
    #When setting the active test index, also clear any test data associated with this index. 
    clear_test_data(test_index, test_location_index, job_manager)    
    job_manager.set_active_test_plan_index(int(test_index), int(test_location_index))

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/active/matchingTestPlanIndex')
@api_node.get('/api/v1/jobs/active/matchingTestPlanIndex')
@use_args(ReferenceInfoForTestTypeSchema(strict=True))
def get_test_plan_index(ref_info_input,job_manager):
    """
    Endpiont for determining a job item index from its test type
    and reference info
    ---
    get:
        tags:
          - jobs
        description: returns the index of the job item having the specified test type and reference info
        responses:
            404:
                description: Job Not Found
            200:
                description: Success.
                schema:
                    title: index of matching job item
                    type: integer
    """
    job = job_manager.get_active_job()

    if not job:
        response.status = 404
        return None

    response.status = 200

    ref_info = ref_info_input['reference_info']
    test_type = ref_info_input['test_type']
    index = job_manager.get_test_plan_index_of_ref(test_type, ref_info)

    if (index == 'None'):
       index = "-1";

    return str(index)

#Use the first URL. Other is legacy.
@api_node.delete('/api/jobs/v1/<work_order_id>')
@api_node.delete('/api/v1/jobs/<work_order_id>')
def remove_job(work_order_id, job_manager):
    """
    Endpoint for deleting a job
    ---
    delete:
        tags:
          - jobs
        description: deletes the job
        parameters:
            -   in: path
                name: job_id
                required: true
                type: string
        responses:
            200:
                description: Success
            401:
                description: Cannot delete active job
            404:
                description: Job Not Found
    """
    try:
        job_manager.delete_job(work_order_id)
        response.status = 200
    except ReferenceError:
        response.status = 401        
    except FileNotFoundError:
        response.status = 404

#Use the first URL. Other is legacy.
@api_node.delete('/api/jobs/v1/template/<type_name>')
@api_node.delete('/api/v1/jobs/template/<type_name>')
def remove_job(type_name, job_manager):
    """
    Endpoint for deleting a template
    ---
    delete:
        tags:
          - jobs
        description: deletes the template
        parameters:
            -   in: path
                name: template type name
                required: true
                type: string
        responses:
            200:
                description: Success
            404:
                description: Template Not Found
    """
    try:
        job_manager.delete_template(type_name)
        response.status = 200
    except FileNotFoundError:
        response.status = 404

#Use the first URL. Other is legacy.
@api_node.put('/api/jobs/v1/template')
@api_node.put('/api/v1/jobs/template')
@use_args(CdmJob(strict=True))
def add_template(cdm_template, job_manager):
    """
    Endpoint for adding a template. This may not be used in customer use case but is handy for test
    ---
    post:
        tags:
          - jobs
        description: Endpoint for adding a template. This may not be used in customer use case but is handy for test
        parameters:
            -   in: body
                name: Artifact
                schema: CdmJob
                required: true
        responses:
            200:
                description: Template successfully created
            400:
                description: Failed to add template
    """
    rv = False
    try:
        if cdm_template['workflow']['template']:
            cdm_template_workflow = {"cdm" : [cdm_template]}
            cdm_template_workflow ["cdmVersion"] = cdm_template.get("cdmVersion")
            rv = job_manager.process_cdm_template_workflow(cdm_template_workflow)
    except:
        log.debug(traceback.format_exc())
    if not rv:
        log.debug('add_template: error')
        response.status = 400
        return {'errorMessage': 'Error'}
    return {}


def save_to_pdf(artifact_args, job):
    """
    Endpoint for saving the contentes of a job to pdf
    ---
    post:
        tags:
          - jobs
        description: saves the contents of a job to a pdf report and optionally create a zip file
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
            -   in: body
                name: Artifact
                schema: ArtifactSchema
                required: true
        responses:
            201:
                description: Summary report successfully created
                schema: FilePathSchema
            403:
                description: Filepath not found
            404:
                description: Job Not Found
    """
    return create_artifacts(job=job, **artifact_args)

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/report')
@api_node.post('/api/v1/jobs/report')
@use_args(ArtifactSchema(strict=True))
def save_current_job_to_pdf(artifact_args, job_manager):
    file_path_return = None
    #To work around some legacy users of the API that always assume the active/current
    #job ID is 1, we'll assume we ned to use the active job
    job = job_manager.get_active_job_object()
    if job:
        file_path_return = save_to_pdf(artifact_args, job)

    if file_path_return:
        response.status = 201
        return {'filePath': file_path_return}
    else:
        response.status = 403
        return {}

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/report/<work_order_id>')
@api_node.post('/api/v1/jobs/report/<work_order_id>')
@use_args(ArtifactSchema(strict=True))
def save_given_job_to_pdf(artifact_args, work_order_id, job_manager):
    job = None
    file_path_return = None
    #To work around some legacy users of the API that always assume the active/current
    #job ID is 1, we'll assume we ned to use the active job
    if work_order_id == 1 or work_order_id == "1":
        job = job_manager.get_active_job_object()
    else:
        job = job_manager.get_job_object(work_order_id)
    if job:
        file_path_return = save_to_pdf(artifact_args, job)

    if file_path_return:
        response.status = 201
        return {'filePath': file_path_return}
    else:
        response.status = 403
        return {}

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/changecount')
@api_node.get('/api/v1/jobs/changecount')
def get_change_count(job_manager):
    """
    Endpoint to get the current change count which is used by UI to when any job data has changed
    ---
    get:
        tags:
          - jobs
        description: get the change count
        responses:
            200:
                description: Job found. Return job. Empty JSON if no current job.
    """
    return {"changeCount" : job_manager.get_change_count()}

#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/testtypes')
@api_node.get('/api/v1/jobs/testtypes')
def get_test_types(job_manager):
    test_types = {"test_types" : []}
    try:
        #test_type = request.body.getvalue().decode('utf-8') #pylint: disable=no-member   
        test_types["test_types"] = list(request.app.config["rest_api.product_specific"].test_definitions.keys())
        response.status = 200
    except:
        print(traceback.format_exc())
        response.status = 400
    return test_types

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/testlaunch')
@api_node.post('/api/v1/jobs/testlaunch')
def get_test_launch_params(job_manager):
    launch_params = None
    response.status = 404
    #test_type = test_type.strip()    
    try:
        test_type = request.body.getvalue().decode('utf-8').strip() #pylint: disable=no-member
        #test_type = request.body.getvalue().decode('utf-8') #pylint: disable=no-member   
        test_definitions = request.app.config["rest_api.product_specific"].test_definitions
        launch_params = get_launch_params(test_definitions[test_type])
        if launch_params:
            response.status = 200
    except:
        print(traceback.format_exc())
    return launch_params

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testUiConfigFormat')
@api_node.post('/api/v1/jobs/active/testUiConfigFormat')
@use_args(CdmTestsSchema())
def get_test_ui_config_format_no_test_location_index(cdm_test, job_manager):
    location_index = 0
    if not "testLocations" in cdm_test:
        location_index = -1
    return get_test_ui_config_format(cdm_test, 0)    

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testUiConfigFormat/<test_location_index:int>')
@api_node.post('/api/v1/jobs/active/testUiConfigFormat/<test_location_index:int>')
@use_args(CdmTestsSchema())
def get_test_ui_config_format_with_test_location_index(cdm_test, test_location_index, job_manager):
    return get_test_ui_config_format(cdm_test, test_location_index)

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/testUiConfigFormat/<test_index:int>/<test_location_index:int>')
@api_node.post('/api/v1/jobs/active/testUiConfigFormat/<test_index:int>/<test_location_index:int>')
@use_args(CdmJob(strict=False))
def get_test_ui_config_format_from_job(cdm_job, test_index, test_location_index, job_manager):
    cdm_test = cdm_job["tests"][test_index]
    return get_test_ui_config_format(cdm_test, test_location_index)

#Get a list of all UI configurations. 
#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/allTestUiConfigFormat')
@api_node.post('/api/v1/jobs/active/allTestUiConfigFormat')
@use_args(CdmJob(strict=False))
def get_all_test_ui_config_format_from_job(cdm_job, job_manager):
    if not cdm_job:
        cdm_job = job_manager.get_active_job()
    ui_config_formats = []
    try:        
        if "tests" in cdm_job:
            for cdm_test in cdm_job["tests"]:
                #for i in len(cdm_test["testLocations"]):
                if "testLocations" in cdm_test:
                    try:
                        for idx, val in enumerate(cdm_test["testLocations"]):
                            formatter = get_test_ui_config_format(cdm_test, idx)
                            ui_config_formats.append(formatter)
                    except:
                        print(traceback.format_exc())
                else:
                    #No locations
                    ui_config_formats.append(get_test_ui_config_format(cdm_test, -1))
    except:
        print(traceback.format_exc())
    response.status = 200
    rv = {"ui_config_formats" : ui_config_formats}
    return rv

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/jobManagerFile')
@api_node.post('/api/v1/jobs/jobManagerFile')
@use_args(FilePathSchema(strict=True))
def load_job_from_file(file_path_input, job_manager, side_effect):
    """
    Endpoint for loading a job to the datastore from a file located at a path supplied by the client.
    This is a legacy API for loading jobs. However it can still be used to import new jobs. 
    ---
    post:
        tags:
          - jobs
        description: loads job to the datastore from a file located at a path supplied by the client
        parameters:
            -   in: body
                name: FilePathSchema
                schema: FilePathSchema
                required: true
        responses:
            201:
                description: Job loaded successfully
                schema: CdmJob
            422:
                description: Cannot process request for a reason supplied by the response
    """
    file_path = file_path_input['file_path']

    if not os.path.isfile(file_path):
        response.status = 422
        return {"filePath": ["Bad file path"]}

    output_job = job_manager.load_job_from_file(file_path)

    if not output_job:
        response.status = 422
        return{file_path: ["Invalid job file"]}

    if job_manager.is_active():
        side_effect(output_job)

    response.status = 201
    return output_job


@api_node.error(422)
def handle_unprocessable_entity(error):
    """
    Method to make error 422 error responses raised from webargs
    more consumer friendly

    The response body will now contain a representation of the schema
    parsing error in the request.  The message will contain information
    like:

    '{"customerName": ["Missing data for required field."]}'

    Args:
        error (HTTPError): the error raised by webargs due to parsing errors
    """
    log.debug(str("ERROR: %s" % str(error.body)))
    return error.body


@api_node.post('/api/testdata/v1/userfiles', apply=JobManagerPlugin())
@api_node.post('/api/signature/v1/userfiles', apply=JobManagerPlugin())
@use_kwargs(UploadArgs(strict=True), locations=('files', 'form'))
def save_item(filedata, job_manager):

    """Endpoint for uploading an additional piece of testdata to the instrument
       like a signature capture, bar code , etc.
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload testdata to the instrument
      consumes:
        - multipart/form-data
      parameters:
        - in: formData
          schema: UploadArgs
      responses:
        200:
          description: success
    """
    reports_dir = request.app.config.get('user_files.reports_dir')

    filepath = os.path.join(reports_dir, filedata.filename)

    filedata.save(filepath, overwrite=True)

    job = job_manager.get_active_job_object()

    if not job_manager.is_active() or not job:
        return

    #mobiletech sends the metadata as a 'files' type (while you might expect it
    #as 'forms', so we had to uses request.files to access it and convert it to a python object

    metadatafile = request.files.get(METADATA_KEY) #pylint: disable=no-member
    metadata = json.loads(metadatafile.file.read().decode('utf-8'))
    test_data = save_image_to_pdf(
        filepath,
        job.get_logo(),
        get_type_of_file(metadata),
        job,
    )

    job_manager.add_test_data(test_data, True)

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/matchingTestAndLocationIndex')
@api_node.post('/api/v1/jobs/active/matchingTestAndLocationIndex')
@use_args(CdmTestsSchema(strict=True))
def get_test_and_location_index_for_cdm_test(cdm_test,job_manager):
    """
    Endpiont for finding test and location indices from a given CdmTest object
    ---
    get:
        tags:
          - jobs
        responses:
            404:
                description: Match Not Found
            200:
                description: Success.
                schema:
                    title: indices of matching job item
                    type: {"testIndex" : int, "testLocationIndex" : int}. Index is -1 when there is no valid entry. 
    """
    #rv = []
    rv = {"testIndex" : -1, "testLocationIndex" : -1}
    index_list = job_manager.get_test_index_of_cdm_test(cdm_test)
    test_index = -1
    test_location_index = -1

    if index_list:
        if index_list[0] != None and index_list[0] >= 0:
            test_index = index_list[0]
        if index_list[1] != None and index_list[1] >= 0:
            test_location_index = index_list[1]    
        rv = {"testIndex" : test_index, "testLocationIndex" : test_location_index}
    else:
        response.status = 404
        return
    return rv

#Use the first URL. Other is legacy.
@api_node.post('/api/jobs/v1/active/getMetaDataForTest')
@api_node.post('/api/v1/jobs/active/getMetaDataForTest')
@use_args(CdmTestsSchema(strict=True))
def get_metadata_for_cdmtest(cdm_test,job_manager):
    """
    Endpiont for finding test and location indices from a given CdmTest object
    ---
    get:
        tags:
          - jobs
        responses:
            404:
                description: Match Not Found
            200:
                description: Success.
                schema:
                    title: indices of matching job item
                    type: {"indexType" : string("workflowId" or "testPlanIndex"), "id" : int}. id is -1 when there is no valid entry. 
    """
    #rv = []
    rv = job_manager.get_metadata_for_cdmtest(cdm_test)
    if not rv:
        response.status = 404
        return
    return rv


#Use the first URL. Other is legacy.
@api_node.get('/api/jobs/v1/currentTestAndLocation')
@api_node.get('/api/v1/jobs/currentTestAndLocation')
def get_current_test_and_location(job_manager):
    """
    Endpoint to get the launched CDM test and location block
    ---
    get:
        tags:
          - jobs
        responses:
            200:
                'schema': CdmTest
                description: If there is an active test/location that has been launched by job manager, the CDM test/location block is returned
            404:
                description: There is no active/launched test
    """
    cdm_test_location = job_manager.get_active_test_and_location()
    if not cdm_test_location:
        response.status = 404
        return {}
    else:
        return cdm_test_location



    

