import os
try:
    import watchdog.observers
    import watchdog.events
    import time
except:
    pass

import traceback
import logging
log = logging.getLogger(__name__)

class WorkflowFileHandler(watchdog.events.FileSystemEventHandler):
    #A class list of all observers so the observers don't go out of scope
    def __init__(self, workflow_path, callback):
        """
        Watch for incoming workflow files and make a callback if updates are received

        Args:
            
        Returns:
            
        """
        try:
            #Before starting to monitor this file we need to make sure it exists. If it doesn't create an empty file. 
            self.workflow_path = workflow_path
            self.workflow_dir = os.path.dirname(workflow_path)
            self.callback = callback
            if not os.path.isdir(self.workflow_dir):
                os.mkdir(self.workflow_dir)

            if not os.path.isfile(workflow_path):
                with open(workflow_path, "w") as workflow_file:
                    workflow_file.write("")

            self.workflow_observer = watchdog.observers.Observer()            
            self.workflow_observer.schedule(self, self.workflow_dir, recursive=False)
            self.workflow_observer.start()
        except:
            log.error("Failed to start workflow file watcher")
            log.error(traceback.format_exc())

    def process_change(self, changed_file):
        log.debug("WorkflowFileHandler process_change: {}".format(changed_file))
        print("WorkflowFileHandler process_change: {} {}".format(changed_file, self.workflow_path))
        try:
            #The file may not be fully written when we get this signal. Make this thread sleep for a finite amount of time. 
            if changed_file == self.workflow_path:
                time.sleep(1)
                with open(changed_file, "r") as f:
                    workflow_json_string = f.read()
                    if workflow_json_string:
                        log.debug("Received new workflow updates: {}".format(changed_file))
                        print("Received new workflow updates: {}".format(changed_file))
                        print("Received new workflow callback: {}".format(self.callback))
                        self.callback(workflow_json_string)
        except:
            pass

    #This gets called when the workflow file changes.
    def on_modified(self, event):
        try:
            if type(event) == watchdog.events.FileModifiedEvent:
                self.process_change(event.src_path)
        except:
            log.error(traceback.format_exc())

    #This gets called when the workflow file get created.
    def on_created(self, event):
        try:
            if type(event) == watchdog.events.FileCreatedEvent:                   
                self.process_change(event.src_path)
        except:
            log.error(traceback.format_exc())

    def on_moved(self, event):
        try:
            self.process_change(event.src_path)
        except:
            log.error(traceback.format_exc())


class ImportedJobsFileHandler(watchdog.events.FileSystemEventHandler):
    #callbacks is a dictionary of file extension->callback (e.g. {'.job.json' : process()})
    def __init__(self, jobs_dir, callbacks):
        self.jobs_dir = jobs_dir
        self.callbacks = callbacks
        try:
            log.debug("ImportedJobsFileHandler: {}".format(self.callbacks))
            #We are going to monitor jobs_dir for changes in the file.
            #Before starting to monitor this file we need to make sure it exists. If it doesn't create an empty file. 
            if not os.path.isdir(jobs_dir):
                log.debug("watch_for_imported_jobs directory does not exist: {}".format(jobs_dir))
                return

            self.imported_jobs_observer = watchdog.observers.Observer()
            self.imported_jobs_observer.schedule(self, jobs_dir, recursive=False)
            self.imported_jobs_observer.start()
        except:
            log.error("Failed to start ImportedJobsFileHandler")
            log.error(traceback.format_exc())

    def process_change(self, filename):
        try:
            #if not filename.startswith('.') and filename.endswith('.json'):
            if not filename.startswith('.'):
                for extension,callback in self.callbacks.items():
                    if filename.endswith(extension):
                        #The file may not be fully written when we get this signal. Make this thread sleep for a finite amount of time. 
                        time.sleep(1)                
                        with open(filename, "r") as f:
                            workflow_json_string = f.read()
                            if workflow_json_string:
                                log.debug("ImportedJobsFileHandler Received new workflow updates: {}, Calling {}".format(filename, callback))
                                callback(workflow_json_string)      
        except:
            pass

    #This gets called when a job file is created.
    def on_created(self, event):
        try:
            log.debug("ImportedJobsFileHandler on_created: {}".format(event.src_path))
            self.process_change(event.src_path)
        except:
            pass

    def on_moved(self, event):
        try:
            log.debug("ImportedJobsFileHandler on_moved: {}".format(event.dest_path))
            self.process_change(event.dest_path)
        except:
            pass
