"""
Module containing code to save job artifacts

Includes funcitons to save the job to a summary PDF file, a consolidated
PDF file and as a zip file
"""
import os
from rest_api.api.job_manager.helper import html_format_job_info
import zipfile
from xml.etree import ElementTree
from subprocess import call
from traceback import print_exc

import PyPDF2
from bottle import template

from . import report_languages

REPORT_TEMPLATE_PATH = os.path.join(os.path.dirname(__file__), './views/job_template.tpl')
TEMP_HTML_FILE = os.path.join(os.path.dirname(__file__), './views/temp_report_html.html')
IMAGE_TEMPLATE_PATH = os.path.join(
    os.path.dirname(__file__),
    './views/image_template.tpl'
)

def save_job_to_pdf(job, file_path):
    """
    Function to save the job object as a pdf file

    Args:
        job (Job): the job object to save as a pdf file
        file_path: the desired file path for the output pdf
    """
    report_strings = report_languages.get_report_strings('en_US')

    job_html = template(REPORT_TEMPLATE_PATH, job=job, i18n_strings=report_strings)

    return save_html_str_to_pdf(job_html, file_path)


def save_image_to_pdf(image, file_path, logo_path, job, image_type):
    """Function to save an image as a pdf document so that it can be
    concatenated with other test reports
    Args:
        image (str): the path of the image
        file_path (str): the desired file path for the output pdf
        logo_path (str): the filepath of the customer logo
        job (Job): the currently active job object for report metadata
        image_type (str): the type of image being saved
    """
    html = template(
        IMAGE_TEMPLATE_PATH,
        logo_path=logo_path,
        image=image
    )
    save_html_str_to_pdf(html, file_path)

    # Add metadata
    metadata = create_metadata(job.cdm_job, image_type)

    merger = PyPDF2.PdfFileMerger(strict=False)
    with open(file_path, 'rb') as pdf_file:
        reader = PyPDF2.PdfFileReader(pdf_file)
        merger.append(reader)
    merger.addMetadata(metadata)
    merger.write(file_path)

    return file_path


def save_html_str_to_pdf(html, file_path):
    """Function to save html as a string to a pdf

    Args:
        html (str): the html to save as a pdf
        file_path: the desired file path for the output pdf
    """
    with open(TEMP_HTML_FILE, 'w') as html_file:
        html_file.write(html)

    try:
        ret_code = call(['qt-html-to-pdf', TEMP_HTML_FILE, file_path])
    except OSError:
        print_exc()
        ret_code = 1

    os.remove(TEMP_HTML_FILE)

    if ret_code != 0:
        return None

    return file_path


def create_artifacts(job, file_path, report_type, create_zip):
    """
    Method to create artifacts of the job.  Can create either a summary
    or consolidated report and optionally add all test data to a zip file

    Args:
        job (Job): the job object that the artifacts are for
        file_path (str): the output file path for the summary or consolidated report
        report_type (str): the user's choice about whether the report should be
            a summary report or a consolidated report (including) all PDF test data
        create_zip (bool): flag to determine if the summary/consolidated report
            and all test data should be included in a zip file
    Returns:
        job_pdf (str): the file path to the pdf report
    """
    job_file_list = make_file_list(job)
    metadata = create_metadata(job.cdm_job, 'Job Report', job.status)
    temp_summary_pdf = save_job_to_pdf(job, 'temp_summary.pdf')
    merger = PyPDF2.PdfFileMerger(strict=False)
    merger.append(temp_summary_pdf, bookmark='Job Summary')
    if report_type != 'summary':
        for jf in job_file_list:
            try:
                merger.append(jf["filepath"], bookmark=jf["bookmark"])
            except PyPDF2.utils.PdfReadError:
                pass
    merger.addMetadata(metadata)
    merger.write(file_path)
    os.remove('temp_summary.pdf')
    job_pdf = file_path
    if create_zip:
        zip_file_list = make_files_to_zip(job_report=job_pdf,
                                          file_list=job_file_list)
        output_zip = zip_file_path(file_path)
        zip_all_files(output_zip, zip_file_list)

    return job_pdf

def zip_file_path(pdf_file_path):
    """
    Method to create a zip file path based on a pdf file path.  Simply drops the
    .pdf extension and substitues a .zip extension
    """
    return pdf_file_path.replace('.pdf', '.zip')

def make_file_list(job):
    """Function to create a list of files for the job.  This list is used to
    create consolidated PDF reports or .zip files for the job

    Args:
        job (Job): the job to create the consolidatd report for

    Returns:
        file_list (list): list of dictionaries contaning file_path and
            bookmark keys
    """
    file_list = []

    # updated function to get file path from cdm
    results = []
    for planned_test in job.cdm_job['tests']:
        if planned_test.get('testLocations'):
            #Handle case where testLocations exist
            for testLocation in planned_test['testLocations']:
                if 'results' in testLocation:
                    results.append(testLocation['results'])
        else:
            #No test locations
            if 'results' in planned_test:
                results.append(planned_test['results'])

    for result in results:
        if 'data' in result:
            append_file_info(file_list, result['data'])
            for previous_data in result['data'].get('ijm_testData',[]):
                append_file_info(file_list, previous_data.get('data',None))
    #Filter for only unique paths
    file_set = set(file_list)
    file_paths = []
    for f in file_set:
        file_name = os.path.basename(f)
        if os.path.isfile(f):
            file_paths.append(
                {
                    'filepath': f,
                    'bookmark': file_name
                }
            )

    return file_paths


def append_file_info(file_list, test_data):
    """Function to append information file information to the list of files
    that will be either merged in a pdf for placed into a zip file

    If the file does not exist on the file system it will not be added to the
    list

    Args:
        file_list (list): list of dictionaries contaning file_path and
            bookmark keys
        test_data (dict): dictionary of the test data including a filepath
    """
    # updated function to get file path from cdm
    if test_data:
        #ijm_filePath can be a single string or an list of strings
        file_paths = test_data.get('ijm_filePath', '')
        if type(file_paths) is not list:
            file_paths = [file_paths]
        for file_path in file_paths:
            file_list.append(file_path)

def make_files_to_zip(job_report, file_list):
    """Funciton to create teh list of files to include in the zip file

    Args:
        job_report (str): the filepath of the temporary summary report
        file_list (list): list of dictionaries containing file_path and
            bookmark keys

    Returns:
        files_to_zip (list): list of dictionaries withthe summary report
            dictionary included
    """
    files_to_zip = [{'filepath': job_report}]
    files_to_zip.extend(file_list)

    return files_to_zip

def create_metadata(job, test_type, verdict=None):
    """Function to create a metadata dictionary for USC based on the job

    args:
        job (Job): a job object
        test_type (str): the type of test

    returns:
        dict with the a single `Keywords` key and the XML metadata as the value
    """

    # updated function to get metadata from cdm
    job_info = html_format_job_info(job)

    meta_data = ElementTree.XML("<Metadata></Metadata>")
    if job_info.get('customer_name'):
        child = ElementTree.Element("CustomerName")
        child.text = job_info['customer_name']
        meta_data.append(child)

    if job_info.get('test_location'):
        child = ElementTree.Element("TestLocation")
        child.text = job_info['test_location']
        meta_data.append(child)

    if job_info.get('test_location_b'):
        child = ElementTree.Element("TestLocationb")
        child.text = job_info['test_location_b']
        meta_data.append(child)

    if job_info.get('job_comments'):
        child = ElementTree.Element("Comments")
        child.text = job_info['job_comments']
        meta_data.append(child)

    if job_info.get('technician_id'):
        child = ElementTree.Element("TechnicianId")
        child.text = job_info['technician_id']
        meta_data.append(child)

    child = ElementTree.Element("Type")
    child.text = test_type
    meta_data.append(child)

    child = ElementTree.Element("Workorder")
    child.text = job_info['job_number']
    meta_data.append(child)

    if job.get('workflow'):
        if job['workflow'].get('workflowId'):
            child = ElementTree.Element("WorkflowId")
            child.text = str(job['workflow']['workflowId'])
            meta_data.append(child)

    if verdict:
        child = ElementTree.Element("Verdict")
        child.text = verdict
        meta_data.append(child)

    if job_info.get('contractor_id'):
        child = ElementTree.Element("ContractorId")
        child.text = job_info['contractor_id']
        meta_data.append(child)

    if job_info.get('cable_id'):
        child = ElementTree.Element("CableId")
        child.text = job_info['cable_id']
        meta_data.append(child)

    return {"/Keywords": ElementTree.tostring(meta_data)}


def zip_all_files(output_file, files_to_zip):
    """
    Method to add all files into a zip file

    Args:
        output_file (str): desired file path for the output file
        files_to_zip (list): list of dictionaries containing file_path keys
    """
    with zipfile.ZipFile(output_file,
                         'w',
                         compression=zipfile.ZIP_DEFLATED) as zipper:

        for file_to_zip in files_to_zip:
            filename = file_to_zip['filepath']
            arcname = os.path.basename(filename)

            zipper.write(filename=filename, arcname=arcname)
