import traceback
import copy
import logging
from collections import OrderedDict
from rest_api.products.config_schema_common import SubTypeRefInfoFormatter
from rest_api.api.job_manager.helper import cdmtest_applies_to_instrument

log = logging.getLogger(__name__)

#View of job that filters out tests that don't apply to this instrument (i.e. deployableTo doesn't match) and
#also adds ijm_uiindex to each test/location for UI identification of a particular test/location
class JobUiView:
    product_specific = None
    def __init__(self, cdm_job, assetType, model):
        self.assetType = assetType
        self.model = model
        self.update(cdm_job)

    def update(self, cdm_job):
        self.ui_job = copy.deepcopy(cdm_job)
        if self.ui_job and 'tests' in self.ui_job:
            self.add_test_location_index(self.ui_job)
            self.filter_non_applicable_tests(self.ui_job, self.assetType, self.model)
            self.add_ui_data(self.ui_job)

    #cdm_job filtered by deployable to and has ijm_indices added
    @property
    def cdm(self):
        return self.ui_job

    def get_ui_view_sorted_by_location(self):
        cdm_job_copy = copy.deepcopy(self.ui_job)
        test_locations = OrderedDict()
        tests_without_locations = []
        for cdm_test in cdm_job_copy.get("tests", []):
            try:
                if "testLocations" in cdm_test:
                    for location in cdm_test["testLocations"]:
                        location_label = location["label"]
                        if location_label not in test_locations:
                            test_locations[location_label] = []
                        cdm_test_copy = copy.deepcopy(cdm_test)
                        cdm_test_copy.pop("testLocations")
                        cdm_test_copy["testLocations"] = [location] 
                        test_locations[location_label].append(cdm_test_copy)
                else:                    
                    tests_without_locations.append(cdm_test) 
            except:
                log.info(traceback.format_exc())
        if tests_without_locations:
            test_locations["No Location"] = tests_without_locations
        cdm_job_copy["tests"] = []
        for k,v in test_locations.items():
            for test in v:
                cdm_job_copy["tests"].append(test)
        return cdm_job_copy

    def get_ui_view_sorted_by_testtype(self):
        cdm_job_copy = copy.deepcopy(self.ui_job)
        test_types = OrderedDict()
        for cdm_test in cdm_job_copy.get("tests", []):
            try:
                test_type = cdm_test["type"]
                if test_type not in test_types:
                    test_types[test_type] = []
                if "testLocations" in cdm_test:
                    for location in cdm_test["testLocations"]:
                        cdm_test_copy = copy.deepcopy(cdm_test)
                        cdm_test_copy.pop("testLocations")
                        cdm_test_copy["testLocations"] = [location] 
                        test_types[test_type].append(cdm_test_copy)
                else:                    
                    test_types[test_type].append(cdm_test) 
            except:
                log.info(traceback.format_exc())
        cdm_job_copy["tests"] = []
        for k,v in test_types.items():
            for test in v:
                cdm_job_copy["tests"].append(test)
        return cdm_job_copy

    def filter_non_applicable_tests(self, cdm_job, assetType, model):
        cdm_job['tests'] = [test for test in cdm_job['tests'] if cdmtest_applies_to_instrument(test, assetType, model)]

    #Add a ijm_uiindex to each test and location.
    #The ijm_uiindex can either be of the form 't' (test index) (e.g. '0') or 't_l' (test and location index) (e.g. '0_1')
    def add_test_location_index(self, cdm_job):
        test_index = 0
        for test in cdm_job.get('tests'):
            test['ijm_uiindex'] = str(test_index)
            locations = test.get('testLocations')
            if locations:
                location_index = 0
                for location in locations:
                    location['ijm_uiindex'] = '{}_{}'.format(test_index, location_index)
                    location_index += 1
            test_index += 1

    def get_test_ui_config_format(self, cdm_test, test_location_index):
        rv = None
        try:
            test_type = cdm_test["type"]
            test_definitions = self.product_specific.test_definitions
            test_type = test_definitions.get(test_type, None)
            
            rv = None
            #Default to SubTypeRefInfoFormatter
            ui_config_parameters_func = SubTypeRefInfoFormatter.ui_config_parameters
            if hasattr(test_type, 'ui_config_parameters'):
                #Check if the test has a custom ui_config_parameters function
                ui_config_parameters_func = test_type.ui_config_parameters
            elif hasattr(self.product_specific, 'ui_config_parameters'):
                #Check if the product has a custom ui_config_parameters function
                ui_config_parameters_func = self.product_specific.ui_config_parameters
            if ui_config_parameters_func:
                rv = ui_config_parameters_func(cdm_test, test_location_index)
        except:
            log.info(traceback.format_exc())
        return rv

    #Add the UI display data for each test/location
    def add_ui_data(self, cdm_job):
        for cdm_test in cdm_job.get("tests", []):
            try:
                if "testLocations" in cdm_test:
                        for idx, location in enumerate(cdm_test["testLocations"]):
                            ijm_ui_extra = self.format_uiextra(self.get_test_ui_config_format(cdm_test, idx))
                            if ijm_ui_extra:
                                location['ijm_uiextra'] = ijm_ui_extra
                else:
                    #No locations
                    ijm_ui_extra = self.format_uiextra(self.get_test_ui_config_format(cdm_test, -1))
                    if ijm_ui_extra:
                        cdm_test['ijm_uiextra'] = ijm_ui_extra

            except:
                log.info(traceback.format_exc())        
    
    def format_uiextra(self, config_format):
        uiextra = ""
        if config_format:
            if type(config_format) is dict:
                index = 0
                length = len(config_format)
                for k,v in config_format.items():
                    index += 1
                    suffix = ""
                    if index <= length:
                        suffix = "\n"
                    uiextra += "{} : {}{}".format(k,v, suffix)
            elif type(config_format) is str:
                uiextra = config_format
        uiextra = uiextra.strip()
        return uiextra


#Get the test and location array indexes of full job from ijm_uiindex
#Returns tuple of test, location index (-1 for location indicates a test index)
def get_test_location_index_from_uiindex(ijm_uiindex):
    rv = (None, None)
    try:        
        sijm_uiindex = ijm_uiindex.split("_")
        l = len(sijm_uiindex) 
        if l == 1:
            t = int(sijm_uiindex[0])
            rv = (t, -1)
        if l == 2:
            t = int(sijm_uiindex[0])
            l = int(sijm_uiindex[1])                
            rv = (t, l)
    except:
        log.info(traceback.format_exc())
    return rv

def get_uiindex_from_test_and_location_index(test_index, location_index):
    rv = ""
    if test_index >= 0:
        rv = str(test_index)
    if location_index >= 0:
        rv += "_{}".format(location_index)
    return rv
