# The MIT License (MIT)

# Copyright (c) 2019 Travis Clarke <travis.m.clarke@gmail.com> (https://www.travismclarke.com/)

# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

from collections import Counter
from copy import deepcopy
from collections.abc import Mapping
from functools import reduce, partial
import logging

log = logging.getLogger(__name__)

def _handle_merge(destination, source, key):
    if isinstance(destination[key], Counter) and isinstance(source[key], Counter):
        # Merge both destination and source `Counter` as if they were a standard dict.
        _deepmerge(destination[key], source[key])
    else:
        # If a key exists in both objects and the values are `different`, the value from the `source` object will be used.
        destination[key] = deepcopy(source[key])

def _is_recursive_merge(a, b):
    both_mapping = isinstance(a, Mapping) and isinstance(b, Mapping)
    both_counter = isinstance(a, Counter) and isinstance(b, Counter)
    return both_mapping and not both_counter

def _deepmerge(dst, src):
    for key in src:
        if key in dst:
            if _is_recursive_merge(dst[key], src[key]):
                # If the key for both `dst` and `src` are both Mapping types (e.g. dict), then recurse.
                _deepmerge(dst[key], src[key])
            elif dst[key] is src[key]:
                # If a key exists in both objects and the values are `same`, the value from the `dst` object will be used.
                pass
            else:
                _handle_merge(dst, src, key)
        else:
            # If the key exists only in `src`, the value from the `src` object will be used.
            dst[key] = deepcopy(src[key])
    return dst    

def merge(destination, *sources):
    return reduce(partial(_deepmerge), sources, destination)  

#Merge the remote job into the local job
#If there is a conflict no change is made to local job
def merge_job(local_job, remote_job):
    rv = False
    local_job_template_name = local_job["workflow"].get("typeName", "")
    remote_job_template_name = remote_job["workflow"].get("typeName", "")
    if local_job_template_name != remote_job_template_name:
        #Job conflict, we just keep local job
        log.debug("merge_job template name conflict local_job_template_name:{} remote_job_template_name:{}".format(local_job_template_name, remote_job_template_name))
        return False

    remote_job_tests = remote_job.pop("tests", [])
    local_job_tests = local_job.pop("tests", [])
    #Assume that remote job will have tests if local job has tests (i.e. remote job always appends to local test list)
    merged_tests = merge_tests(local_job_tests, remote_job_tests)
    if merged_tests:
        merge(local_job, remote_job)
        local_job["tests"] = merged_tests
        rv = True
    else:
        #else - if merge_tests fails then we don't modify local job so just tack its tests back on. 
        local_job["tests"] = local_job_tests    

    #local_job is the merged job
    return rv

#Attempt to merge remote and local test lists. 
#Returns None if there is a conflict
def merge_tests(local_test_list, remote_test_list):
    merged_tests = []
    try:
        for idx, remote_test in enumerate(remote_test_list):
            if idx >= len(local_test_list):
                #We have hit the end of the local tests, so just append the remaining remote tests to the end of the merged array
                merged_tests.extend(remote_test_list[idx:])
                break
            else:
                merged_test = merge_test(local_test_list[idx], remote_test)
                if merged_test == None:
                    #There was a merge conflict, return None
                    log.debug("merge_tests test conflict")
                    return None
                else:
                    merged_tests.append(merged_test)
    except:
        merged_tests = None
    return merged_tests

#Attempt to merge remote and local test. 
#Returns None if there is a conflict
def merge_test(local_test, remote_test):
    merged_test = None
    merged_locations = []    
    if remote_test["type"] == local_test["type"]:
        remote_locations = remote_test.pop("testLocations", None)
        local_locations = local_test.pop("testLocations", None)
        if remote_locations: 
            #Absurd edge case: If we have remote locations but not local locations and a result has been saved at the local test object 
            #remove it since it is no longer valid
            local_test.pop("results", None)
            #Iterate the remote locations
            for idx, remote_location in enumerate(remote_locations):
                if not local_locations or idx >= len(local_locations):
                    #We have hit the end of the local locations, so just append the remaining remote tests to the end of the merged array
                    merged_locations.extend(remote_locations[idx:])
                    break
                else:
                    merged_location = merge_location(local_locations[idx], remote_location)
                    if merged_location == None:
                        #There was a merge conflict, return None
                        log.debug("merge_test merged_location fail")
                        return None
                    else:
                        merged_locations.append(merged_location)
    else:
        #Tests type do not match, test conflict, return None
        log.debug("merge_test test types do not match")
        return None

    #Merge the tests then add in the merged locations
    merge(local_test, remote_test)
    if merged_locations:
        local_test["testLocations"] = merged_locations              

    return local_test

def merge_location(local_location, remote_location):
    remote_location_tpi = remote_location.get("testPlanIndex", None)
    local_location_tpi = local_location.get("testPlanIndex", None)
    remote_location_wfi = remote_location.get("workflowId", None)
    local_location_wfi = local_location.get("workflowId", None)
    #If we have different incoming testPlanIndex than the local testPlanIndex we need to clean up results if we have any
    if remote_location_tpi and remote_location_tpi != local_location_tpi:
        local_location.pop("results", None)
    #If we have different local workflowId than the imcoming workflowId we need to clean up results if we have any        
    if local_location_wfi and remote_location_wfi != local_location_wfi:
        local_location.pop("results", None)
    merge(local_location, remote_location)
    return local_location
