"""
Module containing classes and helper functions for
manipulating jobs
"""
import logging
import json
import os
import traceback
import datetime
import time
import copy
import shutil
from . import data_store
from .job import Job
from .job_uiview import JobUiView
from .template import convert_template_to_job, default_template_typename
from .cdm import cdm_schemas
from .workflow_file_watcher import WorkflowFileHandler, ImportedJobsFileHandler
from rest_api.utils.marshmallow_compat import marshmallow_data_compat
from .helper import set_job_to_locally_managed, test_is_additional, get_workorder_ids, get_next_test_plan_index, get_template_from_template_array, job_has_test_plan_index
from .helper import get_job_from_workflow, set_job_modified_to_now, get_externally_managed_job_workorder_ids, merge_jobs, get_local_timestamp_8601, is_job_archived
#get_internally_managed_job_workorder_ids_from_job_info
#from .helper import get_conflicted_workorder_ids_from_workflow, get_conflicted_workorder_ids_from_job_info, set_job_conflict, clear_job_conflict
from rest_api.api.job_manager.job_uiview import get_test_location_index_from_uiindex
from rest_api.api.job_manager.job_uiview import get_uiindex_from_test_and_location_index
from rest_api.api.job_manager.import_locations import import_csv_to_job
from .job_merge import merge as merge_dict

log = logging.getLogger(__name__)


class JobManager(object):
    '''
    Class for managing job data

    Attibutes:
        job: a Job object used for tranforming the the information in a job
             None prior to loading the job or if the job does not exist
    '''
    index_of_last_added_test_data = []

    def __init__(self, job_manager_file, jobs_dir, product_specific=None, ss_workflow_path=None, ss_template_path=None):
        self.job_manager_file = job_manager_file
        self.data_store = data_store.DataStore(job_manager_file)
        self.job = None      
        #This is incremented whenever there is a change to any job. This is used
        #by the UI to determine when it needs to update data   
        self.change_count = 0
        self.product_specific = product_specific 
        self.jobs_dir = jobs_dir
        #This path is registered with USC by cdm_sync and is used for northbound templates and job instance files
        if jobs_dir:
            self.northbound_jobs_cdm_dir = os.path.join(jobs_dir, "northbound_jobs/")
            if not os.path.exists(self.northbound_jobs_cdm_dir):   
                os.makedirs(self.northbound_jobs_cdm_dir)
            self.northbound_cdm_dir = os.path.join(jobs_dir, "northbound_cdm/")
            if not os.path.exists(self.northbound_cdm_dir):   
                os.makedirs(self.northbound_cdm_dir)
            self.job_artifacts_dir = os.path.join(jobs_dir, "job_artifacts/")
            if not os.path.exists(self.job_artifacts_dir):   
                os.makedirs(self.job_artifacts_dir)
        #Add workflow watchers
        if ss_workflow_path:
            self.template_watcher = WorkflowFileHandler(ss_workflow_path, self.process_cdm_json_workflow)
        if ss_template_path:
            self.template_watcher = WorkflowFileHandler(ss_template_path, self.process_cdm_template_workflow)
        if product_specific and jobs_dir:
            job_extension = self.product_specific.get_job_extension()
            #Replace the * with empty to get the extension
            job_extension = job_extension.replace('*', '')
            template_extension = self.product_specific.get_template_extension()
            template_extension = template_extension.replace('*', '')
            #Add imported job watcher. Map template and job files to the handlers.
            self.imported_jobs_watcher = ImportedJobsFileHandler(jobs_dir, {job_extension : self.process_local_json_workflow, template_extension : self.process_cdm_template_workflow})
        # Get asset info
        self.assetInfo = {}
        try:
            self.assetInfo = self.product_specific.make_instrument_info()['assetInfo']
            Job.asset_type = self.assetInfo.get('assetType')
            Job.model = self.assetInfo.get('model')
        except:
            log.warning("JobManager could not retrieve asset info")

        JobUiView.product_specific = product_specific
        #Current state of the UI view grouping. Valid values are job_order, location, testtype
        self.available_uiview_group_states = ["job_order","location","testtype"]
        self.current_uiview_group_state = "job_order"

        #Attempt to load the last active job from the data store
        self.last_active_job = None
        job = self.data_store.read_active()
        if job:
            self.job = Job(job)
            self.active_state = True
        else:
            self.active_state = False
            
    def get_job(self, workorder_id):
        """
        Gets the content of a specific Work Order (aka Job)

        Args: workorder_id of job to get
            
        Returns: Job CDM
            
        """
        return self.data_store.read_job_by_work_order_id(workorder_id)

    def get_job_clean_view(self, workorder_id=''):
        """
        Gets the content of a specific Work Order (aka Job). Removes any added data. 

        Args: workorder_id of job to get. Empty to get current job.
            
        Returns: Job CDM
            
        """
        job = None
        if workorder_id:
            job = self.data_store.read_job_by_work_order_id(workorder_id)
        else:
            job = copy.deepcopy(self.job.cdm)
        if job:
            self.clean_job(job)
        return job

    def clean_job(self, cdm_job):
        """
        Remove an IJM added data to get a "clean" southbound view of the job
            
        """
        #Remove all added tests
        tests = cdm_job.get('tests', [])
        tests = [t for t in tests if not test_is_additional(t)]
        cdm_job['tests'] = tests
        # remove all test results
        for test in tests:
            if test.get('results'):
                del test['results']
            for testLoc in test.get('testLocations', []):
                if testLoc.get('results'):
                    del testLoc['results']

    def get_uiview_group_state(self):
        return self.current_uiview_group_state

    def set_uiview_group_state(self, groupstate):
        rv = False
        if groupstate in self.available_uiview_group_states:
            self.current_uiview_group_state = groupstate
            rv = True
        return rv

    def get_job_ui_view(self, workorder_id=None, grouping=None):
        """
        Gets the content of a specific Work Order (aka Job)

        Args: workorder_id of job to get
              grouping: Indicates if the view should be grouped/sorted in particular manner. 
                     Default is none which implies job_order. 
                     location grouping returns a UI job view with tests sorted by location
                     testtype grouping returns a UI job view with tests sorted by test types
                     currentstate uses the current state set by set_uiview_group_state
            
        Returns: Job CDM
            
        """
        log.debug("get_job_ui_view: workorder_id:{} grouping:{}".format(workorder_id, grouping))
        rv = None
        job_object = None
        if workorder_id:
            job_object = self.get_job_object(workorder_id)
        else:
            job_object = self.job
        if job_object:
            if grouping == "currentstate":
                grouping = self.current_uiview_group_state
            if grouping == "location":
                rv = job_object.job_ui_view.get_ui_view_sorted_by_location()
            elif grouping == "testtype":
                rv = job_object.job_ui_view.get_ui_view_sorted_by_testtype()                
            else:
                rv = job_object.job_ui_view.cdm
        return rv

    def get_template(self, typename):
        """
        Gets the content of a specific template identified by typename

        Args: typename of template to get
            
        Returns: Template CDM
            
        """
        template = None
        return self.data_store.read_template_by_typename(typename)


    def get_job_object(self, workorder_id):
        """
        Gets the content of a specific Work Order (aka Job)

        Args: workorder_id of job to get
            
        Returns: A job object
            
        """
        job = None
        job_cdm = self.data_store.read_job_by_work_order_id(workorder_id)
        if job_cdm:
            job = Job(job_cdm)
        return job

    def get_active_job_object(self):
        '''
        Returns the active job

        Returns:
            job (Job): Returns Job object
                       None if no active job
        '''
        if self.is_active() and self.job:
            return self.job
        else:
            return None

    def get_active_job(self):
        '''
        Returns the active job

        Returns:
            job (Job): Returns Job cdm
                       None if no active job
        '''
        if self.is_active() and self.job:
            return self.job.cdm_job
        else:
            return None

    def activate(self):
        '''
        Activate the job manager if there is a current job

        Args:
            True to activate 
                0 to set all jobs inactive
        Returns: True on success, False otherwise
        '''
        if not self.active_state and self.last_active_job:# and self.job and self.job.workOrderId():
            self.set_active_job(self.last_active_job)
        return self.active_state

    def deactivate(self):
        '''
        Deactivate the job manager but leaves the current job in place

        Args:
            True to activate 
                0 to set all jobs inactive
        '''
        self.active_state = False
        if self.job:
            self.last_active_job = self.job.workOrderId()
        else:
            self.last_active_job = None
        self.job = None
        self.data_store.deactivate_job()
        self.increment_change_count()
        try:
            self.product_specific.update_loaded_job_side_effect(0, '')
        except:
            pass

    def is_active(self):
        return self.active_state

    def set_active_job(self, workorder_id):
        """
        Sets the current work order (Job) on the instrument by the work order ID and also activates job manager
        
        Args: work order ID of job to active
            
        Returns: The CDM job if found, None if not found
            
        """
        job = self.data_store.set_active(workorder_id)
        #If switching jobs reset the test data index
        self.set_active_test_plan_index(-1)
        if job:
            self.job = Job(job)
            self.active_state = True
            self.increment_change_count()
            try:
                self.product_specific.update_loaded_job_side_effect(1, self.job.status_ratio())
            except:
                pass
        else:
            self.deactivate()            
        return job


    def delete_job(self, workorder_id):
        #Make sure the given job isn't active
        if self.job and self.job.workOrderId() == workorder_id:
            self.deactivate()
        if not self.data_store.delete_job_by_work_order_id(workorder_id):
            raise FileNotFoundError
        self.delete_job_artifacts(workorder_id)
        self.increment_change_count()
        return True

    def delete_all_jobs(self):
        #Deactive the active job
        self.deactivate()
        for j in self.data_store.get_jobs_info():
            self.data_store.delete_job_by_work_order_id(j['workflow']['workOrderId'])
        self.delete_all_job_artifacts()
        self.increment_change_count()
        return True
    
    def delete_job_artifacts(self, workorder_id):
        artifacts_dir = os.path.join(self.job_artifacts_dir, workorder_id)
        if os.path.isdir(artifacts_dir):
            shutil.rmtree(artifacts_dir)
    
    def delete_all_job_artifacts(self):
        if os.path.isdir(self.job_artifacts_dir):
            shutil.rmtree(self.job_artifacts_dir)
            os.makedirs(self.job_artifacts_dir)

    def delete_template(self, typename):
        if not self.data_store.delete_template_by_typename(typename):
            raise FileNotFoundError
        #Inform StrataSync the template has been deleted
        self.sync_templates()
        self.increment_change_count()
        return True

    def delete_all_templates(self):
        for t in self.data_store.get_templates_info():
            self.data_store.delete_template_by_typename(t['workflow']['typeName'])
        #Inform StrataSync the template has been deleted
        self.sync_templates()
        self.increment_change_count()
        return True

    def process_cdm_json_workflow(self, cdm_json):
        cdm_workflow = cdm_schemas.CdmWorkflow().loads(cdm_json).data
        if cdm_workflow:
            self.process_cdm_workflow_tpa(cdm_workflow, False)
        else:
            log.error("failed to validate incoming workflow: ", cdm_json)

    def process_vmt_cdm_workflow(self, cdm_workflow):
        self.process_cdm_workflow_tpa(cdm_workflow, True)

    def process_vmt_cdm_template_workflow(self, cdm_template_workflow):
        #First do the normal processing of the work
        self.process_cdm_template_workflow(cdm_template_workflow)

    def process_cdm_workflow_tpa(self, cdm_workflow, vmt=False):
        """Function to parse the contents of a CDM job from StrataSync or VMT and save
        the individual job as seperate files

        args:
            cdm_json (str): the json contents of the CDM file
        """        
        try:
            #Remove any jobs that are externally managed that are no longer part of the incoming workflow
            all_job_info = self.get_all_job_info()
            existing_job_woids = [j['workflow']['workOrderId'] for j in all_job_info]
            incoming_job_ids = [j['workflow']['workOrderId'] for j in cdm_workflow.get('cdm', [])]
            woids_to_delete = None
            if vmt:
                #For mobile jobs delete jobs that are not in the incoming list and are not archived. Since archived jobs aren't in the
                #vmt workflow list, don't delete them. For VMT, typically jobs (included archived jobs) will be removed in process_vmt_workorder_list. 
                non_archived_job_list = list(filter(lambda j: not j.get('ijm_archived', False), all_job_info))
                non_archived_workorder_ids = [x['workflow']['workOrderId'] for x in non_archived_job_list]
                print("non_archived_workorder_ids: ", non_archived_workorder_ids)
                woids_to_delete = list(set(non_archived_workorder_ids) - set(incoming_job_ids))
            else:
                #When processing workflow.json, delete jobs that are externally managed and no longer are in the incoming workflow list
                existing_external_managed_job_woids = get_externally_managed_job_workorder_ids(all_job_info)
                woids_to_delete = list(set(existing_external_managed_job_woids) - set(incoming_job_ids))

            print("workorder_ids_to_delete: ", woids_to_delete)
            for w in woids_to_delete:
                self.delete_job(w)

            for woid in existing_job_woids:
                internal_job = self.get_job(woid)
                incoming_job = get_job_from_workflow(cdm_workflow, woid)
                if internal_job and incoming_job:        
                    if merge_jobs(internal_job, incoming_job):
                        self.data_store.write_job(internal_job, True)

            woids_to_add = list(set(incoming_job_ids) - set(existing_job_woids))
            jobs_to_add = [j for j in cdm_workflow['cdm'] if j['workflow']['workOrderId'] in woids_to_add]
            for j in jobs_to_add:
                if "cdmVersion" not in j: 
                    j["cdmVersion"] = cdm_workflow['cdmVersion']
                #Since we received the job externally we don't need to send a job instance file when saving job
                self.save_southbound_job(j, False)
            self.increment_change_count()
        except:
            print(traceback.format_exc())
            log.debug(traceback.format_exc())

    def process_local_json_workflow(self, cdm_workflow_json):
        cdm_workflow = cdm_schemas.CdmWorkflow().loads(cdm_workflow_json).data
        if cdm_workflow:
            try:
                for cdm_job in cdm_workflow['cdm']:
                    if "cdmVersion" not in cdm_job: 
                        cdm_job["cdmVersion"] = cdm_workflow['cdmVersion']
                    if cdm_job['workflow'].get('workOrderId'):                        
                        self.check_and_add_techid_to_job(cdm_job)
                        set_job_modified_to_now(cdm_job)
                        cdm_job = add_test_plan_indexes_to_job(cdm_job)                        
                        #Mark this job as locally managed. Once StrataSync has processed our job instance file and it starts showing up in the SB workflow
                        #then we'll set it to externally managed.
                        set_job_to_locally_managed(cdm_job)
                        self.add_asset_info_to_job(cdm_job)
                        self.save_southbound_job(cdm_job)
                    else:
                        log.error("process_local_json_workflow missing workOrderId")
            except:
                log.error("process_local_json_workflow invalid job")
        else:
            log.error("process_local_json_workflow failed to validate incoming workflow: ", cdm_workflow_json)
        self.increment_change_count()

    def process_vmt_workorder_list(self, workorder_list):
        """Function to delete and archive work orders from vmt

        args:
            workorder_list (CdmWorkOrderList)
        """        
        try:
            all_job_info = self.get_all_job_info()
            existing_job_woids = [j['workflow']['workOrderId'] for j in all_job_info]
            incoming_active_job_ids = [j['workOrderId'] for j in workorder_list.get('workOrderInfo', [])]
            incoming_archive_job_ids = workorder_list.get('archivedIds', [])
            all_incoming_job_ids = incoming_active_job_ids + incoming_archive_job_ids

            #Delete jobs no longer are in the incoming workflow list
            woids_to_delete = list(set(existing_job_woids) - set(all_incoming_job_ids))
            log.debug("workorder_ids_to_delete: {}".format(woids_to_delete))
            for w in woids_to_delete:
                self.delete_job(w)

            #Archive jobs that VMT has marked to archive
            for w in incoming_archive_job_ids:
                job = self.get_job(w)
                if job and not is_job_archived(job):                    
                    self.archive_job(w)

            #Restore jobs that VMT has marked as not archived
            for w in incoming_active_job_ids:
                job = self.get_job(w)
                if job and is_job_archived(job):
                    self.restore_job(w)

        except:
            print(traceback.format_exc())
            log.debug(traceback.format_exc())

    def process_cdm_template_workflow(self, cdm_template_workflow):
        """Function to parse the contents of a CDM template workflow

        args:
            cdm_json (str): the json contents of the CDM file
        """
        rv = True
        updated_templates = False
        #If we are given a json string, convert it to a CdmWorkflow dictionary.
        if isinstance(cdm_template_workflow, str):
            cdm_template_workflow = cdm_schemas.CdmWorkflow().loads(cdm_template_workflow).data
        if not cdm_template_workflow or 'cdm' not in cdm_template_workflow:
            raise ValueError
        #Remove any jobs that are externally managed that are no longer part of the incoming workflow
        existing_template_info = self.get_all_template_info()
        existing_template_names = [t['workflow']['typeName'] for t in existing_template_info]
        incoming_template_names = [t['workflow']['typeName'] for t in cdm_template_workflow.get('cdm', [])]

        #Delete templates that are no longer in the deployment
        templates_to_delete = list(set(existing_template_names) - set(incoming_template_names))
        for t in templates_to_delete:
            self.data_store.delete_template_by_typename(t)
            updated_templates = True

        for incoming_template in cdm_template_workflow.get('cdm', []):
            #Make sure this is a template
            if incoming_template['workflow'].get('template'):
                if 'cdmVersion' not in incoming_template:
                    incoming_template['cdmVersion'] = cdm_template_workflow['cdmVersion']

                incoming_template_name = incoming_template['workflow'].get('typeName', '')
                if incoming_template_name in existing_template_names:
                    #See if we need to overwrite our existing template
                    existing_template = get_template_from_template_array(existing_template_info, incoming_template_name)
                    existing_template_workflow = None
                    if existing_template:
                        existing_template_workflow = existing_template.get('workflow')
                    if incoming_template['workflow'] != existing_template_workflow:
                        write_status = self.data_store.write_template(incoming_template, True)
                        rv = rv & write_status
                        updated_templates = True
                else:
                    #New template
                    write_status = self.data_store.write_template(incoming_template, True)
                    rv = rv & write_status
                    updated_templates = True

        if updated_templates:
            self.sync_templates()
            self.increment_change_count()
        return rv

    def save_southbound_job(self, cdm_job, send_job_instance=True):
        """Function to push a cdm job into the datastore. 
        args:
            cdm_job (CdmJob): the job to save
            send_job_instance(bool): indicates whether we should send a job instance file. Should be 
                                     True when creating a job on instrument, False if receiving the job from VMT, SS
        returns:
            true on success, false on failure
        """
        rv = False
        if 'jobModifiedOn' not in cdm_job:
            set_job_modified_to_now(cdm_job)
        if 'date' not in cdm_job['workflow']:
            cdm_job['workflow']['date'] = get_local_timestamp_8601()
        self.check_and_add_techid_to_job(cdm_job)
            
        if not cdm_job['workflow'].get("template"):
            #If this job doesn't have a workflowID and it doesn't already exist in our database 
            if cdm_job['workflow'].get("workflowId") == None and \
            not self.data_store.read_job_by_work_order_id(cdm_job['workflow'].get("workOrderId")) and \
            not job_has_test_plan_index(cdm_job):                
                log.debug("New job detected wihout workflowId or testPlanIndex {}".format(cdm_job['workflow'].get("workOrderId")))
                #If the workflow doesn't have a workflowId, this job has not been issued through StrataSync
                # Add test plan indexes and sync to StrataSync
                cdm_job = add_test_plan_indexes_to_job(cdm_job)
                #Since we modified the job we sync the job to StrataSync regardless of whether the input arg indicated to
                send_job_instance = True                
            if self.data_store.write_job(cdm_job, False):
                rv = True
            if rv and send_job_instance:
                self.sync_job_instance(cdm_job)
        else:
            log.error("save_southbound_job found template")
        return rv 

    def create_job(self, create_job_schema):
        """Function to create a new job
        args:
            create_job_schema (CdmJobWithCsvImport): create a job
        Returns: The new CDM job or None if something went wrong
        """
        rv = None
        if not create_job_schema.get('workOrderId'):
            log.error("create_job received empty workOrderId")
            return None

        if not create_job_schema.get('templateName'):
            create_job_schema['templateName'] = default_template_typename

        template = self.get_template(create_job_schema['templateName'])            
        if not template:
            log.error("create_job invalid template: {}".format(create_job_schema['templateName']))
            return None

        template['workflow']['workOrderId'] = create_job_schema['workOrderId']
        if not template['workflow'].get("techInfo"):
            template['workflow']["techInfo"] = {}

        techId = create_job_schema.get('techId', '')
        if not techId:
            try:
                techId = self.product_specific.get_instrument_login_config()['techId']
            except:
                log.info("create_job: failed to get techId")        
        template["workflow"]["techInfo"]["techId"] = techId
        
        cdm_job = convert_template_to_job(template)
        if cdm_job:
            if 'jobModifiedOn' not in cdm_job:
                set_job_modified_to_now(cdm_job)
            cdm_job = add_test_plan_indexes_to_job(cdm_job)
            csv_location_import_filepath = create_job_schema.get('csvImportFilepath')
            if csv_location_import_filepath:
                if not import_csv_to_job(csv_location_import_filepath, cdm_job, self.assetInfo.get('assetType', ''), self.assetInfo.get('model', '')):
                    log.info("create_job failed to import csv")
                    cdm_job = None
        if cdm_job:                  
            #Mark this job as locally managed. Once StrataSync has processed our job instance file and it starts showing up in the SB workflow
            #then we'll set it to externally managed.
            set_job_to_locally_managed(cdm_job)
            self.add_asset_info_to_job(cdm_job)
            if self.save_southbound_job(cdm_job):
                rv = cdm_job
                self.increment_change_count()
        return rv


    def update_job(self, cdm_job):
        """Function to update an existing cdm job in the database.
        Called by UI and might be refactored in 2.0 and combined with save_southbound_job
        args:
            cdm_job (CdmJob): the job to save
        """
        if not cdm_job:
            return False
        workorder_id = cdm_job['workflow']['workOrderId']
        if not workorder_id:
            log.info("update_job: empty workOrderId")
            return False

        if 'jobModifiedOn' not in cdm_job:
            set_job_modified_to_now(cdm_job)
        if 'date' not in cdm_job['workflow']:
            cdm_job['workflow']['date'] = get_local_timestamp_8601()
        self.check_and_add_techid_to_job(cdm_job)            

        if not self.get_job(workorder_id):            
            #New job. Need to write job instance file
            self.sync_job_instance(cdm_job)
            set_job_to_locally_managed(cdm_job)

        self.data_store.write_job(cdm_job)
        if self.job and self.job.workOrderId() == workorder_id:
            #If the incoming job is the current/active job, then we need to update our self.job.            
            self.job = Job(cdm_job)            
        self.increment_change_count()
        return True

    def get_all_job_info(self):
        """Function to get all job info (workflow block plus jobModifiedTime)

        """

        return self.data_store.get_jobs_info()

    def get_all_template_info(self):
        """Function to get all template info (workflow block plus jobModifiedTime)
        """

        return self.data_store.get_templates_info()


    def get_templates(self):
        return self.data_store.get_templates()

    #TODO Needs unit test
    def get_job_status(self):
        """
        Method to return the job status
        """
        if self.job:
            return self.job.status_ratio()
        else:
            return ""

    def archive_job(self, workorder_id):
        #Make sure the given job isn't active
        if self.job and self.job.workOrderId() == workorder_id:
            self.deactivate()

        cdm_job = self.data_store.read_job_by_work_order_id(workorder_id)
        if not cdm_job:
            return False
        if not 'configuration' in cdm_job:
            cdm_job['configuration'] = {}

        cdm_job['configuration']['ijm_archived'] = True
        if not self.data_store.write_job(cdm_job):
            return False
        self.increment_change_count()
        return True

    def restore_job(self, workorder_id):
        cdm_job = self.data_store.read_job_by_work_order_id(workorder_id)
        if not cdm_job or not 'configuration' in cdm_job or not 'ijm_archived' in cdm_job['configuration']:
            return False

        del cdm_job['configuration']['ijm_archived']
        if not self.data_store.write_job(cdm_job):
            return False
        self.increment_change_count()
        return True

    def add_test_location(self, cdm_test, index):

        if not cdm_test.get('type'):
            log.debug('job_manager.add_test_location: no type in cdm_test')
            return None

        if not cdm_test.get('testLocations'):
            log.debug('no test location present')
            return None

        cdm_job, location_index = self.job.add_test_location(cdm_test, index)
        self.data_store.write_job(cdm_job)
        self.increment_change_count()
        return cdm_job

    def close_open_ended_test(self, index):
        if index != -1 and len(self.job.cdm_job['tests']) > index:
            test = self.job.cdm_job['tests'][index]
        else:
            return False
        
        test['openEndedStatus'] = 'closed'
        set_job_modified_to_now(self.job.cdm_job)
        if not self.data_store.write_job(self.job.cdm_job):
            return False
        self.sync_job_instance(self.job.cdm_job)
        self.increment_change_count()
        return True
    
    def add_test(self, cdm_test):

        if not cdm_test.get('type'):
            log.debug('job_manager.add_test: no type in cdm_test')
            return None

        cdm_job = self.job.add_test(cdm_test)
        self.data_store.write_job(cdm_job)
        self.increment_change_count()
        return cdm_job

    def close_open_ended_job(self, close_oe_tests=True):
        self.job.cdm_job['workflow']['openEndedStatus'] = 'closed'

        if close_oe_tests:
            for test in self.job.cdm.get('tests', []):
                if test.get('openEndedStatus', '') == 'open':
                    test['openEndedStatus'] = 'closed'

        set_job_modified_to_now(self.job.cdm_job)
        if not self.data_store.write_job(self.job.cdm_job):
            return False
        self.sync_job_instance(self.job.cdm_job)            
        self.increment_change_count()
        return True
    
    def add_test_data(self, cdm_test_data, legacy = False, report_filepaths = [], last_test_data = True, saveResults = False, test_array_index = None):
        """
        Method called when processing new test data

        Args:
            cdm_test_data (dict): cdm test data (CdmTestSchema)
            legacy(bool): indicates if the old API structures were used
            report_filepaths(list(str)): any report filepaths that should be saved with data 
            last_test_data(bool): indicates if this is the last of series of calls made by test to register data. True initiates the side effect.
        """
        if not cdm_test_data.get('type') or not cdm_test_data.get('results'):
            log.debug('job_manager.add_test_data: no type or results in cdm_test_data')
            return None

        if not 'testTime' in cdm_test_data['results'] or not cdm_test_data['results']['testTime']:
            cdm_test_data['results']['testTime'] = get_local_timestamp_8601()
        
        if not test_array_index:
            test_array_index = self.get_active_test_plan_index()

        # Two indexes returned but for old instruments, there is only 1 location so only the test plan index matters
        cdm_test_data_copy = copy.deepcopy(cdm_test_data)
        test_plan_index, test_location_index = self.job.add_test_data(cdm_test_data, self.product_specific, legacy, test_array_index, report_filepaths, saveResults)
        cdm_job = self.job.cdm        


        #Since NB CDM expects workflowId/testPlanIndex at the test level rather than the location level, 
        # copy the SB workflowId/testPlanIndex into test object if it isn't already there. 
        cdm_test_data_copy = self.populate_workflowid_andor_testplanindex(cdm_test_data_copy, test_plan_index, test_location_index)
        nb_cdm_job = {}
        nb_cdm_job["cdmVersion"] = cdm_job["cdmVersion"]
        nb_cdm_job["tests"] = [cdm_test_data_copy]
        nb_cdm_job["workflow"] = cdm_job["workflow"]
        self.add_asset_info_to_job(nb_cdm_job)
        jnb_cdm_job = cdm_schemas.addCheckSumToCdmJob(nb_cdm_job)

        self.data_store.write_job(cdm_job)
        if test_plan_index is not None:
            if last_test_data:
                try:
                    if cdm_test_data['results']['status'] == 'Skipped':
                        test_plan_index = -1
                except:
                    pass
                if self.product_specific:
                    combined_index = test_plan_index 
                    if test_plan_index != -1 and test_location_index != None and test_location_index != -1:
                        #Create a combined test and location index pair (e.g. 1,2)
                        combined_index = "{},{}".format(test_plan_index, test_location_index)
                    self.product_specific.update_test_data_side_effect(combined_index,self.job.status_ratio())
            else:
                JobManager.index_of_last_added_test_data = [test_plan_index, test_location_index]
                if cdm_test_data['results']['status'] == 'Skipped':
                    self.product_specific.update_test_data_side_effect(-1,self.job.status_ratio())
        

        self.increment_change_count()
        return jnb_cdm_job

    def set_all_test_data_added(self):
        """
        Method called when all test data has been added to invoke the delayed side efffect

        Args:
            None
        """
        if JobManager.index_of_last_added_test_data:
            combined_index = JobManager.index_of_last_added_test_data[0] 
            if JobManager.index_of_last_added_test_data[0] != -1 and JobManager.index_of_last_added_test_data[1] != None and JobManager.index_of_last_added_test_data[1] != -1:
                #Create a combined test and location index pair (e.g. 1,2)
                combined_index = "{},{}".format(JobManager.index_of_last_added_test_data[0] ,JobManager.index_of_last_added_test_data[1])
            self.product_specific.update_test_data_side_effect(combined_index,self.job.status_ratio())
        JobManager.index_of_last_added_test_data = []
        self.increment_change_count()

    #Test plan index is required for the side effect
    def set_active_test_plan_index(self, test_index, test_location_index=0):
        '''
        Method to set the index of the active job item within a test plan

        Args:
            active_job_item_index (int): the index of the actively launched job item in the active job

        '''
        rv = False
        index_list = []
        if test_index != -1:
            index_list = [test_index, test_location_index]
        if self.is_active() or test_index == -1:
            #Only write the index if the job manager is active or we are resetting the index
            self.data_store.set_active_job_item_index(index_list)
            rv = True
        return rv

    #Return a cdm object (no results) from the given indices
    def get_cdm_test_and_location_from_indices(self, test_index, location_index):
        rv_cdm_test = None
        try:
            if test_index >= 0:
                cdm_test = self.job.cdm['tests'][test_index]
                rv_cdm_test = copy.deepcopy(cdm_test)
                if 'results' in rv_cdm_test:
                    rv_cdm_test.pop('results')     
                if 'testLocations' in rv_cdm_test:                    
                    rv_cdm_test.pop('testLocations') 
                if location_index != None and location_index >= 0:
                    cdm_location = copy.deepcopy(cdm_test['testLocations'][location_index])
                    if 'results' in cdm_location:
                        cdm_location.pop('results')
                    rv_cdm_test['testLocations'] = [cdm_location]
                    #Since NB CDM expects workflowId/testPlanIndex at the test level rather than the location level, 
                    # copy the SB workflowId/testPlanIndex into test object. 
                    ids = self._workflowid_or_testplanindex(rv_cdm_test)
                    if ids["workflowId"]  != None:
                        rv_cdm_test["workflowId"] = ids["workflowId"]
                    if ids["testPlanIndex"] != None:
                        rv_cdm_test["testPlanIndex"] = ids["testPlanIndex"]
        except:
            pass        
        return rv_cdm_test
    
    #Return a cdm object (no results) from the given workflowId and/or testPlanIndex. 
    #Set workflowId or testPlanIndex to None if you don't want to search on that parameter 
    def get_cdm_test_and_location_from_workflowid_or_testplanindex(self, cdm_job, workflowid, test_plan_index):
        rv_cdm_test = None
        match = False
        test_array_index = 0
        test_location_array_index = 0

        #Iterate through each test until we find a matching index for workflowid or testPlanIndex
        for test in cdm_job.get('tests', []):
            test_locations = test.get('testLocations', None)
            
            test_location_array_index = -1
            if test_locations:

                test_location_array_index = 0
                for testLoc in test_locations:
                    wfi = testLoc.get('workflowId', None)
                    tpi = testLoc.get('testPlanIndex', None)
                    if (workflowid and wfi == workflowid) or (test_plan_index and tpi == test_plan_index):
                        match = True
                    if match:
                        rv_cdm_test = copy.deepcopy(test)
                        #Purge the results if we have any
                        rv_cdm_test.pop("results", None)
                        testLoc.pop("results", None)
                        #Only want to return one test location
                        rv_cdm_test.pop('testLocations')
                        rv_cdm_test['testLocations'] = [testLoc]                        
                        break
                    
                    test_location_array_index += 1
            else:
                #No locations in this test so check the test for match
                wfi = test.get('workflowId', None)
                tpi = test.get('testPlanIndex', None) 
                if (workflowid and wfi == workflowid) or (test_plan_index and tpi == test_plan_index):
                    match = True
                if match:
                    rv_cdm_test = copy.deepcopy(test)
                    #Purge the results if we have any
                    rv_cdm_test.pop("results", None)
                    break
            if match:
                break

            test_array_index += 1

        if match:
            return (rv_cdm_test, [test_array_index, test_location_array_index])
        else:
            return None

    #Return a cdm object (no results) from the given test index
    def get_cdm_test_with_locations(self, test_index):
        rv_cdm_test = None
        try:
            if test_index >= 0:
                cdm_test = self.job.cdm['tests'][test_index]
                rv_cdm_test = copy.deepcopy(cdm_test)
                if 'results' in rv_cdm_test:
                    rv_cdm_test.pop('results')     
                for cdm_location in rv_cdm_test.get('testLocations', []):
                        cdm_location.pop('results', None)
        except:
            pass        
        return rv_cdm_test

    
    #returns {"workflowId" : None/int, "testPlanIndex" : None/int}
    def get_matching_workflowid_and_testplanindex(self, cdm_test):      
        matching_cdm_test = None
        test_index, location_index = self.get_test_index_of_cdm_test(cdm_test)
        if test_index >= 0:
            matching_cdm_test = self.get_cdm_test_and_location_from_indices(test_index, location_index)
        rv = self._workflowid_or_testplanindex(matching_cdm_test)
        return rv

    #Get the workflowId and/or testPlanIndex from the given test
    #returns {"workflowId" : None/int, "testPlanIndex" : None/int}
    def _workflowid_or_testplanindex(self, cdm_test):
        rv = {"workflowId" : None, "testPlanIndex" : None}
        if cdm_test == None:
            return rv
        if 'testLocations' in cdm_test and cdm_test['testLocations']:
            if "workflowId" in cdm_test['testLocations'][0]:
                rv["workflowId"] = cdm_test['testLocations'][0]["workflowId"]
            if "testPlanIndex" in cdm_test['testLocations'][0]:
                rv["testPlanIndex"] = cdm_test['testLocations'][0]["testPlanIndex"]
        else:                
            if "workflowId" in cdm_test: 
                rv["workflowId"] = cdm_test["workflowId"]
            elif "testPlanIndex" in cdm_test:
                rv["testPlanIndex"] = cdm_test["testPlanIndex"]
        return rv

    def populate_workflowid_andor_testplanindex(self, cdm_test, test_index, location_index):
        '''
        Method to populate the given cdm_test object with the proper workflowId and/or testPlanIndex and openEndedStatus. This is used to generate
        a northbound CDM object. 
        Returns a CdmTest object for northbound CDM with workflowId or testPlanIndex and openEndedStatus populated. 
        '''
        try:
            matching_cdm_test = self.get_cdm_test_and_location_from_indices(test_index, location_index)

            if 'testLocations' in matching_cdm_test and matching_cdm_test['testLocations']:
                # We have a match and there are locations and the openEndedStatus is not open
                if 'required' in matching_cdm_test['testLocations'][0]:
                    cdm_test['required'] = matching_cdm_test['testLocations'][0]['required']
                elif 'required' in matching_cdm_test:
                    cdm_test['required'] = matching_cdm_test['required']
                    
                # For NB test we put workflow id/test plan index in the test level for consistency
                # NB test will only have 1 location
                found_id = False
                if "workflowId" in matching_cdm_test['testLocations'][0]:
                    cdm_test["workflowId"] = matching_cdm_test['testLocations'][0]["workflowId"]
                    found_id = True
                if "testPlanIndex" in matching_cdm_test['testLocations'][0]:
                    cdm_test["testPlanIndex"] = matching_cdm_test['testLocations'][0]["testPlanIndex"]
                    found_id = True
                if not found_id and matching_cdm_test.get('openEndedStatus', '').lower() == 'open':
                    #For open ended tests, we inherit the workflowId/testPlanIndex at the test level
                    if "workflowId" in matching_cdm_test:
                        cdm_test["workflowId"] = matching_cdm_test["workflowId"]
                    if "testPlanIndex" in matching_cdm_test:
                        cdm_test["testPlanIndex"] = matching_cdm_test["testPlanIndex"]
            else:  
                #We have a match but no locations or there are locations but it is an open ended test
                if 'required' in matching_cdm_test:
                    cdm_test['required'] = matching_cdm_test['required']         
                               
                if "workflowId" in matching_cdm_test:
                    cdm_test["workflowId"] = matching_cdm_test["workflowId"]
                if "testPlanIndex" in matching_cdm_test:
                    cdm_test["testPlanIndex"] = matching_cdm_test["testPlanIndex"]
            #NB results should always have the workflowId or testPlanIndex at the test level not the location level
            if 'testLocations' in cdm_test:
                cdm_test['testLocations'][0].pop('workflowId', None)
                cdm_test['testLocations'][0].pop('testPlanIndex', None)
            if "openEndedStatus" in matching_cdm_test:
                #If openEndedStatus is in the matching test, always report it NB                      
                cdm_test["openEndedStatus"] = matching_cdm_test["openEndedStatus"] 

        except:
            log.info("populate_workflowid_andor_testplanindex threw exception")
        return cdm_test

    #Returns metadata for the given cdm test. cdm_test should be a northbound cdm_test type with the proper test configuraiton and populated results
    def get_metadata_for_cdmtest(self, cdm_test):
        rv = {}
        matching_cdm_test = None
        test_index, location_index = self.get_test_index_of_cdm_test(cdm_test)        
        if test_index >= 0:
            matching_cdm_test = self.get_cdm_test_and_location_from_indices(test_index, location_index)

        #workOrderId
        rv["Workorder"] = self.job.cdm_job['workflow']['workOrderId']

        #workflowId / testPlanIndex
        if self.job.cdm_job['workflow'].get('workflowId'):
            #Default workflowId to the job level. 
            rv["WorkflowId"] = self.job.cdm_job['workflow']['workflowId']
        if matching_cdm_test:
            ids = self._workflowid_or_testplanindex(matching_cdm_test)
            if ids["workflowId"] != None:
                rv["WorkflowId"] = ids["workflowId"]
            if ids["testPlanIndex"] != None:
                rv["TestPlanIndex"] = ids["testPlanIndex"]

        #Customer name
        try:
            rv["CustomerName"] = self.job.cdm_job["workflow"]["customerInfo"]["company"]
        except:
            pass
        #Tech id
        try:
            rv["TechnicianId"] = self.job.cdm_job["workflow"]["techInfo"]["techId"]
        except:
            pass
        
        #Test type       
        try:
            rv["Type"] = cdm_test["type"]
        except:
            pass

        #TestLocation
        #First look for location in the northbound test label.
        #cdm_test should have a single test and at most one location
        if matching_cdm_test and matching_cdm_test.get('testLocations'):
            rv["TestLocation"] = matching_cdm_test['testLocations'][0]['label']
        else:
            try:
                rv["TestLocation"] = self.job.cdm_job["workflow"]["jobManagerAttributes"]["testLocation"]["value"]
            except:
                pass

        #Test status/verdict
        try:
            rv["Verdict"] = cdm_test["results"]["status"]
        except:
            pass

        return rv

    #Returns the launched CDM test and location block
    def get_active_test_and_location(self):
        '''
        Method to get the CDM test/location object from the active/launched test plan indices
        Returns a CdmTest object with at most one location 
        '''
        rv = None
        indices = self.get_active_test_plan_index()
        if indices:
            rv = self.get_cdm_test_and_location_from_indices(indices[0], indices[1])
        return rv

    def get_active_test_plan_index(self):
        '''
        Method to get the indices of the launched job test/location within a test plan. This corresponds to the launched test/location
        of the launched test. 
        Returns a list pair of [testIndex, locationIndex] that corresponds to the test and location within the active job.  
        Note that this is not the same as the CDM's testPlanIndex
        '''
        rv = []
        if self.is_active():
            index_list = self.data_store.read_active_job_item_index()['active_job_item_index']
            if isinstance(index_list, list):
                rv = index_list
            else:
                #Handle backwards compatiblity where the index is a single digit
                if index_list == -1 or index_list == "-1":
                    #No active index case
                    rv = []
                elif index_list:
                    #If the index is not empty, convert the single digit to a list with location 0
                    rv = [int(index_list), 0]
        return rv

    def get_active_test_plan_uiindex(self):
        rv = ""
        tpi = self.get_active_test_plan_index()
        print("get_active_test_plan_uiindex tpi: ", tpi)
        if tpi:
            rv = get_uiindex_from_test_and_location_index(tpi[0], tpi[1])
        return rv

    # This function is for old instruments to get test index based on ref info and type
    def get_test_plan_index_of_ref(self, test_type, ref_info):
        index = 0
        job = self.job.cdm
        for test in job['tests']:
            if ref_info_matches(test, test_type, ref_info):
                return index
            index += 1
    
    def get_test_index_of_cdm_test(self, cdm_test):
        '''
        Method to get the indices of the job item within a test plan. This corresponds to the launched test/location
        of the launched test. 
        Returns a tuple pair of [testIndex, locationIndex] that corresponds to the test and location within the active job.  
        Note that this is not the same as the CDM's testPlanIndex
        '''
        rv = (-1, -1)
        try:
            test_indices = self.job.find_test_indices(cdm_test, self.product_specific, self.get_active_test_plan_index())
            if test_indices:
                rv = test_indices
        except:
            print("find_test_indices exception")
            print(traceback.format_exc())
            pass
        return rv

        

    def clear_test_data(self, test_index, test_location_index=-1):
        """
        Method called when redoing a planned test

        Args:
            test_plan_index (int): index of test plan whose data
            to be cleared and status to be reset to "To Do"
        """
        rv = False
        try:
            test = self.job.cdm_job['tests'][test_index]
            if test_location_index != -1 and 'testLocations' in test and len(test['testLocations']) > test_location_index and 'results' in test['testLocations'][test_location_index]:
                del test['testLocations'][test_location_index]['results']
            elif 'results' in  test:
                del test['results']

            #Decrement the job status ratio count
            self.product_specific.update_test_data_side_effect(-1,self.job.status_ratio())
            if not self.data_store.write_job(self.job.cdm_job):
                return False
            self.increment_change_count()
            rv = True
        except:
            log.warn(traceback.format_exc())
        return rv


    def load_job_from_file(self, file_path):
        """
        Method to load a job from a file containing job data in json format

        Args:
            file_path (str): the file_path of the file to load
        """
        cdm_job = None
        with open(file_path) as job_file:
            job_string = job_file.read()

        if job_string:
            cdm_job = marshmallow_data_compat(cdm_schemas.CdmJob().loads(job_string))
            if cdm_job:
                workOrderId = cdm_job["workflow"]["workOrderId"]
                if self.get_job(workOrderId):                    
                    #Job already exists in database. We'll load it as our active job. 
                    log.debug('## load_job_from_file already exists in database')
                    self.set_active_job(workOrderId)
                else:
                    #Job doesn't already exist. First we'll load it into our database and then we'll activate it
                    set_job_to_locally_managed(cdm_job)
                    self.save_southbound_job(cdm_job)
                    self.set_active_job(workOrderId)
                    log.debug('## load_job_from_file new job')
        else:
            log.debug('## load_job_from_file empty job')
            #If this is an empty file, it may have already been loaded into the database but we put the old file there for 
            #backwards compatiblity with the job manager file picker
            workOrderId = self.get_workorderId_from_filepath(file_path)
            if workOrderId:
                cdm_job = self.set_active_job(workOrderId)
                log.debug('## load_job_from_file empty job loaded from database')
        return cdm_job

    def get_workorderId_from_filepath(self, file_path):
        workOrderId = None
        job_ext = ".job.json"
        try:
            file_name = os.path.basename(file_path)
            job_ext = self.product_specific.get_job_extension()
        except:
            pass
        try:
            workOrderId = file_name.split(job_ext)[0]
        except:
            pass
        return workOrderId

    def increment_change_count(self):
        self.change_count += 1

    def get_change_count(self):
        return self.change_count

    #When templates are received/updated, send them to StrataSync. According to Lalit, wrap all the templates into a workflow object
    #and send them all to StrataSync. StrataSync will parse and figure out if templates need to be updated on the instrument. 
    def sync_templates(self):
        #cdm_sync registers USC with the path {jobs-dir}/cdm-northbound with template extension .tpl.json
        try:
            templates = self.get_templates()
            #templates is an array of templates in northbound format. We need to wrap these in a workflow southbound style            
            ss_cdm_templates = {"cdm" : templates}
            ss_cdm_templates["cdmVersion"] = cdm_schemas.CURRENT_CDM_VERSION
            filepath = os.path.join(self.northbound_jobs_cdm_dir, 'templates.tpl.json')
            with open(filepath, "w") as f:
                json.dump(ss_cdm_templates, f)
        except:
            log.warn("job_manager.sync_templates failed")
            log.debug(traceback.format_exc())

    def add_asset_info_to_job(self, cdm_job, use_cached=False):
        assetInfo = {}
        if use_cached or getattr(self.product_specific, 'use_cached_instrument_info', False):
            assetInfo = self.assetInfo
        else:
            try:            
                insinfo = self.product_specific.make_instrument_info()
                assetInfo = insinfo['assetInfo']
                login_config = self.product_specific.get_instrument_login_config()
                assetInfo['techId'] = login_config['techId']
            except:
                log.debug("add_asset_info_to_job failed")
                log.debug(traceback.format_exc())
        #assetinfo can have dbus strings, so we need to convert it to json and back to a dictionary to get rid of dbus types.
        assetInfo = json.dumps(assetInfo)
        assetInfo = json.loads(assetInfo)
        cdm_job["assetInfo"] = assetInfo

    def sync_job_instance(self, cdm_job):
        if 'jobModifiedOn' not in cdm_job:
            set_job_modified_to_now(cdm_job)
        if 'date' not in cdm_job['workflow']:
            cdm_job['workflow']['date'] = get_local_timestamp_8601()
        #StrataSync requires typeName to be present in job instance even if empty
        if 'typeName' not in cdm_job['workflow']:
            cdm_job['workflow']['typeName'] = ''

        #cdm_sync registers USC with the path /tmp/cdm-northbound with job instance extension .job.json
        ss_cdm_job = None
        cdm_job_copy = copy.deepcopy(cdm_job)
        self.clean_job(cdm_job_copy)
        try:
            #Convert to southbound type CDM as expected by StrataSync
            ss_cdm_job = {"cdm" : [cdm_job_copy]}
            ss_cdm_job["cdmVersion"] = cdm_job_copy.get("cdmVersion", "")
            work_order_id = cdm_job_copy['workflow']['workOrderId']
            filepath = os.path.join(self.northbound_jobs_cdm_dir, work_order_id + '.job.json')
            with open(filepath, "w") as f:
                json.dump(ss_cdm_job, f)
        except:
            log.warn("job_manager.sync_job_instance failed: ", ss_cdm_job)
            log.debug(traceback.format_exc())

    #Get the test and location array indexes of full job from ijm_uiindex
    #Returns tuple of test, location index (None for location indicates a test index)
    def get_test_location_index_from_uiindex(self, uiindex):
        return get_test_location_index_from_uiindex(uiindex)

    #Patch in the workflow block
    def patch_workflow(self, work_order_id, cdm_workflow_patch):
        incoming = {"workflow" : cdm_workflow_patch}
        rv = False
        cdm_job = None
        if work_order_id:
            cdm_job = self.get_job(work_order_id)
        elif self.is_active() and self.job:
            cdm_job = self.job.cdm
        if cdm_job:            
            merge_dict(cdm_job, incoming) 
            set_job_modified_to_now(cdm_job)   
            status = self.data_store.write_job(cdm_job, True)
            if status:
                rv = True
                self.sync_job_instance(cdm_job)
                self.increment_change_count()
        return rv

    #Patch a test
    def patch_test_attributes(self, ijm_uiindex, cdm_test_patch):
        rv = False
        cdm_job = None
        if self.is_active() and self.job:
            cdm_job = self.job.cdm
        if not cdm_job:
            log.debug("patch_test_attributes: no valid job")
            return False

        test_index, location_index = get_test_location_index_from_uiindex(ijm_uiindex)
        if test_index >= 0:
            tests = self.job.cdm_job.get('tests')
            if tests and test_index < len(tests):
                test = tests[test_index]
                merge_dict(test, cdm_test_patch) 
                set_job_modified_to_now(cdm_job)   
                status = self.data_store.write_job(cdm_job, True)
                if status:
                    rv = True
                    self.sync_job_instance(cdm_job)
                    self.increment_change_count()
            else:
                log.debug("patch_test_attributes: test_index invalid {}".format(test_index))
        else:
            log.debug("patch_test_attributes: test_index invalid {}".format(test_index))
        return rv

    def check_and_add_techid_to_job(self, cdm_job):
        try:
            if "techInfo" not in cdm_job["workflow"]:
                cdm_job["workflow"]["techInfo"] = {}

            if not cdm_job["workflow"]["techInfo"].get("techId"):
                techId = self.product_specific.get_instrument_login_config()['techId']
                cdm_job["workflow"]["techInfo"]["techId"] = techId
        except:
            pass

    def set_manual_test_response(self, ijm_uiindex, manual_step_schema):
        rv = False
        userResp, comm = "",""
        if manual_step_schema.get('userResponse'):
            userResp = manual_step_schema.get('userResponse')
        if manual_step_schema.get('comment'):
            comm = manual_step_schema.get('comment')
        log.debug("manual step received  userResponse:{} , comment: {}".format(userResp, comm))

        #Look up the cdm_test based on the UI index
        test_index, location_index = get_test_location_index_from_uiindex(ijm_uiindex)
        if test_index != None and test_index >=0:
            cdm_test = self.get_cdm_test_and_location_from_indices(test_index, location_index)
            if cdm_test and cdm_test['type'] == 'manual':
                #Append the results to the cdm_test
                cdm_test['results'] = {'status' : 'none', "comment" : comm, "data" : {"response" : userResp}}
                #Notify job manager of the added data and get back the NB CDM
                nb_cdm = self.add_test_data(cdm_test, saveResults=True, test_array_index=[test_index, location_index])
                #Write the NB CDM to a StrataSync registered path
                if self.job.workOrderId():
                    northbound_cdm_data_file = os.path.join(self.jobs_dir, "northbound_cdm/", self.job.workOrderId() + '_manual' + ijm_uiindex + '.cdm.json')
                    with open(northbound_cdm_data_file, "w") as f:
                        f.write(nb_cdm)
                    rv = True
            else:
                rv = False
        return rv

def ref_info_matches(planned_test, test_type, ref_info):
        """Method to determine if the passed in reference info and test type
        match the test type and reference info in the planned test

        The reference info dictionaries can can appear in any any order because
        they are sorted as part of the comparision

        Args:
            ref_info (dict): dictionary of the ref info

        Returns:
            bool: True if matches False otherwise
        """
        cdm_ref_info = {}
        ref_info_list = []
        
        if(planned_test['type'] != test_type):
            return False

        # in cdm ref info is a dict which we need to convert to dict so we can compare with legacy ref info
        if planned_test.get('testLocations'):
            if planned_test['testLocations'][0].get('attributes'):
                cdm_ref_info = planned_test['testLocations'][0]['attributes'].get('referenceInfo')

        if cdm_ref_info:
            for key, val in cdm_ref_info.items():
                ref_info_list.append({'key':key, 'value':val})
        
        
        our_reference_info = sorted(ref_info_list,
                                    key=lambda x: sorted(x.values()))

        incomming_reference_info = sorted(ref_info,
                                          key=lambda x: sorted(x.values()))

        return our_reference_info == incomming_reference_info



def add_test_plan_indexes_to_job(cdm_job):
    #Templates should have testPlanIndex populated already. However some legacy template may not have so we'll populate them if they don't already exist.
    #For each test, add a testPlanIndex in the test object if there are not locations
    #If there are locations, we only add testPlanIndex for the locations. 
    if 'tests' in cdm_job and not job_has_test_plan_index(cdm_job):
        for test in cdm_job['tests']:
            #Always add an index at the test level
            test['testPlanIndex'] = get_next_test_plan_index()
            test_locations = test.get('testLocations')
            if test_locations:
                for testLoc in test_locations:
                    testLoc['testPlanIndex'] = get_next_test_plan_index()
    return cdm_job
