'''Module containing a class for working with job information

Contains the Job class and associated helper functions
'''

import sys
import json
import logging
import copy
from functools import partial
from .job_uiview import JobUiView
from .job_status import job_status
from .job_status import job_status_ratio
from rest_api.utils.marshmallow_compat import marshmallow_data_compat
from .cdm import cdm_schemas
from rest_api.utils.marshmallow_compat import marshmallow_data_compat
from .helper import html_format_job_info, html_format_test_data, html_format_test_plan
from .helper import get_next_test_plan_index
log = logging.getLogger(__name__)

class Job(object): #pylint: disable=too-many-instance-attributes
    '''
    Class for holding a CDM job

    Args:
        cdm_job(cdm_schemas.CdmJob): an unmarshalled CdmJob

    '''
    #Add class variable for asset type and model number. These won't change over time and
    #we don't need to pass them in everytime
    asset_type = None
    model = None
    def __init__(self, cdm_job):
        self.cdm_job = cdm_job
        #Create a UI view of the job
        self.job_ui_view = JobUiView(cdm_job, self.asset_type, self.model)

    @property
    def status(self):
        return job_status(self.cdm_job, self.asset_type, self.model)
    
    def status_ratio(self):
        return job_status_ratio(self.cdm_job, self.asset_type, self.model)

    def find_test_indices(self, cdm_test_data, product_specific, test_launch_indices=None, legacy=False):
        '''
        Method to find the test/location entry in the CDM corresponding to the given cdm_test_data

        Args:
            cdm_test_data (dict): cdm of the test/location
            product_specific(object): product_specific used to determine if there are specific match function for certain test defintions
            legacy(bool): True indicates this was called from a client using the legacy API
            test_launch_indices(list(int) of length 1 or 2): [testIndex, locationIndex] of the test launch. This prioritizes two tests of the exact same configuration with the test index that was launched
        Returns:
            tuple (testIndex(int), locationIndex(int)) 
                None if the test_data does not match any entries in the test plan
        '''
        # finds the matching test location block (can be a dict or none)
        if not 'tests' in self.cdm_job:
            self.cdm_job['tests'] = []
        test = cdm_test_data
        test_plan_copy = remove_empty_objects_list(copy.deepcopy(self.cdm_job['tests']))
        test_data_copy = remove_empty_objects_dict(copy.deepcopy(test))

        # enumerate the test plan so that we know each test's original index after sorting
        test_plan_copy = list(enumerate(test_plan_copy))
        test_launch_index = -1
        if test_launch_indices and test_launch_indices[0] >= 0:
            test_launch_index = test_launch_indices[0]
        # sort the test plan by restriction so that most restricitve is matched first
        #If we have a valid active test plan index (i.e. the index of the launched test),
        #we want to use that as a parameter to our sorting algorithm so
        #bind that index into the sort_restriction algorithm prior to sorting            
        test_plan_copy.sort(key=partial(sort_restriction, test_launch_index), reverse=True)

        # try to use product specific test matching function, if not found then use generic one
        func = None
        if product_specific and test['type'] in product_specific.test_definitions:
            func = getattr(product_specific.test_definitions[test['type']], 'find_test_plan_index_generic', None)
        if not callable(func):
            func = self.find_test_plan_index_generic
        return func(test_plan_copy, test_data_copy, legacy)            


    def add_test_location(self, cdm_test, index):
        '''
        Method to add new test location to a test

        Args:
            cdm_test (dict): dictionary of the test with location to add
        '''
        test = None
        if index != -1 and len(self.cdm_job['tests']) > index:
            test = self.cdm_job['tests'][index]
        else:
            return (None, None)

        # Test needs to have workflow id or test plan index to add location to it
        if test.get('workflowId', None) == None and test.get('testPlanIndex', None) == None:
            return (None, None)
        
        open_ended_status = test.get('openEndedStatus')
        if not open_ended_status or open_ended_status.lower() == 'closed':
            return (None, None)
        
        if test.get('testLocations'):
            test_location = cdm_test['testLocations'][0]
            test_location['required'] = False
            test['testLocations'].append(test_location)
            location_index = len(test['testLocations']) - 1
        else:
            test['required'] = False
            test['testLocations'] = cdm_test['testLocations']
            location_index = 0
        
        self.job_ui_view.update(self.cdm_job)
        return (self.cdm_job, location_index)
    
    def add_test(self, cdm_test):
        '''
        Method to add new test to an open ended job

        Args:
            cdm_test (dict): dictionary of the test to add
        '''
        open_ended_status = self.cdm_job['workflow'].get('openEndedStatus')
        if not open_ended_status or open_ended_status.lower() == 'closed':
            return None
        
        if self.cdm_job.get('tests'):
            self.cdm_job['tests'].append(cdm_test)
        else:
            self.cdm_job['tests'] = cdm_test

        self.job_ui_view.update(self.cdm_job)
        return self.cdm_job

    
    def add_test_data(self, cdm_test_data, product_specific, legacy, test_launch_indices, report_filepaths = [], saveResults = False):
        '''
        Method to add new test data to a job

        Args:
            test_data (dict): dictionary of the test data to add
        '''
        if not 'results' in cdm_test_data:
            log.debug('job.add_test_data: no results in cdm_test_data')
            return (None, None)
        test_tuple = self.find_test_indices(cdm_test_data, product_specific, test_launch_indices, legacy)
        #We don't want to save all results in the job manager. We filter results for the things we care about (i.e. we filter ['results']['data'] only for ijm specific keys)
        jm_results = copy.deepcopy(cdm_test_data['results'])
        results_data = None
        if 'data' in jm_results:
            results_data = jm_results.pop('data')
        jm_results['data'] = {}
        if results_data:
            for k,v in results_data.items():
                if k.startswith('ijm') or saveResults==True:
                    jm_results['data'][k] = v

        #If there any reports associated with this test, we add them to an internal array in the test data
        if report_filepaths:
            jm_results['data']['ijm_filePath'] = report_filepaths

        if not test_tuple:
            # If a nb test does not match any tests in the sb job we delete workflowId and add a testPlanIndex
            #Added tests are always non-required
            cdm_test_data['required'] = False
            # Then the nb test data's test block is appended as a test in the sb job
            #Create a copy of the data since we'll be modifying it
            cdm_test_data_copy = copy.deepcopy(cdm_test_data)
            cdm_test_data_copy.pop('workflowId', None)
            #cdm_test_data.pop('testPlanIndex', None)
            cdm_test_data_copy['testPlanIndex'] = get_next_test_plan_index()
            jm_results['data']['ijm_additionalTest'] = True
            if cdm_test_data_copy.get('testLocations'):
                cdm_test_data_copy['testLocations'][0].pop('workflowId', None)
                cdm_test_data_copy['testLocations'][0].pop('testPlanIndex', None)
                cdm_test_data_copy['testLocations'][0]['required'] = False
                cdm_test_data_copy['testLocations'][0]['results'] = jm_results
                cdm_test_data_copy.pop('results', None)
            else:
                cdm_test_data_copy['required'] = False
                cdm_test_data_copy['results'] = jm_results

            # since test data will always have data only for 1 test, the results block can stay where it is
            self.cdm_job['tests'].append(cdm_test_data_copy)

            # don't return an index since this is an added test
            test_index = len(self.cdm_job['tests']) - 1
            test_loc_index = None
            #return (None, None)
        else:
            test_index, test_loc_index = test_tuple
            # if its an empty test location, i keep the results block in the test block (like the original cdm)
            # otherwise the results block is inside each test location
            # for the test wihout a location, if it is matched the sb test will be overwritten each time by a new result
            #results will point to either the results object in the test/location or test if there is no location
            if test_loc_index is not None:
                self._add_test_data_results(jm_results, self.cdm_job['tests'][test_index]['testLocations'][test_loc_index])
            else:
                if cdm_test_data.get('testLocations') and not self.cdm_job['tests'][test_index].get('testLocations'):
                    #This is a case where the SB job doesn't have testLocations, but the results have a recorded location. 
                    #If this happens we want to update our SB job to include the location for future matching. 
                    self.cdm_job['tests'][test_index]['testLocations'] = [cdm_test_data['testLocations'][0]]
                    self._add_test_data_results(jm_results, self.cdm_job['tests'][test_index]['testLocations'][0])
                else:
                    self._add_test_data_results(jm_results, self.cdm_job['tests'][test_index])
        
        self.job_ui_view.update(self.cdm_job)
        return (test_index, test_loc_index)

    def _add_test_data_results(self, jm_results, test_entry):
        #We want to replace the results with the latest results, but for report generation we want to keep previous test results so we have this mess here. 
        #ijm_testData is an array of test objects to store current and previous test results. 
        ijm_testData = []
        if not 'results' in test_entry:
            test_entry['results'] = {}

        if test_entry['results'].get('data') and test_entry['results']['data'].get('ijm_testData'):
            #We already have some previous historical results (i.e. this test has been run 2 or more times already). We copy these results to a temp. 
            #Copy existing previous results into our temp ijm_testData
            ijm_testData = copy.deepcopy(test_entry['results']['data'].pop('ijm_testData'))     
            ijm_testData.append(copy.deepcopy(test_entry['results']))   
            #Since our matching algorithm will find test/location match if this is an added test that has been repeated, we need to make sure we 
            #copy in ijm_additionalTest=True if this is a repeated additional test
            if test_entry['results']['data'].get('ijm_additionalTest') and 'data' in jm_results:
                jm_results['data']['ijm_additionalTest'] = True
            #Add our latest results
            test_entry['results'] = jm_results
            #Add the previous results to our historical data (ijm_testData)     
            test_entry['results']['data']['ijm_testData'] = ijm_testData
        elif test_entry['results'].get('data'):
            #We already have one run. We'll copy our previous result into our ijm_testData test array
            ijm_testData.append(copy.deepcopy(test_entry['results']))
            #Since our matching algorithm will find test/location match if this is an added test that has been repeated, we need to make sure we 
            #copy in ijm_additionalTest=True if this is a repeated additional test
            if test_entry['results']['data'].get('ijm_additionalTest') and 'data' in jm_results:
                jm_results['data']['ijm_additionalTest'] = True
            #Add our latest results
            test_entry['results'] = jm_results
            #Add the previous result to our historical data (ijm_testData)
            test_entry['results']['data']['ijm_testData'] = ijm_testData
        else:
            #We don't have any previous historical results so we just save our latest results
            test_entry['results'] = jm_results

    def get_test_data_html_format(self):
        """
        Method to prepair the test_data list to be rendered as html in the
        creation of reports
        """
        test_data_html_format = []
        for test_data in self.get_other_test_data():
            if 'results' in test_data:
                test_data_html_format.append(html_format_test_data(test_data, test_data['results']))                                 
        return test_data_html_format

    def get_test_plan_html_format(self):
        """
        Method to prepair the test_plan list to be rendered as html in the
        creation of reports
        """
        test_plan_html_format = [html_format_test_plan(sb_test)
                                 for sb_test
                                 in self.get_required_test()]
        return test_plan_html_format

    def get_job_info_html_format(self):
        """
        Method to prepair the job_info list to be rendered as html in the
        creation of reports
        """
        job_info_html_format = html_format_job_info(self.cdm_job)
        return job_info_html_format

    def get_logo(self):
        # gets the logo from cdm for report
        logo_val = ""
        workflow = self.cdm_job['workflow']
        if workflow.get('domainAttributes'):
            logo_val = workflow['domainAttributes'].get('logo','')
        return logo_val

    def get_other_test_data(self):
        # gets list additional tests
        other_test_data_list = []
        for test in self.cdm_job.get('tests', []):
            if ('required' in test) and (test['required'] == False):
                other_test_data_list.append(test)
        return other_test_data_list
    
    def get_required_test(self):
        # gets list of required tests
        required_test_list = []
        for test in self.cdm_job.get('tests', []):
            if test.get('testLocations'):
                #We have locations. We'll create a new test entry for each location with only one location per entry.
                test_copy = copy.deepcopy(test)
                test_copy.pop('testLocations')
                for location in test['testLocations']:
                    if 'required' not in location or location['required'] == True:
                        test_with_current_location = copy.deepcopy(test_copy)
                        if 'label' in location and location['label']:
                            test_with_current_location['testLocations'] = {}
                            test_with_current_location['testLocations']['label'] = [location['label']]
                            #Append the location label to the test label
                            test_with_current_location['label'] = test_with_current_location['label'] + " : " + location['label']
                        if 'results' in location:    
                            test_with_current_location['results'] = location['results']
                        required_test_list.append(test_with_current_location)
            else:
                if 'required' not in test or test['required'] == True:
                    required_test_list.append(test)

        return required_test_list

    def has_other_test_data(self):
        # checks if job has any added tests
        if self.cdm_job.get('tests'):
            for test in self.cdm_job['tests']:
                if ('required' in test) and (test['required'] == False):
                    return True
        return False

    def has_required_test(self):
        # checks if job has any required tests
        if self.cdm_job.get('tests'):
            for test in self.cdm_job['tests']:
                if ('required' not in test) or (test['required'] == True):
                    return True
        return False

    def get_cdm_workflow(self):
        '''
        Method to get the JSON CDM 'workflow' block

        Args:

        Returns:
            workflow (string): the json 'workflow' block
        '''
        return self.cdm_job['workflow']

    @property
    def logo(self):
        #TODO. Put this here to make unit test pass
        return None

    #property 
    def workOrderId(self):
        wo = None
        try:
            wo = self.cdm_job['workflow']['workOrderId']
        except:
            pass
        return wo

    @property
    def cdm(self):
        return self.cdm_job
        

    def find_test_plan_index_generic(self, test_plan, test_data, legacy):
        '''
        Function to determine if test data is an expected test in the test plan

        Args:
            test_plan (list): list of sb_test dictionaries
            test_data (dict): dictionay of the new test data

        Returns:
            tuple testIndex(int), locationIndex(int)) 
                None if the test_data does not match any entries in the test plan
        '''
        
        # we need to check the test type of test_data and get all the test which have that test type
        # then we need to sort those tests according to priority 
        # then we check if that test type has a product specific matching function or not
        match_tests = 0
        # iterate through the sorted test plan
        for index, sb_test in test_plan:
            # for legacy instruments skip matching for optional tests
            # find the match
            match_tuple = self.matches_generic(sb_test, test_data, legacy, index)
            # if the test matched or not
            match_result = match_tuple[0]
            # if test matched exactly with the loc attributes
            match_with_attribs = match_tuple[1]
            # the location at which test matched
            match_loc = match_tuple[2]
            # the loc index within the test, at which test data matched
            match_loc_index = match_tuple[3]
            if match_result:
                # if there is an exact match with location attribute return that matched test
                if match_with_attribs:
                    # return the matched location block, the test index, and the test location index within that test
                    return (index, match_loc_index)
                else:
                    # if there is no match with the exact location attributes, then assign it to the first wildcard loc attrib encountered
                    if match_tests == 0:
                        match_tuple2 = (index, match_loc_index)
                        match_tests = match_tests + 1
        # if there are no matched test then return none
        if not match_tests:
            return None
        else:
            return match_tuple2

    def matches_generic(self, test_dict, test_data, legacy, test_index):
            """
            Method to determine if test data matches that expected by the planned test

            test_type has to match
            if there is reference info that has to match in any order of key/value pairs
            if there is not reference info matches if the planned test is in To Do status
            if there is extra subType info this has to match in addition to the reference info matching

            Args:
                test_data (dict): dictionary of the test data

            Returns:
                (bool): True if the test data matches False if not
                tuple (matchFound-True/False, this boolean is true when an exact match with loc attributes is found, Found a matched test location, index of matched test location)
            """

            testLoc_found = False
            testLoc = False
            # this boolean is true when an exact match with loc attributes is found
            testLoc_withAttribs = False 
            testLoc_matched = None
            index = None
            # first check whether test types match
            if not (test_dict.get('type') == test_data.get('type')):
                return (False, testLoc_withAttribs, testLoc_matched, index)
            
            # check whether test attributes match
            if not legacy and test_dict.get('attributes'):
                attrib_result = test_data.get('attributes', {})
                attrib_test = test_dict['attributes']
                # this checks if sb test attributes is a subset of test data attributes
                # we use <= because test data can have more items than sb test, but should also contain all items of sb test attribs
                if not dict_compare_compat(attrib_test, attrib_result):
                    return (False, testLoc_withAttribs, testLoc_matched, index)

            # check whether test configs match
            # for old instruments skip this match
            if not legacy and test_dict.get('configuration'):
                config_result = test_data.get('configuration', {})
                config_test = test_dict['configuration']
                # same logic as test attributes
                if not dict_compare_compat(config_test, config_result):
                    return (False, testLoc_withAttribs, testLoc_matched, index)

            if test_dict.get('testLocations'):
                testLoc_result_dict = {}
                testLoc_attribs_result = {}
                testLoc = True
                testLoc_results_list = test_data.get('testLocations', [])
                if testLoc_results_list:
                    testLoc_result_dict = testLoc_results_list[0] # Assuming test data returns a single location as a list containing only 1 item                    
                if testLoc_result_dict.get('attributes'):
                    testLoc_attribs_result = testLoc_result_dict.get('attributes') # we want to compare the test location attributes seperately
                for testLoc_index, testLoc_test in enumerate(test_dict['testLocations']):
                    if testLoc_test.get('attributes'):
                        testLoc_attribs_test = testLoc_test.get('attributes', {}) # we want to compare the test location attributes seperately
                        testLoc_result = testLoc_result_dict
                        if not legacy:
                            # this checks if the location labels and the location attributes match
                            if (testLoc_test.get('label') == testLoc_result.get('label') and dict_compare_compat(testLoc_attribs_test, testLoc_attribs_result)):
                                    testLoc_found = True
                                    # found an exact match of test loc attributes
                                    testLoc_withAttribs = True
                                    testLoc_matched = testLoc_test
                                    index = testLoc_index
                                    break
                        else:
                            # For legacy just match reference info since it won't have lavel or configInfo
                            if dict_compare_compat(testLoc_attribs_test.get('referenceInfo', {}), testLoc_attribs_result.get('referenceInfo', {})):
                                    testLoc_found = True
                                    # found an exact match of test loc attributes
                                    testLoc_withAttribs = True
                                    testLoc_matched = testLoc_test
                                    index = testLoc_index
                                    break

                # this is for when the location attributes do not have an exact match, then we have to look for wildcards
                if not testLoc_found:
                    for testLoc_index2, testLoc_test2 in enumerate(test_dict['testLocations']):
                        if not testLoc_test2.get('attributes'):
                            if (legacy or (testLoc_test2.get('label') == testLoc_result_dict.get('label'))):
                                testLoc_found = True
                                testLoc_withAttribs = False
                                testLoc_matched = testLoc_test2
                                index = testLoc_index2
                                break

            if not testLoc_found and test_dict.get('openEndedStatus', '').lower() == 'open':            
                cdm_job, location_index = self.add_test_location(test_data, test_index)
                testLoc = True
                testLoc_found = True
                testLoc_withAttribs = True
                testLoc_matched = True
                index = location_index

            if testLoc:
                if not testLoc_found:
                    return (False, testLoc_withAttribs, testLoc_matched, index)
            else:
                testLoc_matched = test_dict

            return (True, testLoc_withAttribs, testLoc_matched, index)


def sort_restriction(launch_test_index, index_test_tuple):
    # The following algorithm gives highest value to most restricive and lowest to least restrictive
    # This function does not consider location attributes, since we compute that after we get our matches
    # The first argument is the test launch index (i.e. the index of the test the user launched) which 
    # gives small amount of priority so that if all other configurations are equal, we prioritize
    # the user launched test
    original_test_index = index_test_tuple[0] 
    test = index_test_tuple[1]
    val = 100
    if not test.get('required', True):
        val = 1
    if test.get('testLocations'):
        val += 50
    if test.get('configuration'):
        val += 25
    if test.get('attributes'):
        val += 10
    #If this test is the user launched test, bump the priority slightly.
    #We want to give this a bit more weight than tests without results
    if launch_test_index >= 0 and launch_test_index == original_test_index:
        val += 2
    # Bump up priority slightly to give weight to tests that are otherwise equally matching,
    # but don't already have a test result 
    if not "results" in test:
        val += 1
    
    return val

#Should use this function for comparison. This makes the copies of the input arguments and then passes them to the below recursive function. 
def dict_compare_compat(dict1, dict2):
    return __dict_compare_compat(copy.deepcopy(dict1), copy.deepcopy(dict2))

#Compare dictionaries. Returns true if all keys in dict1 exist in dict2 and their values match. dict2 can be a superset of dict1 
#since solutions may add additional data to the CDM attributes/configuration. 
#This is a recursive function that recurses when it finds subdictionaries. 
#WARNING THIS FUNCTION MAY MODIFY THE INPUT DICTIONARIES so be sure to use copy.deepcopy(dict) for the input arguments. 
def __dict_compare_compat(dict1, dict2):
    dict1_dicts = {}
    dict2_dicts = {}
    for k in list(dict1.keys()):
        if type(dict1[k]) is dict:
            dict1_dicts[k] = dict1[k]
            del(dict1[k])
    for k in list(dict2.keys()):
        if type(dict2[k]) is dict:
            dict2_dicts[k] = dict2[k]
            del(dict2[k])    
    if dict1_dicts:
        for k in dict1_dicts.keys():
            if k in dict2_dicts:
                compat = __dict_compare_compat(dict1_dicts[k], dict2_dicts[k])
                if not compat:
                    return False
            else:
                return False
    return dict1.items() <= dict2.items()

def remove_empty_objects_dict(d):
    rd = {}
    for k, v in d.items():
        if isinstance(v, dict):
            v = remove_empty_objects_dict(v)
        elif isinstance(v, list):
            v = remove_empty_objects_list(v)
        if v not in (None, '', [], {}):
            rd[k] = v
    return rd

def remove_empty_objects_list(l):
    rl = []
    for v in l:
        if isinstance(v, dict):
            v = remove_empty_objects_dict(v)
        elif isinstance(v, list):
            v = remove_empty_objects_list(v)
        if v not in (None, '', [], {}):
            rl.append(v)
    return rl
