import csv
import traceback
from rest_api.api.job_manager.helper import get_next_test_plan_index
from rest_api.api.job_manager.helper import cdmtest_applies_to_instrument
import logging
log = logging.getLogger(__name__)

"""
Modifies incoming cdm_job with imports from csv_filepath
https://conf1.ds.jdsu.net/wiki/display/CDM/CSV+Label+lists

Args: 
csv_filepath filepath of csv import
cdm_job job to add imports to. Will be modified
assetType, model - use to determine which tests should be applicable for imports. Only imports into tests in job applicable to this instrument
Returns: True on success / False on failure
    
"""

def import_csv_to_job(csv_filepath, cdm_job, assetType, model):
    rv = False
    try:
        with open(csv_filepath, newline='', encoding='utf-8-sig') as csvfile:
            has_header = csv.Sniffer().has_header(csvfile.read(1024))
            csvfile.seek(0)
            if has_header:
                location_import = list(csv.DictReader(csvfile, delimiter=','))
                rv = _apply_location_to_job(location_import, cdm_job, assetType, model)
            else:
                #Legacy format as a single row of label imports
                label_import = list(csv.reader(csvfile, delimiter=','))
                rv = _apply_simple_label_to_job(label_import, cdm_job, assetType, model)
    except:
        rv = False
        log.debug(traceback.format_exc())
    return rv

def _apply_simple_label_to_job(imported_labels, cdm_job, assetType, model):
    #Assume we need at least more than one label to make this a valid import
    if len(imported_labels) < 2:
        return False
    for test in cdm_job.get('tests', []):          
        if cdmtest_applies_to_instrument(test, assetType, model):
            if 'testPlanIndex' not in test:
                test['testPlanIndex'] = get_next_test_plan_index()            
            #Check test type to see if it allows location imports
            if not test.get('testLocations'):
                test['testLocations'] = []
                #https://conf1.ds.jdsu.net/wiki/display/CDM/workflowId+and+testPlanIndex+Usage says to removed testPlanIndex from the test when importing labels
                #if the test is not open ended.
            for label in imported_labels:
                #If the first enty in the imported_labels is a string named label then it is a header so don't include it. 
                if len(label) > 0 and label[0].lower() != "label":
                    test['testLocations'].append({"label": label[0], "testPlanIndex" : get_next_test_plan_index()})
    return True

def _apply_location_to_job(imported_locations, cdm_job, assetType, model):
    if len(imported_locations) < 2:
        return False

    for test in cdm_job.get('tests', []):     
        if cdmtest_applies_to_instrument(test, assetType, model):
            if 'testPlanIndex' not in test:
                test['testPlanIndex'] = get_next_test_plan_index()            
            #Check test type to see if it allows location imports
            for imported_location in imported_locations:
                testType = imported_location.get('testType', '')
                #if not testType or testType == '""' or testType == "''" or testType == test['type']: 
                if not testType or testType == test['type']: 
                    #Only import to this test if the import doesn't specify testType or if the testType matches the test
                    if not test.get('testLocations'):
                        test['testLocations'] = []
                    locationObject = {}
                    locationObject["label"] = imported_location['label']
                    locationObject["testPlanIndex"] = get_next_test_plan_index()            
                    #newlist = [x for x in fruits if "a" in x]
                    refinfo = {(k.split(":")[1]):v.strip() for (k,v) in imported_location.items() if k.startswith("ref:")}
                    configinfo = {(k.split(":")[1]):v.strip() for (k,v) in imported_location.items() if k.startswith("config:")}
                    if refinfo or configinfo:
                        locationObject["attributes"] = {}
                        if refinfo:
                            locationObject["attributes"]["referenceInfo"] = refinfo
                        if configinfo:
                            locationObject["attributes"]["configInfo"] = configinfo
                    test['testLocations'].append(locationObject)
    return True
