import traceback
import copy
import logging

from bottle import request, response
from rest_api.products.config_schema_common import SubTypeRefInfoFormatter
from .job_merge import merge_job
import datetime
import time
from dateutil import parser
import random
log = logging.getLogger(__name__)

"""
Module containing helper functions primarily for formating objects for rendering
in html
"""

def html_format_job_info(job):
    job_info_dict = {}
    job_workflow = job['workflow']
    if job_workflow.get('customerInfo'):
        job_info_dict['customer_name'] = job_workflow['customerInfo'].get('company','')
    job_info_dict['job_number'] = job_workflow.get('workOrderId','')
    if job_workflow.get('techInfo'):
        job_info_dict['technician_id'] = job_workflow['techInfo'].get('techId','')
    if job_workflow.get('jobManagerAttributes'):
        if job_workflow['jobManagerAttributes'].get('cableId'):
            job_info_dict['cable_id'] = job_workflow['jobManagerAttributes']['cableId']['value']
        if job_workflow['jobManagerAttributes'].get('testLocationA'):
            job_info_dict['test_location'] = job_workflow['jobManagerAttributes']['testLocationA']['value']
        if job_workflow['jobManagerAttributes'].get('testLocation'):
            job_info_dict['test_location'] = job_workflow['jobManagerAttributes']['testLocation']['value']
        if job_workflow['jobManagerAttributes'].get('testLocationB'):
            job_info_dict['test_location_b'] = job_workflow['jobManagerAttributes']['testLocationB']['value']

        # More flexible but doesn't guarantee the order needed for report headers
        # for attrib_key in job_workflow['jobManagerAttributes']:
        #     log.debug('jobManagerAttributes key: %s, value = %s', attrib_key, get_job_attr_value( job_workflow['jobManagerAttributes'], attrib_key) )
        #     if attrib_key == 'testLocationA':
        #         job_info_dict['test_location'] = job_workflow['jobManagerAttributes']['testLocationA']['value']
        #     elif attrib_key == 'testLocation': # Metro doesn't use Loc A & Loc B
        #         job_info_dict['test_location'] = job_workflow['jobManagerAttributes']['testLocation']['value']
        #     elif attrib_key == 'testLocationB':
        #         job_info_dict['test_location_b'] = job_workflow['jobManagerAttributes']['testLocationB']['value']
        #     elif attrib_key == 'cableId':
        #         job_info_dict['cable_id'] = job_workflow['jobManagerAttributes']['cableId']['value']
        #     elif attrib_key == 'jobComments':
        #         job_info_dict['job_comments'] = job_workflow['jobManagerAttributes']['jobComments']['value']
        #     else:
        #         # would be nice to be able to catch any extra keys here but they won't make the report (cf. report_languages.py)
        #         # job_info_dict[ str(attrib_key) ] = get_job_attr_value( job_workflow['jobManagerAttributes'], attrib_key )
        #         log.debug('Unexpected jobManagerAttributes key: %s, value = %s', attrib_key, get_job_attr_value( job_workflow['jobManagerAttributes'], attrib_key) )

    job_info_dict['contractor_id'] = job_workflow.get('contractorId','')
    if job_workflow.get('jobManagerAttributes'):
        if job_workflow['jobManagerAttributes'].get('jobComments'):
            job_info_dict['job_comments'] = job_workflow['jobManagerAttributes']['jobComments']['value']
    
    return job_info_dict

# def get_job_attr_value(structure, value):
#     if (value in structure):
#         return structure[value]['value']
#     else:
#         return ""

def html_format_test_plan(planned_test):
    planned_test_dict = {}
    planned_test_dict['test_type'] = planned_test['type']
    planned_test_dict['test_label'] = planned_test.get('label','')
    if planned_test.get('results'):
        planned_test_dict['status'] = get_formated_status(planned_test['results']['status'])
    else:
        planned_test_dict['status'] = 'To Do'
    config = get_test_ui_config_format(planned_test, 0)
    planned_test_dict['reference_info'] = config
    planned_test_dict['procedures'] = planned_test.get('procedures','')
    test_data_html_format = []
    if planned_test.get('results') and planned_test['results'].get('data'):
        test_data_html_format.append(html_format_test_data(planned_test, planned_test['results']))
        for previous_test in planned_test['results']['data'].get('ijm_testData',[]):
            if 'results' in previous_test:
                #For backwards compatiblity with jobs saved with IJM 4.0
                previous_test = previous_test['results']
            test_data_html_format.append(html_format_test_data(planned_test, previous_test))
    planned_test_dict['test_data'] = test_data_html_format
    return planned_test_dict

def html_format_test_data(planned_test, test_data):
    '''
    Helper function to format a single dictionary of test data into a format
    that is easier for the template to consume

    Args:
        test_data (dict): dictionary of the test data

    Returns:
        test_data (dict): the formatted test data
    '''
    test_data_dict = {}
    test_data_dict['verdict'] = get_formated_verdict(test_data['status'])
    test_data_dict['test_type'] = planned_test['type']
    test_data_dict['filename'] = get_file_name(test_data['data'].get('ijm_filePath',''))

    reference_info_str_list = get_test_ui_config_format(planned_test, 0)

    test_data_dict['reference_info_str_list'] = reference_info_str_list

    test_data_dict['comments'] = test_data.get('comment','')
    return test_data_dict

def get_formated_verdict(status):
    """
    Function to format the verdict report redable format

    Agrs:
        test_data (dict): the test data dict
    """
    if status == "none":
        verdict = 'N/A'
    elif status == 'pass':
        verdict = '<span class="pass">Pass</span>'
    elif status == 'fail':
        verdict = '<span class="fail">Fail</span>'
    elif status == 'marginal':
        verdict = '<span class="marginal">Marginal</span>'
    elif status == 'skipped':
        verdict = 'Skipped'
    else:
        verdict = status
    return verdict

def get_formated_status(status):
    """
    Function to format the planned test status into a format
    that is able to be processed by the template

    Args:
        status (str): the status of the planned test
    """
    if not status:
        formated_status = "To Do"
    
    if status == 'pass':
        formated_status = '<span class="pass">Pass</span>'
    elif status == 'fail':
        formated_status = '<span class="fail">Fail</span>'
    elif status == 'marginal':
        formated_status = '<span class="marginal">Marginal</span>'
    elif status == 'skipped':
        formated_status = 'Skipped'
    elif status == 'none':
        formated_status = 'Complete'
    else:
        formated_status = status

    return formated_status

def get_file_name(file_path):
    """
    Function to return the file name from the file path

    Args:
        file_path (str): the filepath associated with test data. If multiple files are associated, a comma separeted list is generated. 
    """
    filenames = ""
    if type(file_path) is list:
        for idx, fp in enumerate(file_path):
            filenames += fp.rsplit('/', 1)[-1]
            if idx < len(file_path) - 1:
                filenames += ", "
    else:        
        filenames = file_path.rsplit('/', 1)[-1]
    return filenames

def get_test_ui_config_format(cdm_test, test_location_index):
    rv = None
    response.status = 404
    try:
        test_type = cdm_test["type"]
        product_specific = request.app.config["rest_api.product_specific"]
        test_definitions = product_specific.test_definitions
        test_type = test_definitions.get(test_type, None)
        
        rv = None
        #Default to SubTypeRefInfoFormatter
        ui_config_parameters_func = SubTypeRefInfoFormatter.ui_config_parameters
        if hasattr(test_type, 'ui_config_parameters'):
            #Check if the test has a custom ui_config_parameters function
            ui_config_parameters_func = test_type.ui_config_parameters
        elif hasattr(product_specific, 'ui_config_parameters'):
            #Check if the product has a custom ui_config_parameters function
            ui_config_parameters_func = product_specific.ui_config_parameters
        if ui_config_parameters_func:
            rv = ui_config_parameters_func(cdm_test, test_location_index)
        if rv:
            response.status = 200
    except:
        print(traceback.format_exc())
    return rv

def gen_dict_extract(key, var):
    if hasattr(var,'items'): 
        for k, v in var.items(): 
            if k == key:
                yield v
            if isinstance(v, dict):
                for result in gen_dict_extract(key, v):
                    yield result
            elif isinstance(v, list):
                for d in v:
                    for result in gen_dict_extract(key, d):
                        yield result

def clean_dict(obj, func):
    """
    This method scrolls the entire 'obj' to delete every key for which the 'callable' returns
    True

    :param obj: a dictionary or a list of dictionaries to clean
    :param func: a callable that takes a key in argument and return True for each key to delete
    """
    if isinstance(obj, dict):
        # the call to `list` is useless for py2 but makes
        # the code py2/py3 compatible
        for key in list(obj.keys()):
            if func(key):
                del obj[key]
            else:
                clean_dict(obj[key], func)
    elif isinstance(obj, list):
        for i in reversed(range(len(obj))):
            if func(obj[i]):
                del obj[i]
            else:
                clean_dict(obj[i], func)

    else:
        # neither a dict nor a list, do nothing
        pass

#Returns true if job has any test plan indexes
def job_has_test_plan_index(cdm_job):
    tpis = list(gen_dict_extract('testPlanIndex', cdm_job))
    return len(tpis) > 0

#Return the next test plan index to use in a job
def get_next_test_plan_index():
    return random.randrange(0, 2147483647)
    
def remove_all_test_plan_index(cdm_job):
    rv = copy.deepcopy(cdm_job)
    clean_dict(rv, lambda key: key == 'testPlanIndex')
    return rv

def remove_all_test_plan_index_and_workflowid(cdm_job):
    rv = copy.deepcopy(cdm_job)
    clean_dict(rv, lambda key: key == 'testPlanIndex' or key == 'workflowId')
    return rv

def test_is_additional(cdm_test):
    rv = False
    results = cdm_test.get('results')
    if results:
        data = results.get('data')
        if data:
            rv = data.get('ijm_additionalTest')
    return rv

def cdmtest_applies_to_instrument(test, assetType, model):        
    applies = False
    deployableTo = test.get('deployableTo')
    if deployableTo:
        for deployable in deployableTo:
            required_asset_type = deployable.get('assetType')
            if not required_asset_type or assetType == required_asset_type:
                required_model =  deployable.get('model')
                if not required_model or model == required_model:
                    applies = True
    else:
        applies = True
    return applies      

def get_externally_managed_job_workorder_ids(jobs_list):
    externally_managed_job_list = list(filter(lambda j: not is_job_locally_managed(j), jobs_list))
    #For mobile jobs create a list of work order IDs
    externally_managed_workorder_ids = [x['workflow']['workOrderId'] for x in externally_managed_job_list]
    return externally_managed_workorder_ids

def set_job_to_locally_managed(cdm_job):
    # if not 'configuration' in cdm_job:
    #     cdm_job['configuration'] = {}
    if 'jobManagement' not in cdm_job['workflow']:
        cdm_job['workflow']['jobManagement'] = {}
    cdm_job['workflow']['jobManagement']['createdBy'] = "inst"
    cdm_job['workflow']['jobManagement']['managedBy'] = "inst"
    
#Job is locally managed if it is a legacy job (2.1) or managedBy == inst
def is_job_locally_managed(cdm_job):
    rv = True
    jobmgmt =  cdm_job['workflow'].get('jobManagement')
    if jobmgmt:
        if jobmgmt.get('managedBy') != 'inst':
            rv = False
    return rv

def job_has_conflict(cdm_job):
    rv = False
    jobmgmt =  cdm_job['workflow'].get('jobManagement')
    if jobmgmt:
        if jobmgmt.get('state') == 'conflict':
            rv = True
    return rv

def set_job_management_state(cdm_job, state):
    if 'jobManagement' not in cdm_job['workflow']:
        cdm_job['workflow']['jobManagement'] = {}
    cdm_job['workflow']['jobManagement']['state'] = state

def set_job_conflicted(cdm_job):
    set_job_management_state(cdm_job, 'conflict')

def remove_job_conflict(cdm_job):
    if 'jobManagement' in cdm_job['workflow']:
        cdm_job['workflow']['jobManagement'].pop('state', None)

def job_is_resolved(cdm_job):
    rv = False
    jobmgmt =  cdm_job['workflow'].get('jobManagement')
    if jobmgmt:
        if jobmgmt.get('state') == 'resolved':
            rv = True
    return rv

def job_is_resolved_clear(cdm_job):
    rv = False
    jobmgmt =  cdm_job['workflow'].get('jobManagement')
    if jobmgmt:
        if jobmgmt.get('state') == 'resolvedClear':
            rv = True
    return rv


def get_workorder_ids(jobs_list):
    workorder_ids = [x['workflow']['workOrderId'] for x in jobs_list]
    return workorder_ids

def get_job_from_workflow(workflow, workOrderId):
    job = None
    for j in workflow['cdm']:
        if j['workflow']['workOrderId'] == workOrderId:
            job = j
            break
    return job

def get_template_from_template_array(template_array, typeName):
    template = None
    for t in template_array:
        if t['workflow']['typeName'] == typeName:
            template = t
            break
    return template

def get_local_timestamp_8601():
    timestamp = None
    try:
        #python versions older than python 3.6, don't handle astimezone, so handle the exception and just use a UTC timestamp.  
        timestamp = datetime.datetime.now().astimezone().replace(microsecond=0).isoformat()
    except:
        pass

    if not timestamp:
        try:
            timestamp = datetime.datetime.utcfromtimestamp(time.time()).replace(microsecond=0).isoformat() + "+00:00"
        except:
            pass
    return timestamp

def set_job_modified_to_now(cdm_job):
    cdm_job['jobModifiedOn'] = get_local_timestamp_8601()

def are_iso8601_timestamps_equal(iso8601_1, iso8601_2):
    rv = False
    try:
        if isinstance(iso8601_1, str):
            iso8601_1 = parser.parse(iso8601_1)
        if isinstance(iso8601_2, str):
            iso8601_2 = parser.parse(iso8601_2)
        rv = iso8601_1 == iso8601_2
    except:
        pass
    return rv

def is_iso8601_timestamps_gt(iso8601_1, iso8601_2):
    rv = False
    try:
        if isinstance(iso8601_1, str):
            iso8601_1 = parser.parse(iso8601_1)
        if isinstance(iso8601_2, str):
            iso8601_2 = parser.parse(iso8601_2)
        rv = iso8601_1 > iso8601_2
    except:
        pass
    return rv


#Merges remote job into localjob if job modified time is more recent
def merge_jobs(localjob, remotejob):
    job_changed = False
    woid = localjob['workflow']['workOrderId']
    try:
        local_modified_on = localjob.get('jobModifiedOn')
        remote_modified_on = remotejob.get('jobModifiedOn')
        if is_iso8601_timestamps_gt(remote_modified_on, local_modified_on):
            log.debug("merge_jobs merging due to modified jobModifiedOn")
            if job_is_resolved_clear(remotejob):
                #Stratasync has resolved the job and we need to clear out any test results and use the StrataSync job
                #Since we are modifying the dictionary in place, we can't just use a simple assignment
                localjob.clear()
                localjob.update(remotejob)
                job_changed = True
            elif job_has_conflict(remotejob):
                #If there is a conflict we want to reflect that in the local job. We don't merge until we get a resolved state 
                set_job_conflicted(localjob)
                #Since we changing the job set rv to True
                job_changed = True
            else: #In this else clause we cover a normal job merge and a workflow.jobManagement.state == 'resolved'
                #Since 'error' is optional, we want to remove the local error before the merge since we know the incoming error is not conlict
                remove_job_conflict(localjob)
                if merge_job(localjob, remotejob):
                    log.debug("merge_jobs success: localjob{}\nremotejob: {}".format(localjob, remotejob))
                    job_changed = True
                else:
                    log.error("merge_jobs failed to merge job localjob: {}\nremotejob: {}".format(localjob, remotejob))
        else:
            log.debug("merge_jobs not merging due to timestamps remote_modified_on:{} local_modified_on:{}".format(remote_modified_on, local_modified_on)) 
    except:
        print(traceback.format_exc())
    return job_changed

def is_job_archived(cdm_job):
    rv = False
    if 'configuration' in cdm_job and cdm_job['configuration'].get('ijm_archived', False):
        rv = True
    return rv

def strip_metadata_type_name(_type):
    """
    Function to strip the name of the type of image file being sent

    args:
        _type: full 'type' sent as part of metadata eg. "MISC.Signature"
    returns:
        stripped type field eg. "Signature"
    """
    return _type.split('.').pop()

def get_type_of_file(metadata):
    """
    Function to return the 'type' field from the incoming metadata

    args:
        metadata: python dict of metadata
    returns:
        'type' field of metadata eg. MISC.Signature
    """
    return strip_metadata_type_name(metadata['type']) if 'type' in metadata else 'Unknown'
