'''Module containing the DataStore class for working with rest api database
'''
import json

import os
import tinydb
import logging
import threading
import traceback
from .job import Job
from .template import default_template
from .cdm import cdm_schemas
from rest_api.utils.marshmallow_compat import marshmallow_data_compat
log = logging.getLogger(__name__)

class DataStore(object):
    '''
    Class to manage the datastore currently implemented as a file on the file system
    '''
    def __init__(self, job_manager_file):
        dir_path = os.path.dirname(job_manager_file)
        if not os.path.isdir(dir_path):
            os.makedirs(dir_path)
        try:
            self.db = tinydb.TinyDB(job_manager_file) # pylint: disable=C0103
        except ValueError:
            log.warning("Unable to parse DB file - starting new DB")
            open(job_manager_file, 'w').close()
            self.db = tinydb.TinyDB(job_manager_file) # pylint: disable=C0103

        self.jobs = self.db.table('jobs')
        self.templates = self.db.table('templates')
        self.active = self.db.table('active')
        self.job_item_active_index = self.db.table('job_item_active_index')

        if not self.templates:
            #Add the default template if the table does not exist or is empty
            self.templates.insert(default_template)

        if not self.active:
            self.active.insert({'active_job': None})

        if not self.job_item_active_index:
            self.job_item_active_index.insert({'active_job_item_index': None})

        self.job_manager_file = job_manager_file
        self.lock = threading.RLock()

    def get_jobs_info(self):
        """
        Method to return a list of the workflow object and jobModifiedOn for each job in datastore

        Returns:
            (list): the list of workflow, jobModifiedOn
        """
        with self.lock:
            jobinfo_array = []
            for j in self.jobs.all():

                # Don't process empty jobs
                if "workflow" not in j or not j["workflow"]:
                    continue

                jobinfo = {}
                try:
                    jobinfo["cdmVersion"] = j["cdmVersion"]
                    jobinfo["workflow"] = j["workflow"]

                    jobinfo["ijm_archived"] = False
                    if "configuration" in j:
                        jobinfo["ijm_archived"] = j["configuration"].get("ijm_archived", False)

                    jobinfo["jobModifiedOn"] = j.get("jobModifiedOn", "")
                    job_object = Job(j)
                    job_status_ratio = job_object.status_ratio()
                    jobinfo["totalNumberOfTests"] = job_status_ratio["total_count"]
                    jobinfo["numberOfCompletedTests"] = job_status_ratio["complete_count"]

                    jobinfo_array.append(jobinfo)
                except:
                    log.error("Exception occured when processing job " + str(j))
                    log.error(traceback.format_exc())
                    continue

            return jobinfo_array

    def get_templates(self):
        """
        Method to return a list of templates

        Returns:
            (list): the list of workflow, jobModifiedOn
        """
        with self.lock:
            return self.templates.all()

    def get_templates_info(self):
        """
        Method to return a list of the workflow object and jobModifiedOn for each template in datastore

        Returns:
            (list): the list of workflow, jobModifiedOn
        """
        templateinfo_array = []
        with self.lock:
            for t in self.templates.all():
                templateinfo = {}
                try:
                    templateinfo["workflow"] = t["workflow"]
                    templateinfo["jobModifiedOn"] = t.get("jobModifiedOn", "")

                    templateinfo_array.append(templateinfo)
                except:
                    log.debug("Exception occured when processing template " + str(t))
                    continue

            return templateinfo_array

    def read_job_by_work_order_id(self, work_order_id):
        '''
        Method to read the job by workOrderId from the datastore

        Returns: the dictionary representation of the job or None
            if the job does not exist
        '''
        with self.lock:
            Job = tinydb.Query()
            return self.jobs.get(Job.workflow.workOrderId == work_order_id)

    def read_template_by_typename(self, typename):
        '''
        Method to read the template by unique typename from the datastore

        Returns: the dictionary representation of the template or None if the template does not exist
        '''
        with self.lock:
            Template = tinydb.Query()
            return self.templates.get(Template.workflow.typeName == typename)

    def write_job(self, cdm_job, overwrite_existing=True):
        """
        Called when a southbound job has been received. 
        1) Insert into the datastore if this is a new job
        2) If it is an existing job in the datastore, only overwrite it if the overwrite_existing is True

        Args:
            cdm_job: cdm job
        Returns: False if failed. 1=new job, 2=update existing job
        """
        with self.lock:
            rv = False
            Job = tinydb.Query()
            if cdm_job:
                work_order_id = cdm_job['workflow']['workOrderId']
                if not self.jobs.get(Job.workflow.workOrderId == work_order_id):
                    log.debug("data_store::write_job new job {}".format(work_order_id))
                    self.jobs.insert(cdm_job)
                    rv = 1
                elif overwrite_existing:
                    log.debug("data_store::write_job updating job {}".format(work_order_id))
                    self.delete_job_by_work_order_id(work_order_id)
                    self.jobs.insert(cdm_job)
                    rv = 2
                else:
                    log.debug("data_store::write_job job already exists {}".format(work_order_id))
            return rv

    def write_template(self, cdm_template, overwrite_existing=True):
        """
        Called when a southbound template has been received. 
        1) Insert into the datastore if this is a new template
        2) If it is an existing template in the datastore, only overwrite it if the overwrite_existing is True

        Args:
            cdm_template: cdm template
        """
        with self.lock:
            rv = False
            Job = tinydb.Query()
            if cdm_template:
                typename = cdm_template['workflow']['typeName']
                if not self.templates.get(Job.workflow.typeName == typename):
                    log.debug("data_store::write_template new template {}".format(typename))
                    self.templates.insert(cdm_template)
                    rv = True
                elif overwrite_existing:
                    log.debug("data_store::write_template updating template {}".format(typename))
                    self.delete_template_by_typename(typename)
                    self.templates.insert(cdm_template)
                    rv = True
                else:
                    log.debug("data_store::write_template template already exists {}".format(typename))
            return rv

    def delete_job_by_work_order_id(self, work_order_id):
        """
        Method to delete a job from the datastore

        """
        with self.lock:
            rv = False
            Job = tinydb.Query()
            if self.jobs.remove(Job.workflow.workOrderId == work_order_id):
                rv = True
            return rv

    def delete_template_by_typename(self, typename):
        """
        Method to delete a template from the datastore

        """
        rv = False
        if typename == default_template['workflow']['typeName']:
            return True
        with self.lock:
            Job = tinydb.Query()
            if self.templates.remove(Job.workflow.typeName == typename):
                rv = True
            return rv

    def read_active(self):
        '''
        Method to read the job that is currently active

        Returns:
            Dictionary containing the active job or None
        '''
        with self.lock:
            job = None
            job_eid = self.active.get(eid=1)
            if job_eid and "active_job" in job_eid and job_eid["active_job"] != None:
                job = self.jobs.get(eid=job_eid["active_job"])
            return job

    def set_active(self, work_order_id):
        '''
        Method to set the active job

        Args:
            work_order_id of the job to set active
        Returns:
            The job CDM if job was found, None otherwise
        '''
        with self.lock:
            Job = tinydb.Query()
            job = self.jobs.get(Job.workflow.workOrderId == work_order_id)
            if job:
                self.active.update({'active_job': job.eid}, eids=[1])
            else:
                log.warning("set_active failed to find job: {}".format(work_order_id))
            return job
        
    def deactivate_job(self):
        with self.lock:
            self.active.update({'active_job': None}, eids=[1])

    def read_active_job_item_index(self):
        '''
        Method to read the job item index that is currently active

        Args:
            active_job_eid (int): the element id of the active job or
                None

        Returns:
            Dictionary containing the active job item index or None
        '''
        with self.lock:
            return self.job_item_active_index.get(eid=1)

    def set_active_job_item_index(self, test_active_index):
        '''
        Method to set the active job item index

        Args:
            active_job_eid (string): the element id of the active job or
                None
            job_item_active_index (string): the index of the active job-item of the active job
        '''
        with self.lock:
            self.job_item_active_index.update({'active_job_item_index': test_active_index}, eids=[1])

    #Deprecated
    def read_job(self, element_id):
        '''
        Method to read the job from the datastore

        Returns: the dictionary representation of the job or None
            if the job does not exist
        '''
        with self.lock:
            return self.jobs.get(eid=element_id)
    #Deprecated
    def get_jobs_eids(self):
        """
        Method to return a list of all the element ids in the jobs table
        of the datastore

        Returns:
            (list): the list of element ids (int)
        """
        with self.lock:
            return [element.eid for element in self.jobs.all()]
    #Deprecated
    def read_jobs(self):
        '''
        Method to read the job from the datastore

        Returns: the dictionary representation of the job or None
            if the job does not exist
        '''
        with self.lock:
            return self.jobs.all()
    #Deprecated
    def delete_job(self):
        """
        Method to delete a job from the datastore

        Note: currently the implementation is hardcoded to delete job 1
        if it exists and do nothing otherwise
        """
        with self.lock:
            if self.jobs.contains(eids=[1]):
                self.jobs.remove(eids=[1])
