"""
Module that contains schemas for Viavi jobs in CDM2.1 format ('viaviJob')

"""

from marshmallow import Schema, fields, post_load, post_dump
#from marshmallow.validate import OneOf, Length
import json
import logging
import traceback

log = logging.getLogger(__name__)

'''
MyTolerantSchema works fine but a bit heavy. 
Calls to self.dumps inside post_load and self.loads inside post_dump generate a big activity
I keep it in case it might come in handy later (at least for inspiration)


class MyTolerantSchema(Schema):
    """
    Unkown Schema data is appended to the resulting output
    """
  
    @post_load(pass_many=True, pass_original=True)
    def add_additional_data(self, valid_data_param, many, original_data_param):

        valid_data = []
        original_data = []
        if (many == True):
            valid_data = valid_data_param
            original_data = original_data_param
        else:
            valid_data.append(valid_data_param)
            original_data.append(original_data_param)

        ignore_this_load = False
        try:
            if (original_data[0]['ignore_this_load'] == True):
                ignore_this_load = True
        except:
            pass

        if (ignore_this_load == False):
            # Dump back to original format
            # self.dumps will call @post_dump and @post_load will call @post_load
            valid_data[0]['ignore_this_dump'] = True
            tmp_json = self.dumps(valid_data, many = True).data
            data_mta_format = json.loads(tmp_json)
            del valid_data[0]['ignore_this_dump']

            # Now compare original data with converted data
            for valid_item, item, original in zip(valid_data, data_mta_format, original_data):
                additional_keys = set(original) - set(item)

                for key in additional_keys:
                    print('Load {0} Unknown key : {1}'.format(self.__class__.__name__, key))
                    log.debug('Load {0} Unknown key : {1}'.format(self.__class__.__name__, key))
                    # store unknown data
                    valid_item[key] = original[key]


        if (many == True):
            return valid_data
        else:
            return valid_data[0]

    @post_dump(pass_many=True, pass_original=True)
    def dump_additional_data(self, valid_data_param, many, original_data_param):

        valid_data = []
        original_data = []
        if (many == True):
            valid_data = valid_data_param
            original_data = original_data_param
        else:
            valid_data.append(valid_data_param)
            original_data.append(original_data_param)


        ignore_this_dump = False
        try:
            if (original_data[0]['ignore_this_dump'] == True):
                ignore_this_dump = True
        except:
            pass

        if (ignore_this_dump == False):
            # Load back to original format
            # self.load will call @post_load and @post_load will call @post_dump
            valid_data[0]['ignore_this_load'] = True
            tmp_json = self.load(valid_data, many = True).data
            data_mta_format = json.loads(json.dumps(tmp_json))

            del valid_data[0]['ignore_this_load']

            # Now compare original data with converted data
            for valid_item, item, original in zip(valid_data, data_mta_format, original_data):
                additional_keys = set(original) - set(item)

                for key in additional_keys:
                    print('Dump {0} Unknown key : {1}'.format(self.__class__.__name__, key))
                    log.debug('Dump {0} Unknown key : {1}'.format(self.__class__.__name__, key))
                    # store unknown data
                    valid_item[key] = original[key]


        if (many == True):
            return valid_data
        else:
            return valid_data[0]
'''

def make_cdm_schema(config_schema=None):
    """CDM Schema factory function to create a cdm schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """

    class CdmSchema2(Schema):
        """
        CDM Job Schema, contains all data related to a job
        """
        job_modified_on=fields.Str(required=False,
                                load_from="jobModifiedOn",
                                dump_to="jobModifiedOn",
                                description='string, Timestamp (ISO8601) updated when anything in the block changes')
        workflow = fields.Nested(
            'WorkflowSchema2',
            required=True,
            load_from='workflow',
            dump_to='workflow',
            description='dictionary, contains all related to workflow')

        tests = fields.Nested(
            make_cdm_test_schema(config_schema),
            required=False,
            load_from='tests',
            dump_to='tests',
            many=True,
            description='dictionary, contains all related to tests')

    class TopSchema(Schema):
        """
        Top-level Schema for the CDM Job
        """
        cdm_version = fields.Str(required=True,
                                 load_from='cdmVersion',
                                 dump_to='cdmVersion',
                                 description='Version')
        cdm = fields.Nested(
            CdmSchema2,
            required=True,
            load_from='cdm',
            dump_to='cdm',
            many=True,
            description='contains workflow, tests, etc.')


    return TopSchema


def empty_attributes():
    """
    Function returns an empty attributes dictionary
    Used to ease conversion if these fields are empty in the cdm structure
    """
    return {}

def empty_customer_info():
    """Function returns an empty customer info dict
    Used to ease conversion if these fields are empty in the cdm structure
    """
    return {
        "company": ""
    }

def empty_tech_info():
    """Function returns an empty tech info dict
    Used to ease conversion if these fields are empty in the cdm structure
    """
    return {
        "techId": ""
    }

def empty_domain_attributes():
    """Function returns an empty domain attributes dict
    Used to ease conversion if these fields are empty in the cdm structure
    """
    return {
        "logoInfo": {
            "path":""
        }
    }

def empty_job_manager_attributes():
    """Function returns an empty job manager attributes dictionary
    Used to ease conversion if these fields are empty in the cdm structure
    """
    return {
         "testLocation": {
             "label":"Location",
             "value":""
        }
    }

class WorkflowSchema2(Schema):
    """
    Workflow Schema for the CDM 2.0 Job (Fiber)
    """
    workflow_id=fields.Integer(
                            missing=0,
                            load_from="workflowId",
                            dump_to="workflowId",
                            description='integer, id of the workflow')
    workflow_type=fields.Str(missing="viaviJob",
                            load_from="type",
                            dump_to="type",
                            description='string, type of the work order')
    workorder_id=fields.Str(missing="",
                            load_from="workOrderId",
                            dump_to="workOrderId",
                            description='string, id of the work order')
    template=fields.Boolean(required=False,
                            load_from="template",
                            dump_to="template",
                            description='boolean, a description is needed')
    type_name=fields.Str(required=False,
                            load_from="typeName",
                            dump_to="typeName",
                            description='string, a description is needed')
    date=fields.Str(required=False,
                            load_from="date",
                            dump_to="date",
                            description='string, workflow creation date ISO-8601 format')
    workorder_label=fields.Str(required=False,
                            load_from="workOrderLabel",
                            dump_to="workOrderLabel",
                            description='string, label of the work order')
    tech_info=fields.Nested('TechInfoSchema',
                            missing=empty_tech_info,
                            load_from="techInfo",
                            dump_to="techInfo",
                            description="contains technician's description")

    customer_info=fields.Nested('CustomerInfoSchema',
                            missing=empty_customer_info,
                            load_from="customerInfo",
                            dump_to="customerInfo",
                            description="contains customer's description")

    contractor_id=fields.Str(missing="",
                            load_from="contractorId",
                            dump_to="contractorId",
                            description='contains contractor or sub-contractor Id')

    workorder_source=fields.Str(missing="",
                            load_from="workOrderSource",
                            dump_to="workOrderSource",
                            description='string, source of any customer-specific workOrderAttributes')

    workorder_attributes=fields.Dict(
                            ordered = True,
                            missing=empty_attributes,
                            values=fields.Nested('Schema'),
                            keys=fields.Str(),
                            load_from="workOrderAttributes",
                            dump_to="workOrderAttributes",
                            description='dictionary containing customer-specific set of key-value pairs')

    job_manager_attributes=fields.Dict(
                            required=False,
                            missing=empty_job_manager_attributes,
                            many=True,
                            values=fields.Nested('Schema'),
                            keys=fields.Str(),
                            load_from="jobManagerAttributes",
                            dump_to="jobManagerAttributes",
                            description='dictionary containing Job Manager attributes (as per CDM 2.1)')

    domain_attributes=fields.Dict(values=fields.Str(),
                            keys=fields.Str(),
                            missing=empty_domain_attributes,
                            load_from="domainAttributes",
                            dump_to="domainAttributes",
                            description='dictionary containing attributes for the domain')

    attributes=fields.Dict( values=fields.Nested('Schema'),
                            keys=fields.Str(),
                            missing=empty_attributes,
                            load_from="attributes",
                            dump_to="attributes",
                            description='dictionary containing key-value pairs for general workflow attributes')


class CdmTestLocAttribsSchema(Schema):
    """
    Schema for testLocation(s) of a test
    """
    ref_info=fields.Dict(
                   values=fields.Nested('Schema'),
                   keys=fields.Str(),
                   required=False,
                   load_from="referenceInfo",
                   dump_to="referenceInfo",
                   description='contains Job Manager referenceInfo for a test location')
    sub_type_info=fields.Dict(
                   values=fields.Nested('Schema'),
                   keys=fields.Str(),
                   required=False,
                   load_from="subTypeInfo",
                   dump_to="subTypeInfo",
                   description='contains Job Manager subTypeInfo for a test location')


class CdmTestLocationsSchema(Schema):
    """
    Schema for testLocation(s) of a test
    """
    workflow_id=fields.Integer(required=False,
                            default=0,
                            load_from="workflowId",
                            dump_to="workflowId",
                            description='integer, id of the workflow')
    label=fields.Str(required=False,
                   load_from="label",
                   dump_to="label",
                   description="label name")
    attributes=fields.Nested( CdmTestLocAttribsSchema,
                   required=False,
                   load_from="attributes",
                   dump_to="attributes",
                   description='contains attributes for a test location')

class TechInfoSchema(Schema):
    """
    Schema for the technician's information
    """
    tech_id = fields.Str(
        required=True,
        load_from="techId",
        dump_to="techId",
        description="the technician's identification number"
    )

class CustomerInfoSchema(Schema):
    """
    Schema for the customer's information
    """
    company = fields.Str(
        required=True,
        load_from="company",
        dump_to="company",
        description="the customer's company name"
    )

def make_cdm_test_schema(config_schema):
    """CDM Test Schema factory function to create a cdm test schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    def make_configuration(config_schema):
        if not config_schema:
            return fields.Dict(allow_none=True)

        return fields.Nested(
            config_schema,
            allow_none=True,
            required=False,
            load_from='configuration',
            dump_to='configuration',
            description='the configuration of the particular test to be launched'
        )

    class CdmTestSchema2(Schema):
        """
        Schema for testing CDM 2.0 Job Test entries
        """
        type=fields.Str(required=True,
                       load_from="type",
                       dump_to="type",
                       description="type of the test")
        label=fields.Str(required=False,
                       load_from="label",
                       dump_to="label",
                       description="label name")
        workflow_id=fields.Integer(required=False,
                                default=0,
                                load_from="workflowId",
                                dump_to="workflowId",
                                description='integer, id of the workflow')
        procedures=fields.Str(required=False,
                       default="",
                       load_from="procedures",
                       dump_to="procedures",
                       description="pathname for the procedures file")
        configuration = make_configuration(config_schema)
        test_locations=fields.Nested(
                       CdmTestLocationsSchema,
                       keys=fields.Str(),
                       allow_none=True,
                       required=False,
                       many=True,
                       load_from="testLocations",
                       dump_to="testLocations",
                       description='testLocations for a test')

    return CdmTestSchema2


def apply_corrected_cdm_values_to_original_cdm(key, original_param, corrected_param):
    try:
        original = None
        corrected = None

        try:
            corrected = corrected_param[key]
        except:
            #print("cdm_corrected_param knows nothing about the key '{}' : Ignored. original remains unchanged".format(key))
            log.debug("cdm_corrected_param knows nothing about the key '{}' : Ignored. original remains unchanged".format(key))
            return

        try:
            original = original_param[key]
        except:
            if (corrected != None):
                original_param[key] = corrected
                print("cdm_original_param knows nothing about the key '{}'. Added...".format(key))
                log.debug("cdm_original_param knows nothing about the key '{}'. Added...".format(key))
            return

        
        
        try:
            #original and corrected can be either lists or single items. If not lists we convert them to lists to have a common process
            original_list = []
            corrected_list = []

            if (isinstance(original, list) == True):
                original_list = original
                corrected_list = corrected
            else:
                original_list.append(original)
                corrected_list.append(corrected)

            for index, original_item in enumerate(original_list):
                corrected_item = corrected_list[index]
                # we need 'index' because original_item will be immutable if it is a single item
                try:
                    if (isinstance(original_item, dict) == True):
                        # It's a dictionary. We call ourself for each subkey of the 2 sources
                        key_list = set(original_item).union(set(corrected_item))
                        for original_key in key_list:
                            apply_corrected_cdm_values_to_original_cdm(original_key, original_item, corrected_item)
                    else:
                        # A single item. Corrected value takes precedence
                        if (original_item != corrected_item):
                            #print('key = {}, original = {}\ncorrected = {}\n'.format(key, original_item, corrected_item))
                            original_list[index] = corrected_item

                except:
                    #print(traceback.format_exc())
                    #print('Unexpected ERROR on key = {}, original = {}\ncorrected = {}. original item remains unchanged'.format(key, original_item, corrected_item))
                    log.debug(traceback.format_exc())
                    log.debug('Unexpected ERROR on key = {}, original = {}\ncorrected = {}. original item remains unchanged'.format(key, original_item, corrected_item))

            if (isinstance(original, list) == True):
                original_param[key] = original_list
            else:
                original_param[key] = original_list[0]


        except:
            #print(traceback.format_exc())
            #print("Unexpected ERROR on key {}".format(key))
            log.debug(traceback.format_exc())
            log.debug("Unexpected ERROR on key {}".format(key))

    except:
        #print(traceback.format_exc())
        #print("Unexpected ERROR on key {}".format(key))
        log.debug(traceback.format_exc())
        log.debug("Unexpected ERROR on key {}".format(key))
