import marshmallow
from marshmallow import Schema, fields, post_load, pre_dump
from marshmallow.validate import OneOf, Length
from dateutil.parser import parse
from marshmallow_oneofschema import OneOfSchema
import json
import hashlib
import decimal

UI_STRING_VALIDATOR = Length(max=60)
CURRENT_CDM_VERSION = '2.2'
class CdmWorkflow(Schema):
    """
    Schema for the top level JSON
    """
    cdmVersion = fields.Str(required=False)
    cdm = fields.Nested('CdmJob', required=False, many=True)

class CdmJob(Schema):
    """
    Schema for the top level CDM job
    """    
    cdmVersion = fields.Str(required=False) #Required in northbound only    
    tests = fields.Nested('CdmTestsSchema', required=False, many=True)
    workflow = fields.Nested('CdmWorkflowSchema', required=False) 
    assetInfo = fields.Nested('CdmAssetInfoSchema', required=False)
    configuration = fields.Dict(required=False, description='instrument specific configuration') 
    jobModifiedOn = fields.Str(required=False, description='string, Timestamp (ISO8601) updated when anything in the block changes')    
    checkSum = fields.Str(required=False)
class CdmWorkflowSchema(Schema):
    """
    Schema for the workflow
    """
    workflowId = fields.Integer(required=False)
    workOrderId = fields.Str(required=False)
    workOrderLabel = fields.Str(required=False)
    type = fields.Str(required=False, missing='viaviJob')
    template = fields.Boolean(required=False)
    typeName = fields.Str(required=False,description='aka templateName')
    templateVersion = fields.Str(required=False)
    date = fields.Str(required=False)
    dueDate = fields.Str(required=False)
    state = fields.Str(required=False)
    techInfo = fields.Nested('CdmTechInfoSchema', required=False)
    customerInfo = fields.Nested('CdmCustomerInfoSchema', required=False)
    workOrderAttributes = fields.Dict(required=False,description='dictionary containing customer-specific set of key-value pairs')
    attributes = fields.Dict(required=False, description='dictionary containing key-value pairs for general workflow attributes')
    domainAttributes = fields.Dict(required=False)
    contractorId = fields.Str(required=False)
    workorderSource = fields.Str(required=False)
    jobManagerAttributes = fields.Dict(required=False, values=fields.Nested('CdmAttributeSchema'), keys = fields.Str(), 
                                    description='dictionary containing Job Manager attributes (as per CDM 2.1)')
    lockedAttributes = fields.List(fields.Str(), required=False)
    jobStatus = fields.Str(required=False)
    openEndedStatus = fields.Str(required=False, description='open ended job status')
    jobManagement = fields.Dict(required=False)

class CdmAssetInfoSchema(Schema):
    """
    Schema for the asset information
    """
    assetType = fields.Str(required=True)
    uniqueId = fields.Str(required=True)
    manufacturer = fields.Str(required=False)
    model = fields.Str(required=False)
    swVersion = fields.Str(required=False)
    hwVersion = fields.Str(required=False)
    calibrationDate = fields.Str(required=False, description='string, calibration date ISO-8601 format')
    modulesInfo = fields.Nested('CdmModulesInfoSchema', required=False, many=True)
    swOptions = fields.Nested('CdmSwOptionsSchema', required=False, many=True)
    hwOptions = fields.List(fields.Str(),required=False)
    instrumentRemote = fields.Nested('CdmInstrumentRemoteSchema', required=False, many=True)
    techId = fields.Str(required=False)
    macAddress = fields.Str(required=False)
    mfgDate = fields.Str(required=False)
    batteryType = fields.Str(required=False)
    batteryModel = fields.Str(required=False)
    batteryDate = fields.Str(required=False)
    firmwareInfo = fields.Nested('CdmFirmwareInfoSchema', required=False, many=True)
    hardwareInfo = fields.Nested('CdmHardwareInfoSchema', required=False, many=True)

class CdmFirmwareInfoSchema(Schema):
    """
    Schema for the firmware information
    """
    name = fields.Str(required=False)
    description = fields.Str(required=False)
    model = fields.Str(required=False)
    version = fields.Str(required=False)

class CdmHardwareInfoSchema(Schema):
    """
    Schema for the hardware information
    """
    name = fields.Str(required=False)
    description = fields.Str(required=False)
    model = fields.Str(required=False)
    version = fields.Str(required=False)

class CdmInstrumentRemoteSchema(Schema):
    """
    Schema for the remote instrument
    """
    assetType = fields.Str(required=True)
    uniqueId = fields.Str(required=True)
    manufacturer = fields.Str(required=False)
    model = fields.Str(required=False)
    swVersion = fields.Str(required=False)
    hwVersion = fields.Str(required=False)
    modulesInfo = fields.Nested('CdmModulesInfoSchema', required=False, many=True)
    swOptions = fields.Nested('CdmSwOptionsSchema', required=False, many=True)
    hwOptions = fields.List(fields.Str(),required=False)
    calibrationDate = fields.Str(required=False, description='string, calibration date ISO-8601 format')


class CdmModulesInfoSchema(Schema):
    """
    Schema for the module information
    """
    assetType = fields.Str(required=True)
    uniqueId = fields.Str(required=True)
    model = fields.Str(required=False)
    swVersion = fields.Str(required=False)
    hwVersion = fields.Str(required=False)
    swOptions = fields.Nested('CdmSwOptionsSchema', required=False, many=True)
    hwOptions = fields.List(fields.Str(),required=False)
    calibrationDate = fields.Str(required=False, description='string, calibration date ISO-8601 format')

class CdmSwOptionsSchema(Schema):
    """
    Schema for the software options
    """
    name = fields.Str(required=True)
    description = fields.Str(required=False)
    optionLicenseType = fields.Str(required=True)
    catalogNumber = fields.Str(required=False)
    expirationDate = fields.Str(required=False, description='string, expiration date ISO-8601 format')

class CdmWoAttribHeader:
    def __init__(self, cdmWoAttribHeader):
        self.cdmWoAttribHeader = cdmWoAttribHeader

class CdmAttribute:
    def __init__(self, cdmAttribute):
        self.cdmAttribute = cdmAttribute

class CdmWoAttribHeaderSchema(Schema):
    workOrderAttribHeaders = fields.List(fields.Str(),required=False)

    @marshmallow.post_load
    def make_WoAttribHeader(self, data, **kwargs):
        return CdmWoAttribHeader(**data)

class CdmAttributeSchema(Schema):
    """
    Schema for the attribute information
    """
    label = fields.Str(required=True)
    value = fields.Str(required=True)
    valueType = fields.Str(required=False)
    editable = fields.Str(required=False)
    regExp = fields.Str(required=False)
    validValues = fields.List(fields.Str(),required=False)
    orderIndex = fields.Integer(required=False)
    visible = fields.Boolean(required=False)
    refPath = fields.Str(required=False)
    dependentAttribute = fields.Dict(required=False) 

    @marshmallow.post_load
    def make_WoAttribHeader(self, data, **kwargs):
        return CdmAttribute(**data)

class CdmWoAttribsSchema(OneOfSchema):
    type_schemas = {"workOrderAttribHeaders": CdmWoAttribHeaderSchema, "Attribute": CdmAttributeSchema}

    def get_obj_type(self, obj):
        if isinstance(obj, CdmWoAttribHeader):
            return "workOrderAttribHeaders"
        elif isinstance(obj, CdmAttribute):
            return "Attribute"
        else:
            raise Exception("Unknown object type: {}".format(obj.__class__.__name__))

class CdmTechInfoSchema(Schema):
    """
    Schema for the technician's information
    """
    techId = fields.Str(required=False)
    firstName = fields.Str(required=False)
    lastName = fields.Str(required=False)

class CdmCustomerInfoSchema(Schema):
    """
    Schema for the customer information
    """
    firstName = fields.Str(required=False)
    lastName = fields.Str(required=False)
    company = fields.Str(required=False)
    phone = fields.Str(required=False)
    email = fields.Str(required=False)
    streetAddress1 = fields.Str(required=False)
    streetAddress2 = fields.Str(required=False)
    city = fields.Str(required=False)
    state = fields.Str(required=False)
    postalCode = fields.Str(required=False)

class CdmTestsSchema(Schema):
    """
    Schema for the test information
    """
    type = fields.Str(required=True, description='type of the test (only SS registered values)')
    label = fields.Str(required=False)
    description = fields.Str(required=False)
    workflowId = fields.Integer(required=False)
    configuration = fields.Dict(required=False, description='test specific configuration') 
    configAttributes = fields.Dict(required=False) 
    results = fields.Nested('CdmResultsSchema', required=False)
    attributes = fields.Dict(required=False,description='dictionary containing key-value pairs for general test attributes and subTypeInfo and referenceInfo')
    testLocations = fields.Nested('CdmTestsLocationsSchema', required=False, many=True)
    deployableTo = fields.Nested('CdmDeployableToSchema', required=False, many=True)
    schemaVersion = fields.Str(required=False)
    subTests = fields.List(fields.Str(),required=False)
    subTestIndex = fields.Integer(required=False)
    allTestsSubmitted = fields.Boolean(required=False)
    testBlockChanged = fields.Boolean(required=False)
    procedures = fields.Str(required=False)
    proceduresUrl = fields.Str(required=False)
    required = fields.Boolean(required=False)
    testPlanIndex = fields.Integer(required=False)
    openEndedStatus = fields.Str(required=False, description='open ended test status')
    #For internal IJM use only. The new add_test_data API adds the additional reports to this field. 
    ijm_filePath = fields.List(fields.String(), required=False)
    index = fields.Str(required=False, description='used for open ended tests')

class CdmTestsLocationsSchema(Schema):
    """
    Schema for the test location information
    """
    workflowId = fields.Integer(required=False)
    label = fields.Str(required=True)
    attributes = fields.Dict(required=False,description='dictionary containing key-value pairs for general test attributes and subTypeInfo and referenceInfo')
    required = fields.Boolean(required=False)
    testPlanIndex=fields.Integer(required=False)
    #Since IJM stores results at locations, we need this here even though it is not valid CDM
    results = fields.Nested('CdmResultsSchema', required=False)

class CdmDeployableToSchema(Schema):
    """
    Schema for the deployable to information
    """
    assetType = fields.Str(required=True)
    model = fields.Str(required=False)

class CdmResultsSchema(Schema):
    """
    Schema for the results
    """
    status = fields.Str(required=True)
    testTime = fields.Str(required=False, description='string, test time ISO-8601 format')
    testDurationMs = fields.Integer(required=False)
    comment = fields.Str(required=False)
    geoLocation = fields.Nested('CdmGeoLocationSchema', required=False) 
    data = fields.Dict(required=False)

class CdmGeoLocationSchema(Schema):
    """
    Schema for the geographic location of the test
    """
    latitude = fields.Float(required=True)
    longitude = fields.Float(required=True)

class ReferenceInfoSchema(Schema):
    """
    Schema for reference information for the test
    each test may have zero or more reference information
    entries to differentiate it from other tests of the same type

    """
    key = fields.Str(required=True,
                     load_from='key',
                     dump_to='key',
                     validate=UI_STRING_VALIDATOR,
                     description='the field name for this reference')
    value = fields.Str(
        required=True,
        load_from='value',
        dump_to='value',
        validate=UI_STRING_VALIDATOR,
        description='the entered value for this particular field')

class ReferenceInfoForTestTypeSchema(Schema):
    """
    Schema for reference information for a test with given
    test_type to differentiate it from other tests of the same type

    """
    reference_info=fields.Nested(
        'ReferenceInfoSchema',
        required=True,
        many=True,
        load_from="referenceInfo",
        dump_to="referenceInfo",
        description=(
            'reference info differentiating this test plan'
            'from others like it'))

    test_type = fields.Str(
        required=True,
        load_from='testType',
        dump_to='testType',
        validate=UI_STRING_VALIDATOR,
        description='the test type the reference belongs to')

#Add the checksum to the cdm_job. Input is a cdm_job dictionary. Output is a json object with checksum.
CDM_CHECKSUM_SALT = "UzkwV1dXR0FKVjE4U1RFVmlhdmlSZXN1bHRz"
def addCheckSumToCdmJob(cdm_job):        
    enc = hashlib.md5()
    #checksum calculation requires compact json with no whitespace
    jcdm_job = json.dumps(cdm_job, indent=None, sort_keys=False, separators=(',', ':'))
    jcdm_job += CDM_CHECKSUM_SALT
    jcdm_job = jcdm_job.encode('utf-8')
    enc.update(jcdm_job)
    cs = enc.hexdigest()
    cdm_job['checkSum'] = cs
    jcdm_job = json.dumps(cdm_job, indent=None, sort_keys=False, separators=(',', ':'))
    return jcdm_job

class CdmJobWithCsvImport(CdmJob):
    """
    Schema for adding a template from CDM job
    """    
    csvImportFilepath = fields.Str(required=False)

class CreateJobSchema(Schema):
    workOrderId = fields.Str(required=True)
    techId = fields.Str(required=False)
    templateName = fields.Str(required=False)
    csvImportFilepath = fields.Str(required=False)

class CdmWorkOrderListWorkOrderInfo(Schema):
    workOrderId = fields.Str(required=True)
    jobManagement = fields.Dict(required=False)
    modifiedTime = fields.Str(required=False, description='string, Timestamp (ISO8601) updated when anything in the block changes')    

class CdmWorkOrderList(Schema):
    """
    Schema for the top level CDM job
    """    
    workOrderInfo = fields.Nested('CdmWorkOrderListWorkOrderInfo', required=False, many=True)
    archivedIds = fields.List(fields.Str(),required=False)

class ManualStepSchema(Schema):
    """
    Schema for the manual step response
    """
    userResponse = fields.Str(required=True)
    comment = fields.Str(required=False)