"""
Module containing functions for user files rest api
"""
import os
import json
from rest_api.api import job_manager
import xml.etree.ElementTree as ET
try:
    from urllib import quote, unquote  # Python 2.X
except ImportError:
    from urllib.parse import quote, unquote  # Python 3+

from webargs.bottleparser import use_kwargs

from bottle import request, response, static_file, Bottle
from .schemas import UploadArgs
from rest_api.api.job_manager.plugin import JobManagerPlugin
import traceback
import logging

log = logging.getLogger(__name__)

api_node = Bottle() # pylint: disable=C0103

__api_version__ = 2
__api_name__ = "filemgr"


METADATA_KEY = 'metadata'


# Local functions


def __filepath_to_dict(filepath, base_dir):
    """Function to create a dictionary representing the file path
    the swagger schema is below
    ---
    type:
      enum:
        - F
      type: string
      description: the type of this object for files this will be F
    name:
      type: string
      description: the name of the file
    path:
      type: string
      description: the relative path to the file from the base directory
    mtime:
      type: string
      description: time of most recent content modification fo rthis file
    size:
      type: number
      description: 'the size of the file, in bytes'
    """

    res = {'type': 'MISC.File'}
    fstats = os.stat(filepath)
    file_name = os.path.basename(filepath)
    file_path_rel = os.path.relpath(filepath, base_dir)
    res['name'] = file_name
    res['modifiedTime'] = fstats.st_mtime
    res['size'] = fstats.st_size
    res['fileUri'] = file_path_rel
    return res


def __get_dir_json(path, base_dir):
    """Function to create a dictionary representing the directory
    the swagger schema is below
    ---
    type:
      enum:
        - D
      type: string
      description: >
                      the type response for this object for directories this
                      will be D
    name:
      type: string
      description: The name of the directory on the file system
    fileUri:
      type: string
      description: the relative path to the directory from the base directory
    files:
        items:
          $ref: '#/definitions/UserFile'
        type: array
        description: list of UseFile objects in this directory
    subDirs:
      items:
        $ref: '#/definitions/UserDirectory'
      type: array
      description: list of UserDirectory for each subdirectory
    """
    res = {'type':'MISC.Directory'}
    dir_path_rel = os.path.relpath(path, base_dir)
    append_base = lambda c: os.path.join(path, c)

    content_paths_abs_file = map(append_base, os.listdir(path))
    content_paths_abs_dir = map(append_base, os.listdir(path))

    product_specific = api_node.config['rest_api.product_specific']
    res['name'] = product_specific.get_dirname(path, base_dir)
    res['fileUri'] = dir_path_rel

    #Directory
    try:
        dirs = filter(os.path.isdir, content_paths_abs_dir)
        res['subDirs'] = [__get_dir_json(d, base_dir) for d in  dirs]
    except:
        res['subDirs'] = []

    #File
    try:
        files = filter(os.path.isfile, content_paths_abs_file)
        res['files'] = [__filepath_to_dict(f, base_dir) for f in  files]
    except:
        res['files'] = []

    return res

def strip_basepath(file_path, base_path):
    """Strip any directories matching the base directories from the file path

    This funciton is so that if the user supplies an absolute path that matches
    the base path the api will return the file
    """
    file_path_dirs = file_path.split('/')
    base_path_dirs = base_path.split('/')

    for dir_name in base_path_dirs:
        if dir_name in file_path_dirs and file_path_dirs.index(dir_name) == 0:
            file_path_dirs.remove(dir_name)

    return '/'.join(file_path_dirs)

# Clean API

@api_node.get('/api/filemgr/v2/userfiles')
def file_list():
    """Endpoint to list the files and directories of the base directory
    ---
    get:
      tags:
        - user files
      description: Endpoint to list the files and directories of the base directory
      responses:
        200:
          description: success
          schema: {'$ref': '#/definitions/UserDirectory'}
    """
    base_dir = request.app.config.get('user_files.filemgr_dir') #pylint: disable=no-member

    return __get_dir_json(base_dir, base_dir)


@api_node.get('/api/filemgr/v2/userfiles/<file_path:path>')
def filemgr_get_files(file_path):
    """
    Endpoint to either list the files and directories in the file_path or to
    return the file to the client
    ---
    get:
      tags:
        - user files
      description: >
                     Endpoint to either list the files and directories in the
                     file_path or to return the file to the client
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
        400:
          description: file or directory not found
    """
    base_dir = request.app.config.get('user_files.filemgr_dir')

    unquoted_file_name = unquote(unquote(file_path))
    rel_path = strip_basepath(unquoted_file_name, base_dir)
    full_path = os.path.join(base_dir, unquoted_file_name)

    if os.path.isfile(full_path):
        static_file_response = static_file(unquoted_file_name, root=base_dir)
        return static_file_response

    response.status = 400
    return {}

@api_node.delete('/api/filemgr/v2/userfiles/<file_path:path>')
def delete_file(file_path):
    """Endpoint for deleting a file from the instrument
    ---
    delete:
      tags:
        - user files
      description: Endpoint for deleting a file from the instrument
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
        404:
          description: invalid file path
        500:
          description: could not delete file
    """
    base_dir = request.app.config.get('user_files.filemgr_dir') #pylint: disable=no-member

    full_path = os.path.join(base_dir, file_path)
    if os.path.isfile(full_path):
        os.remove(full_path)
        response.status = 200
    else:
        response.status = 404

@api_node.post('/api/filemgr/v2/userfiles')
def filemgr_save_file():
    """Endpoint for uploading a file to the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload a file to the instrument
      consumes:
        - multipart/form-data
      parameters:
        - in: formData
          schema: UploadArgs
      responses:
        200:
          description: success
    """
    fileUri = '' #  'except:' needs fileUri
    try:
        metadata = request.files.get('metadata')
        log.error('## filemgr_save_file: request: {}'.format(request))
        log.error('## filemgr_save_file: request metadata: {}'.format(metadata))
        metadata_contents = metadata.file.read()
        if not isinstance(metadata_contents, str):
            metadata_contents = metadata_contents.decode('utf-8')
        metadata_data = json.loads(metadata_contents)

        filename = metadata_data['name']
        fileUri = metadata_data['fileUri']
        log.error('## filemgr_save_file: fileUri = %s, name = %s', fileUri, filename )

        listAllowedDir = []

        listAllowedDir.append(api_node.config.get('user_files.filemgr_dir'))

        bHappy = False

        for allowedDir in listAllowedDir:
            if fileUri.startswith(allowedDir):
                bHappy = True
                break

        if bHappy == False:
            while (fileUri.startswith('/')):
                # os.path.join will stop joining paths when it comes accross an absolute path
                fileUri = fileUri[1:]
            fileUri = os.path.join(listAllowedDir[0], fileUri)


        # no need to save the file if the path doesn't exist
        pathOnly = os.path.dirname(fileUri)

        if os.path.isdir(pathOnly):

            filedata = request.files.get('filedata')

            filedata.save(fileUri, overwrite=True)

            response.status = 200
        else:
            log.debug('## filemgr_save_file 2 : failed for %s', fileUri )
            response.status = 400

    except:
        log.debug((traceback.format_exc()))
        if fileUri:
          log.debug('## filemgr_save_file 3 : failed for %s', fileUri )
        response.status = 400

    return {}

@api_node.get('/api/filemgr/v2/metadata/<file_path:path>')
def filemgr_get_metadata(file_path):
    """Endpoint for getting the metadata for a file from the instrument

    See doc at https://conf1.ds.jdsu.net/wiki/display/MOBILE/Mobile+Tech+REST+API

    ---
    post:
      tags:
        - user files
      description: Endpoint for getting the metadata for a file from the instrument
      parameters:
        - in: path
          name: file_path
          type: string
    """

    unquoted_file_name = unquote(file_path)

    base_dir = request.app.config.get('user_files.filemgr_dir')

    product_specific = api_node.config['rest_api.product_specific']

    # filemgr should get the metadata from the same place that syncfiles obtains it
    res = product_specific.get_sync_file_info( unquoted_file_name, base_dir )

    return res
