import struct
import time
import sys
import threading
import queue
import os
import select
import evdev as ev
from time import sleep

DEBUG = False

def get_event_device_by_name(name):
    for f in os.listdir("/sys/class/input"):
        if f.startswith("event"):
            with open("/sys/class/input/%s/device/name"%f) as n:
                real = n.readline().strip()
                if real in name:
                    return (real, "/dev/input/%s"%f)

def re_cal(raw, calmin, calmax, wantmin, wantmax):
    if raw < calmin:
        return wantmin
    if raw > calmax:
        return wantmax
    val = int(((raw - calmin) * wantmax) / (calmax - calmin))
    if val > wantmax:
        return wantmax
    if val < wantmin:
        return wantmin
    return val

class InputDevice(threading.Thread):
    '''
    Class to directly event from an input device (keyborad, mouse) like evdev
    '''

    def __init__(self, path):
        (self.__name, self.__path) = get_event_device_by_name(path)
        self.__calibration = None
        if os.path.isfile("/etc//X11/xorg.conf.d/99-calibration.conf"):
            with open("/etc/X11/xorg.conf.d/99-calibration.conf", "r") as conf:
                match = False
                for line in conf:
                    if line.startswith("#"):
                        continue
                    item = line.strip().split()
                    if len(item) < 2:
                        continue
                    if item[0] == "Section":
                        match = False
                    if item[0] == "MatchProduct":
                        if item[1].strip('"') == self.__name:
                            match = True
                            print("Find Calibration")
                    if match and item[0] == "Option" and item[1].strip('"') == "Calibration":
                        self.__calibration = [int(i) for i in " ".join(item[2:]).strip('"').split()]
                        print("Use touch screen calibration: %s"%self.__calibration)
        self.__exit = False
        self.__stack = []
        self.__depth = 0
        self.__timeout = None
        self.__queue = queue.Queue(10)
        self.__posx = 0
        self.__posy = 0
        self.__sync = False
        self.__pos = None
        self.__absinfo = {}
        threading.Thread.__init__(self)

    def run(self):
        dev = ev.InputDevice(self.__path)
        capa = dev.capabilities()
        if 3 in capa.keys():
            for caps in capa[3]:
                self.__absinfo[caps[0]] = caps[1]
        while not self.__exit:
            r, w, es = select.select([dev.fileno()], [], [], 1)
            if not dev.fileno() in r:
                if self.__timeout is not None:
                    self.__timeout -= 1
                    if self.__timeout == 0:
                        self.__queue.put((None,))
                continue
            event = dev.read_one()
            if(DEBUG): print("event --> %s "%(event), end='')
            if self.__exit:
                break
            e = ev.categorize(event)
            if event.type == ev.ecodes.EV_KEY:
                if(DEBUG): print("EV_KEY")
                if event.value == 1:
                    self.__stack.append(event.code)
                    self.__depth += 1
                elif event.value == 0:
                    self.__depth -= 1
                    if self.__depth == 0:
                        if self.__pos is not None:
                            self.__stack.append(self.__pos)
                            self.__queue.put(tuple(self.__stack))
                            self.__stack.clear()
                            self.__pos = None
                        else:
                            self.__queue.put(tuple(self.__stack))
                            self.__stack.clear()
            elif event.type == ev.ecodes.EV_ABS:
                if(DEBUG): print("EV_ABS")
                if type(e) == ev.events.AbsEvent:
                    if e.event.code in (ev.ecodes.ABS_X, ev.ecodes.ABS_MT_POSITION_X):
                        if self.__calibration is not None:
                            self.__posx = re_cal(e.event.value, self.__calibration[0], self.__calibration[1], 0, 800)
                        else:
                            self.__posx = int(( e.event.value * 800 ) / self.__absinfo[e.event.code].max)
                    elif e.event.code in (ev.ecodes.ABS_Y, ev.ecodes.ABS_MT_POSITION_Y):
                        if self.__calibration is not None:
                            self.__posy = re_cal(e.event.value, self.__calibration[2], self.__calibration[3], 0, 480)
                        else:
                            self.__posy = int(( e.event.value * 480 ) / self.__absinfo[e.event.code].max)
                else:
                    print("OTHER EVENT")
            elif event.type == ev.ecodes.EV_SYN:
                if(DEBUG): print("EV_SYN (%i %i)"%(self.__posx, self.__posy))
                self.__pos = (self.__posx, self.__posy)
                if e.event.code != ev.ecodes.SYN_MT_REPORT:
                    continue
                if self.__posx + self.__posy == 0:
                    self.__sync = False
                    # Release
                else:
                    if not self.__sync:
                        self.__queue.put( (ev.ecodes.BTN_TOUCH, (self.__pos)) )
                        self.__sync = True
                        if(DEBUG): print("TOUCH %i %i"%(self.__pos[0],self.__pos[1]))
                    self.__posx = 0
                    self.__posy = 0

                self.__queue.put( (ev.ecodes.KEY_MOVE, (self.__pos)) )
            else:
                print("OTHER EVENT")
        dev.close()

    def quit(self):
        self.__exit = True

    def getkey(self, purge=False, timeout=None):
        k = None
        if purge:
            while not self.__queue.empty():
                self.__queue.get()
        self.__timeout = timeout
        while k is None and not self.__exit:
            try:
                k = self.__queue.get(timeout=1)
                if len(k) > 0 and k[0] == ev.ecodes.KEY_MOVE:
                    k = None
            except queue.Empty:
                pass
        return k
