import QtQuick 2.5
import QtQuick.Controls 2.0
import elf_plugin 1.0
import graph_plugin 1.0
import formatters_plugin 1.0

import "ViaviStyle"

FocusScope {
    id: rootId

    property real zoomXValue:1
    property real zoomYValue:1

    property int keyPressed: 0
    readonly property int axisZ: 100
    readonly property int graphZ: 150
    readonly property int integralZ: 140
    readonly property int segmentZ: 160
    readonly property int timerInterval: 100
    readonly property int keyCursorDisplacement: 1
    readonly property int keyCursorAcceleratedDisplacement: 5

    readonly property int nrOfVerticalTicks : 10

    readonly property var verticalAxisDashPattern : [3.0, 3 * ViaviStyle.layouts.grapLineWidth]
    readonly property var horizontalAxisDashPattern: [5.0, 2 * ViaviStyle.layouts.grapLineWidth]

    readonly property string cursorALabel: "A"
    readonly property string cursorAdditionalALabel: "a"
    readonly property string cursorBLabel: "B"
    readonly property string cursorAdditionalBLabel: "b"
    readonly property string cursorAdditionalCLabel: "C"

    property bool backgroundVisible : false

    property bool osaAxisVisible: status_res_osaGridLinesEnabled.value && osaGridLines_res_specific.value
    property bool xTicksVisible: !status_res_osaGridLinesEnabled.value
                                 || ( status_res_osaGridLinesEnabled.value && osaGridLines_res_visible.value
                                     && !osaGridLines_res_specific.value )
    property bool yTicksVisible: !status_res_osaGridLinesEnabled.value
                                 || ( status_res_osaGridLinesEnabled.value && osaGridLines_res_visible.value )

    property bool zoomUpdateInProgress: false

    property bool additionalCursors: status_res_analysisEnabled.value
                                     && analysis_cfg_measType.enums.emtLoss === analysis_cfg_measType.value
                                     && analysis_cfg_selectedMethod.enums.esm5PtLoss === analysis_cfg_selectedMethod.value

    readonly property int additionalCursorsInitialOffset: ViaviStyle.layouts.buttonHeight

    property bool additionalCursorALock: false
    property bool additionalCursorBLock: false

    property alias cursorDragA: cursorDragA
    property alias cursorDragB: cursorDragB
    property alias cursorDragAdditionalA: cursorDragAdditionalA
    property alias cursorDragAdditionalB: cursorDragAdditionalB

    property alias oXAxisVisibleLeft: xAxis.minVisible
    property alias oXAxisVisibleRight: xAxis.maxVisible

    property alias graphCount: graphRepeaterId.model
    property alias tailingGraphCount: graphTailingRepeaterId.model

    signal keysPressAndHold()
    signal zoomLevelChanged(real xScale, real yScale, real wScale, real hScale)

    property CursorDragFiber lastUsedCursor

    function updateLastUsedCursorFromBackend(){
        if(!additionalCursors){
            if(trace_cfg_cursorSelected.value === trace_cfg_cursorSelected.enums.ecCursor_B){
                lastUsedCursor = cursorDragB
            }else{
                lastUsedCursor = cursorDragA
            }
        }
    }

    property bool initialised: false
    Connections{
        target: trace_cfg_cursorSelected

        onValueChanged: updateLastUsedCursorFromBackend()
    }

    onLastUsedCursorChanged: {
        if(!additionalCursors
                && trace_cfg_cursorSelected.value !== trace_cfg_cursorSelected.enums.ecCursor_AB
                && initialised){
            trace_cfg_cursorSelected.value = lastUsedCursor.backendId
        }

        if (additionalCursors) {
            updateCurrentCursors()
        }
    }

    onAdditionalCursorsChanged: {
        updateLastUsedCursorFromBackend()

        trace_cfg_additionalCursors.value = additionalCursors

        if (additionalCursors) {
            updateCurrentCursors()
        }
    }

    Component.onCompleted: {
        updateLastUsedCursorFromBackend()
        initialised = true
    }

    Keys.onUpPressed: {
        if(rootId.visible){
            zoomButtonsLayout.zoomBy(1, zoomButtonsLayout.zoomInFactor)
        }
    }
    Keys.onDownPressed: {
        zoomButtonsLayout.zoomBy(1, 1/zoomButtonsLayout.zoomInFactor)
    }

    Timer {
        id: pressAndHoldTimer
        interval: timerInterval
        running: false
        repeat: true

        onTriggered: {
            rootId.keysPressAndHold();
        }
    }

    onKeysPressAndHold: {
        if(rootId.visible && pressAndHoldTimer.running){
            if(rootId.keyPressed === Qt.Key_Left){
                trace_act_updateCursorsPosition.invokeWithStringArg(-keyCursorAcceleratedDisplacement)
            }

            if(rootId.keyPressed === Qt.Key_Right){
                trace_act_updateCursorsPosition.invokeWithStringArg(keyCursorAcceleratedDisplacement)
            }
        }
    }

    Keys.onReleased: {
        if(rootId.visible && (event.key === Qt.Key_Left || event.key === Qt.Key_Right)){
            pressAndHoldTimer.stop()
        }
    }

    Keys.onLeftPressed: {
        if(rootId.visible){
            if(trace_res_cursorsLocked.value){
                zoomButtonsLayout.zoomBy(1/zoomButtonsLayout.zoomInFactor, 1)
            }
            else {
                rootId.keyPressed = Qt.Key_Left
                updateCursorsPosition(-keyCursorDisplacement)
            }
        }
    }

    Keys.onRightPressed: {
        if(rootId.visible){
            if(trace_res_cursorsLocked.value){
                zoomButtonsLayout.zoomBy(zoomButtonsLayout.zoomInFactor, 1)
            }
            else {
                rootId.keyPressed = Qt.Key_Right
                updateCursorsPosition(keyCursorDisplacement)
            }
        }
    }
    Keys.onReturnPressed: {
        if(rootId.visible){
            trace_act_doAutoZoom.invoke()
        }
    }

    Connections {
        target: getFiberMainPage()
        onTopLevelUpPressed: {
            if(rootId.visible){
                zoomButtonsLayout.zoomBy(1, zoomButtonsLayout.zoomInFactor)
            }
        }
        onTopLevelDowPressed:{
            if(rootId.visible){
                zoomButtonsLayout.zoomBy(1, 1/zoomButtonsLayout.zoomInFactor)
            }
        }
        onTopLevelLeftPressed:{
            if(rootId.visible){
                if(trace_res_cursorsLocked.value){
                    zoomButtonsLayout.zoomBy(1/zoomButtonsLayout.zoomInFactor, 1)
                }
                else {
                    rootId.keyPressed = Qt.Key_Left
                    updateCursorsPosition(-keyCursorDisplacement)
                }
            }
        }
        onTopLevelRightPressed:{
            if(rootId.visible){
                if(trace_res_cursorsLocked.value){
                    zoomButtonsLayout.zoomBy(zoomButtonsLayout.zoomInFactor, 1)
                }
                else {
                    rootId.keyPressed = Qt.Key_Right
                    updateCursorsPosition(keyCursorDisplacement)
                }
            }
        }
        onTopLevelReturnPressed:{
            if(rootId.visible){
                trace_act_doAutoZoom.invoke()
            }
        }
        onTopLevelReleased:{
            if(rootId.visible){
                pressAndHoldTimer.stop()
            }
        }
    }


    function transformX(value) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        return (value - graphItem.width / graphItem.axisX.visibleRange * ( graphItem.axisX.minVisible - graphItem.axisX.minValue ))
    }


    function workingValueToScreenValue(value) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        if(graphItem){
            if( trace_res_isXAxisDecreasing.value )
            {
                value = graphItem.axisX.minValue + ( graphItem.axisX.maxValue - value )
            }
            return (( value - graphItem.axisX.minVisible ) * graphItem.width / graphItem.axisX.visibleRange )
        }

        return value
    }

    function screenValueToWorkingValue(value) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        if(graphItem){
            var result = (value / graphItem.width * graphItem.axisX.visibleRange + graphItem.axisX.minVisible)
            if( trace_res_isXAxisDecreasing.value )
            {
                result = graphItem.axisX.minValue + ( graphItem.axisX.maxValue - result )
            }
            return result
        }
        return 0
    }

    function onUpdateCursorPosFromUser(screenPos, updateElf){
        if(updateElf){
            updateElf.invokeWithStringArg(screenValueToWorkingValue(screenPos))
        }
    }

    function workingPointToZoomRect(point) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        var result = Qt.point(
                    (point.x - graphItem.axisX.minValue) / (graphItem.axisX.maxValue - graphItem.axisX.minValue) ,
                    (point.y - graphItem.axisY.minValue) / (graphItem.axisY.maxValue - graphItem.axisY.minValue) )
        return result
    }

    function zoomRectPointToWorking(point) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        var x = point.x * (graphItem.axisX.maxValue - graphItem.axisX.minValue) + graphItem.axisX.minValue
        var y = point.y * (graphItem.axisY.maxValue - graphItem.axisY.minValue) + graphItem.axisY.minValue
        var result = Qt.point(x, y)
        return result
    }

    Connections{
        target: trace_cfg_cursorAPosition
        onValueChanged:{
            if(!cursorDragA.pressed){
                cursorDragA.x = workingValueToScreenValue(trace_cfg_cursorAPosition.value)
            }
        }
    }


    Connections{
        target: trace_cfg_cursorBPosition
        onValueChanged:{
            if(!cursorDragB.pressed){
                cursorDragB.x = workingValueToScreenValue(trace_cfg_cursorBPosition.value)
            }
        }
    }

    Connections{
        target: trace_cfg_cursorAdditionalAPosition
        onValueChanged:{
            if(!cursorDragAdditionalA.pressed){
                cursorDragAdditionalA.x = workingValueToScreenValue(trace_cfg_cursorAdditionalAPosition.value)
            }
        }
    }

    Connections{
        target: trace_cfg_cursorAdditionalBPosition
        onValueChanged:{
            if(!cursorDragAdditionalB.pressed){
                cursorDragAdditionalB.x = workingValueToScreenValue(trace_cfg_cursorAdditionalBPosition.value)
            }
        }
    }

    Connections{
        target: trace_cfg_cursorAdditionalCPosition
        onValueChanged:{
            if(!cursorDragAdditionalC.pressed){
                cursorDragAdditionalC.x = workingValueToScreenValue(trace_cfg_cursorAdditionalCPosition.value)
            }
        }
    }

    function updateZoomToUI(){
        if(!zoomUpdateInProgress && !flick.moving  && !flick.pinching){
            var xCenter = (trace_cfg_zoomAreaToUI.value.left + trace_cfg_zoomAreaToUI.value.width / 2 - xAxis.minValue) / xAxis.valueRange
            var yCenter = (trace_cfg_zoomAreaToUI.value.top - trace_cfg_zoomAreaToUI.value.height / 2 - yAxis.minValue) / yAxis.valueRange
            var xScale = trace_cfg_zoomAreaToUI.value.width / xAxis.valueRange
            var yScale = trace_cfg_zoomAreaToUI.value.height / yAxis.valueRange
            graphId.zoomRect.setCenterAndZoom(Qt.vector2d(xScale, yScale)
                                              , Qt.point(xCenter, yCenter))

            rootId.zoomXValue = graphId.zoomRect.width
            rootId.zoomYValue = graphId.zoomRect.height
        }
    }

    Connections{
        target: trace_cfg_zoomAreaToUI
        onValueChanged:{
            updateZoomToUI()
        }
    }


    FiberContainer
    {
        id: rootFiberContainer
        anchors.fill: parent

        CurveCursorInfoBar {
            id: curveCursorInfoBar

            width: rootId.width
            trace: rootId
        }

        GraphView {
            id : graphId

            anchors.top: curveCursorInfoBar.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.bottom: parent.bottom

            anchors.topMargin: ViaviStyle.layouts.smallMargin
            anchors.leftMargin: ViaviStyle.layouts.smallMargin
            anchors.rightMargin: ViaviStyle.layouts.smallMargin
            anchors.bottomMargin: ViaviStyle.layouts.smallMargin

            zoomRect.zoomAxis: Qt.Horizontal | Qt.Vertical
            zoomRect.minWidth: trace_cfg_zoomRectMinWidth.value
            zoomRect.minHeight: trace_cfg_zoomRectMinHeight.value

            /* Update cursor position when Graph width changes because initial calculations can provide negative results*/
            onWidthChanged: {
                cursorDragA.x = workingValueToScreenValue(trace_cfg_cursorAPosition.value)
                cursorDragB.x = workingValueToScreenValue(trace_cfg_cursorBPosition.value)
                updateZoomToUI()
            }

            onHeightChanged: {
                updateZoomToUI()
            }

            zoomRect.onRectChanged: { setZoomLevelChanged() }

            GraphGestureArea {
                id: flick
                flickableDirection: Flickable.HorizontalAndVerticalFlick
                multipleDirectionPinch: true
                zoomRect: graphId.zoomRect

                onContentWidthChanged: {
                    if(pinching){
                        updateTraceZoomValuesFromUI()
                    }
                }

                onContentHeightChanged: {
                    if(pinching){
                        updateTraceZoomValuesFromUI()
                    }
                }

                onContentXChanged: {
                    if(moving){
                        updateTraceZoomValuesFromUI()
                    }
                }

                onContentYChanged: {
                    if(moving){
                        updateTraceZoomValuesFromUI()
                    }
                }

                MouseArea {
                    anchors.fill: parent
                    hoverEnabled: false
                    preventStealing: false
                    onClicked: {
                        var currentCursor = rootId.lastUsedCursor

                        var mouseXTransformed = transformX(mouseX)
                        if(currentCursor.connectedCursor){
                            var connectedCursor = currentCursor.connectedCursor

                            var currentDist = Math.abs(currentCursor.x - mouseXTransformed)
                            var connectedDist = Math.abs(connectedCursor.x - mouseXTransformed)

                            if(connectedDist < currentDist){
                                currentCursor = connectedCursor
                            }
                        }
                        currentCursor.updateConnectedCursorOffset()
                        currentCursor.updatePosition(mouseXTransformed, true, true)
                        rootId.forceActiveFocus()
                    }
                    onPressed: {
                        rootId.forceActiveFocus()
                    }
                }
            }

            ValueAxis{
                id: xAxisOsa
                visible: osaAxisVisible
                objectName: "xAxisOsa"
                alignment: Qt.AlignBottom
                margin: ViaviStyle.layouts.largeMargin
                linePen: ViaviStyle.colors.borderColor
                labelPen: ViaviStyle.colors.secondaryInfoColor
                lineJoinStyle: Qt.MiterJoin
                lineDashPattern: horizontalAxisDashPattern
                lineWidth: ViaviStyle.layouts.graphTickWidth
                labelFont: ViaviStyle.layouts.traceUnitFont
                maxValue: trace_res_points0_xRangeEnd.value
                minValue: trace_res_points0_xRangeBegin.value
                unit: trace_res_unitDisplayedX.value
                tickInterval: ValueAxis.FixedTicks
                ticks: {
                    var ticksArray = []
                    var arrayLength = osaGridLines_res_lines_doubleARRAY.valueCount
                    for (var i = 0; i < arrayLength ; i++) {
                        ticksArray.push ( osaGridLines_res_lines_doubleARRAY.at(i) )
                    }
                    return ticksArray;
                }
                axisStyle: ValueAxis.TicksOnly
                rangeOrder: trace_res_isXAxisDecreasing.value ? ValueAxis.Decreasing : ValueAxis.Increasing
            }

            ValueAxis{
                id: xAxis
                objectName: "xAxis"
                alignment: Qt.AlignBottom
                margin: ViaviStyle.layouts.largeMargin
                linePen: ViaviStyle.colors.borderColor
                labelPen: ViaviStyle.colors.secondaryInfoColor
                lineJoinStyle: Qt.MiterJoin
                lineDashPattern: horizontalAxisDashPattern
                lineWidth: ViaviStyle.layouts.graphTickWidth
                labelFont: ViaviStyle.layouts.traceUnitFont
                maxValue: trace_res_points0_xRangeEnd.value
                minValue: trace_res_points0_xRangeBegin.value
                unit: trace_res_unitDisplayedX.value
                axisStyle: xTicksVisible ? ValueAxis.TicksAndLabels : ValueAxis.LabelsOnly
                rangeOrder: trace_res_isXAxisDecreasing.value ? ValueAxis.Decreasing : ValueAxis.Increasing

                formatter: DecimalFormatter{
                    totalDigits: 8
                }

                onMaxValueChanged:{
                    cursorDragA.x = workingValueToScreenValue(trace_cfg_cursorAPosition.value)
                    cursorDragB.x = workingValueToScreenValue(trace_cfg_cursorBPosition.value)
                    updateZoomToUI()
                }

                onMinValueChanged:{
                    cursorDragA.x = workingValueToScreenValue(trace_cfg_cursorAPosition.value)
                    cursorDragB.x = workingValueToScreenValue(trace_cfg_cursorBPosition.value)
                    updateZoomToUI()
                }
            }

            ValueAxis{
                id: yAxis
                objectName: "yAxis"
                alignment: Qt.AlignLeft
                margin: ViaviStyle.layouts.buttonHeight + ViaviStyle.layouts.largeMargin
                linePen: ViaviStyle.colors.borderColor
                labelPen: ViaviStyle.colors.secondaryInfoColor
                lineJoinStyle: Qt.MiterJoin
                lineDashPattern: verticalAxisDashPattern
                lineWidth: ViaviStyle.layouts.graphTickWidth
                labelFont: ViaviStyle.layouts.traceUnitFont
                maxValue: trace_res_points0_yRangeEnd.value
                minValue: trace_res_points0_yRangeBegin.value
                numTicks : nrOfVerticalTicks
                unit: trace_res_unitDisplayedY.value
                axisStyle:  yTicksVisible ? ValueAxis.TicksAndLabels : ValueAxis.LabelsOnly
                formatter: DecimalFormatter{
                    totalDigits: 5
                }

                onMaxValueChanged:{
                    updateZoomToUI()
                }
                onMinValueChanged:{
                    updateZoomToUI()
                }
            }

            Repeater {
                id: graphRepeaterId

                GraphPlot {
                    id: reflections
                    objectName: "graph" + index

                    property bool controlledVisibility: !misc_res_hideAllTraceVisible.value ? true : emphasize
                    visible: platform.getElfData(':result:Fiber_Optic:trace:points' + index + '_xData:doubleARRAY:').valueCount > 0
                             && controlledVisibility

                    property ElfData pointAsSegment: platform.getElfData(":result:Fiber_Optic:trace:pointsAsSegments:" + index + ":")
                    property ElfData groupIndex: platform.getElfData(":result:Fiber_Optic:trace:traceGroupIndex:" + index + ":")

                    property bool isPmdScreen: status_res_osaMode.value === status_res_osaMode.enums.ecFunction_Pmd
                    property bool emphasize: (isPmdScreen && multigraphsStates_res_isPmdFftActive.value) ? true : (trace_cfg_selectedIndex.value === colorIndex)

                    plotData: PlotPoints {
                        dbusAddress: "gem.fiber.Fiber_Optic,:result:Fiber_Optic:trace:points" + index
                        axisY: yAxis
                        axisX: osaAxisVisible ? xAxisOsa: xAxis
                        usePointsAsSegments: pointAsSegment.value

                        active: reflections.colorIndex === trace_cfg_selectedIndex.value
                    }
                    antialiasing : false

                    property int colorIndex : (groupIndex.value !== -1) ? groupIndex.value : index

                    penColor: emphasize ? ViaviStyle.colors.activeColor : ViaviStyle.colors.traceColors[colorIndex]
                    plotStyle: GraphPlot.LinePlot
                    maxLineWidth: emphasize ? 3 * ViaviStyle.layouts.grapLineWidth : ViaviStyle.layouts.grapLineWidth
                    minLineWidth: emphasize ? 2 * ViaviStyle.layouts.grapLineWidth : ViaviStyle.layouts.grapLineWidth
                    useIntermittentPlot: false

                    //the axises have a z of 100
                    //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                    z: rootId.graphZ + (emphasize ? 1 : 0)
                }
            }

            GraphPlot {
                id: eventsId
                objectName: "events"

                plotData: PlotPoints {
                    dbusAddress: "gem.fiber.Fiber_Optic,:result:Fiber_Optic:trace:events" + trace_cfg_selectedIndex.value
                    axisY: yAxis
                    axisX: osaAxisVisible ? xAxisOsa: xAxis
                }
                graphPoint: GraphPointActiveMarker{
                    //opacity: parent.opacity
                    firstLabels: trace_res_activeMarkerFirstLabels_stringARRAY
                    secondLabels: trace_res_activeMarkerSecondLabels_stringARRAY
                    pointSize: ViaviStyle.layouts.activeMarkerPointSize
                    labelColor: ViaviStyle.colors.mainInfoColor
                    labelFontSize: ViaviStyle.layouts.minuteFontSize

                    secondLabelColor: ViaviStyle.colors.activeColor
                    secondLabelFontSize: ViaviStyle.layouts.minuteFontSize
                }

                antialiasing : false

                penColor: ViaviStyle.colors.mainInfoColor
                plotStyle: GraphPlot.ScatterPlot

                maxLineWidth: ViaviStyle.layouts.traceEventLineWidth
                minLineWidth: ViaviStyle.layouts.traceEventLineWidth

                //the axises have a z of 100
                //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                z: rootId.graphZ
            }
            //<Tailing>
            // this is where we plot the tailing in debug Mode
            Repeater {
                id: graphTailingRepeaterId

                GraphPlot {
                    id: tailingId
                    objectName: "graphDebugTailing"+index

                    plotData: PlotPoints {
                        dbusAddress: "gem.fiber.Fiber_Optic,:result:Fiber_Optic:debugInfoTailing:points"
                        axisY: yAxis
                        axisX: xAxis
                    }
                    antialiasing : false

                    penColor: '#f58231'
                    plotStyle: GraphPlot.LinePlot

                    maxLineWidth: 3 * ViaviStyle.layouts.grapLineWidth//ViaviStyle.layouts.traceEventLineWidth
                    minLineWidth: 3 * ViaviStyle.layouts.grapLineWidth//ViaviStyle.layouts.traceEventLineWidth

                    //the axises have a z of 100
                    //in order to have the trace above the dashes from the vertical axis we need to set the z to something heigher
                    z: rootId.graphZ+10
                }
            }
            //</Tailing>
            ContainerPlot{
                id: crosses
                axisY: yAxis
                axisX: xAxis

                Repeater{
                    model: trace_res_crosses.value
                    CrossPainter{
                        point1X: model.point1X
                        point1Y: model.point1Y
                        crossColor: model.crossColor

                        lineWidth: ViaviStyle.layouts.separatorSize
                        crossSize: ViaviStyle.layouts.crossEventSize
                    }
                }
            }

            ContainerPlot{
                id: segments
                axisY: yAxis
                axisX: xAxis


                Repeater{
                    model: trace_res_segments.value
                    SegmentPainter {
                        point1: Qt.point(model.point1X, model.point1Y)
                        point2: Qt.point(model.point2X, model.point2Y)

                        lineColor: model.lineColor
                        label: model.label
                        labelColor: model.labelColor
                        labelFontSize: ViaviStyle.layouts.minuteFontSize

                        drawLimits: model.drawLimits
                        drawSlope: model.drawSlope

                        labelOffset: ViaviStyle.layouts.mediumMargin
                    }
                }

                z: rootId.segmentZ
            }

            ContainerPlot{
                id: integrals
                axisY: yAxis
                axisX: xAxis

                Repeater{
                    model: trace_res_integrals.value
                    IntegralPainter{
                        x1: model.x1
                        x2: model.x2

                        fillColor: model.fillColor
                        label: model.label
                        labelColor: model.labelColor
                        labelFontSize: ViaviStyle.layouts.minuteFontSize
                        labelOffset: 3 * ViaviStyle.layouts.mediumMargin

                        plotData: graphRepeaterId.count > 0 ? graphRepeaterId.itemAt(trace_cfg_selectedIndex.value).plotData : null

                    }
                }

                z: rootId.integralZ
            }

            ContainerPlot{
                id: texts
                axisY: yAxis
                axisX: xAxis

                Repeater{
                    model: trace_res_texts.value
                    TextPainter {
                        origin: Qt.point(model.point1X, model.point1Y)

                        label: model.label
                        labelColor: model.labelColor
                        font: Qt.font({pointSize: ViaviStyle.layouts.minuteFontSize})
                        hTextAlignment: model.hTextAlignment
                        vTextAlignment: model.vTextAlignment
                    }
                }

                z: rootId.integralZ
            }

            Item {
                x: eventsId.x
                //Note: icons has an Item as parent because GraphView re-anchors with 'fill' all its direct children, thus causing icons to lose their size binding
                id: iconsHolder
                Repeater {
                    id: icons
                    model: trace_res_icons.value
                    FiberColorImage {
                        source: model.iconPath
                        color: ViaviStyle.colors.mainInfoColor

                        function getAllignmentOffset()
                        {
                            return (model.hIconAlignment === Text.AlignLeft) ? (- width)
                                                                             : (model.hIconAlignment === Text.AlignHCenter) ? (- width / 2) : 0
                        }

                        x: (model.point1X - xAxis.minVisible) * (eventsId.width - 1.0 ) / xAxis.visibleRange + getAllignmentOffset()
                        y: zoomButtonsLayout.height

                        fillMode: Image.PreserveAspectFit

                        width: ViaviStyle.layouts.traceIconEventSize
                        height: ViaviStyle.layouts.traceIconEventSize

                        visible: model.point1X >= xAxis.minVisible && model.point1X <= xAxis.maxVisible - paintedWidth * xAxis.visibleRange / (eventsId.width - 1.0 )

                        z: rootId.integralZ
                    }
                }
            }

            GraphPlot{
                objectName: "pictograms"


                plotData: PlotPoints {
                    dbusAddress: "gem.fiber.Fiber_Optic,:result:Fiber_Optic:trace:pictograms" + trace_cfg_selectedIndex.value
                    axisY: yAxis
                    axisX: xAxis
                }
                graphPoint: GraphPointPictogramMarker{
                    labels: trace_res_pictogramMarkerLabels_stringARRAY
                    pointSize: ViaviStyle.layouts.traceEventLineWidth
                    labelColor: ViaviStyle.colors.mainInfoColor
                    labelFontSize: ViaviStyle.layouts.minuteFontSize
                }
                antialiasing : false

                penColor: ViaviStyle.colors.mainInfoColor
                plotStyle: GraphPlot.ScatterPlot

                maxLineWidth: ViaviStyle.layouts.traceEventLineWidth
                minLineWidth: ViaviStyle.layouts.traceEventLineWidth

                z: rootId.graphZ
            }


            GraphCursor {
                id: cursorA
                visible: status_res_cursorsEnabled.value
                enabled: status_res_cursorsEnabled.value

                opacity: status_res_cursorsEnabled.value ? 1.0 : 0.0
                color: ViaviStyle.colors.redCursorColor
                staysOnScreen: false

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                //This CursorDrag is getting in the way of pinch-to-zoom!
                CursorDragFiber {
                    id: cursorDragA

                    connectedCursor: additionalCursors
                                     ? (additionalCursorALock ? cursorDragAdditionalA : null)
                                     : trace_res_selectedCursors.value === "Cursors_AB"
                                       ? cursorDragB : null


                    parentTrace: rootId
                    updateElf: trace_act_updateCursorAFromUI
                    xAxis: xAxis
                    backendId: trace_cfg_cursorSelected.enums.ecCursor_A

                    onVisibleChanged: {
                        if(visible){
                            cursorDragA.x = workingValueToScreenValue(trace_cfg_cursorAPosition.value)
                        }
                    }
                    text: cursorALabel
                    color: ViaviStyle.colors.redCursorColor
                }
            }

            GraphCursor {
                id: cursorAdditionalA
                visible: status_res_cursorsEnabled.value
                enabled: status_res_cursorsEnabled.value

                //must use opacity instead of visible because GraphCursor redefines a final property. but for backwords compatability it must kept as it is
                opacity: status_res_cursorsEnabled.value && additionalCursors ? 1.0 : 0.0
                staysOnScreen: false
                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                CursorDragFiber {
                    id: cursorDragAdditionalA

                    connectedCursor: (additionalCursors && additionalCursorALock)
                                     ? cursorDragA : null
                    parentTrace: rootId
                    updateElf: trace_act_updateCursorAdditionalAFromUI
                    xAxis: xAxis

                    onVisibleChanged: {
                        if(visible){
                            x = cursorDragA.x + additionalCursorsInitialOffset
                            trace_act_updateCursorAdditionalAFromUI.invokeWithStringArg(screenValueToWorkingValue(x))
                        }
                    }
                    text: cursorAdditionalALabel
                    color: ViaviStyle.colors.redCursorColor
                }
            }

            GraphCursor {
                id: cursorB
                visible: status_res_cursorsEnabled.value
                enabled: status_res_cursorsEnabled.value

                opacity: status_res_cursorsEnabled.value ? 1.0 : 0.0
                color: ViaviStyle.colors.blueCursorColor
                staysOnScreen: false

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                CursorDragFiber {
                    id: cursorDragB

                    connectedCursor: additionalCursors
                                     ? (additionalCursorBLock ? cursorDragAdditionalB : null)
                                     : trace_res_selectedCursors.value === "Cursors_AB"
                                       ? cursorDragA : null

                    parentTrace: rootId
                    updateElf: trace_act_updateCursorBFromUI
                    xAxis: xAxis
                    backendId: trace_cfg_cursorSelected.enums.ecCursor_B

                    onVisibleChanged: {
                        if(visible){
                            cursorDragB.x = workingValueToScreenValue(trace_cfg_cursorBPosition.value)
                        }
                    }
                    text: cursorBLabel
                    color: ViaviStyle.colors.blueCursorColor
                }
            }

            GraphCursor {
                id: cursorAdditionalB
                visible: status_res_cursorsEnabled.value
                enabled: status_res_cursorsEnabled.value

                //must use opacity instead of visible because GraphCursor redefines a final property. but for backwords compatability it must kept as it is
                opacity: status_res_cursorsEnabled.value && additionalCursors ? 1.0 : 0.0

                staysOnScreen: false

                color: ViaviStyle.colors.blueCursorColor

                autoPrimaryPlot: true

                paintedWidth: ViaviStyle.layouts.cursorWidth

                CursorDragFiber {
                    id: cursorDragAdditionalB

                    connectedCursor: (additionalCursors && additionalCursorBLock)
                                     ? cursorDragB : null
                    parentTrace: rootId
                    updateElf: trace_act_updateCursorAdditionalBFromUI
                    xAxis: xAxis

                    onVisibleChanged: {
                        if(visible){
                            x = cursorDragB.x + additionalCursorsInitialOffset
                            trace_act_updateCursorAdditionalBFromUI.invokeWithStringArg(screenValueToWorkingValue(x))
                        }
                    }

                    text: cursorAdditionalBLabel
                    color: ViaviStyle.colors.blueCursorColor
                }
            }

            GraphCursor {
                id: cursorAdditionalC
                visible: status_res_cursorsEnabled.value
                enabled: status_res_cursorsEnabled.value

                //must use opacity instead of visible because GraphCursor redefines a final property. but for backwords compatability it must kept as it is
                opacity: status_res_cursorsEnabled.value && additionalCursors ? 1.0 : 0.0

                staysOnScreen: false

                color: ViaviStyle.colors.secondaryInfoColor

                autoPrimaryPlot: true

                CursorDragFiber {
                    id: cursorDragAdditionalC

                    parentTrace: rootId
                    updateElf: trace_act_updateCursorAdditionalCFromUI
                    xAxis: xAxis

                    onVisibleChanged: {
                        if(visible){
                            x = (cursorDragA.x + cursorDragB.x) / 2
                            trace_act_updateCursorAdditionalCFromUI.invokeWithStringArg(screenValueToWorkingValue(x))
                        }
                    }

                    text: cursorAdditionalCLabel
                    color: ViaviStyle.colors.secondaryInfoColor
                }
            }
        }
        //    }




        Row {
            id: zoomButtonsLayout
            visible: trace_res_hasZoomButtons.value ? true : false

            anchors.top: curveCursorInfoBar.bottom
            anchors.right: parent.right

            clip: true

            property real zoomInFactor: 2/3
            function zoomBy(zoomFactorX, zoomFactorY) {
                var zoomXValue = Math.min(1.0, rootId.zoomXValue * zoomFactorX)
                var zoomYValue = Math.min(1.0, rootId.zoomYValue * zoomFactorY)
                var invariantPoint = findInvariantPoint()
                var center = Qt.point(graphId.zoomRect.x + graphId.zoomRect.width / 2,
                                      graphId.zoomRect.y + graphId.zoomRect.height / 2)

                /*  View center will move towards the zoom invariant point when zooming in, and in the opposite direction on zoom out
                The computation cannot be done by a simple multiplication by the factor, as a portion of the distance is already offset. */
                center.x += (invariantPoint.x - center.x) * (1 - zoomFactorX)
                center.y += (invariantPoint.y - center.y) * (1 - zoomFactorY)

                graphId.zoomRect.setCenterAndZoom(Qt.vector2d(zoomXValue, zoomYValue), center)
                updateTraceZoomValuesFromUI()
                rootId.zoomXValue = graphId.zoomRect.width
                rootId.zoomYValue = graphId.zoomRect.height
            }

            property int buttonApparentSize: ViaviStyle.layouts.imageButtonHeight
            property int buttonTopPadding: ViaviStyle.layouts.mediumMargin
            property int buttonHPadding: ViaviStyle.layouts.largeMargin

            FiberImageButton  {
                id: zoomInBtn
                smooth: true
                source: ViaviStyle.images.zoomInBtnImg
                color: 'transparent'
                width: parent.buttonApparentSize + leftPadding + rightPadding
                height: parent.buttonApparentSize + topPadding
                leftPadding: parent.buttonHPadding
                rightPadding: parent.buttonHPadding
                topPadding: parent.buttonTopPadding
                onClicked: {
                    zoomButtonsLayout.zoomBy(zoomButtonsLayout.zoomInFactor, zoomButtonsLayout.zoomInFactor)
                }
            }

            FiberImageButton  {
                id: zoomOutBtn
                smooth: true
                source: ViaviStyle.images.zoomOutBtnImg
                color: 'transparent'
                width: parent.buttonApparentSize + leftPadding + rightPadding
                height: parent.buttonApparentSize + topPadding
                leftPadding: parent.buttonHPadding
                rightPadding: parent.buttonHPadding
                topPadding: parent.buttonTopPadding
                onClicked: {
                    zoomButtonsLayout.zoomBy(1/zoomButtonsLayout.zoomInFactor, 1/zoomButtonsLayout.zoomInFactor)
                }
            }

            FiberImageButton  {
                id: zoomAutoBtn
                smooth: true
                source:ViaviStyle.images.zoomAutoBtnImg
                color: 'transparent'
                width: parent.buttonApparentSize + leftPadding + rightPadding
                height: parent.buttonApparentSize + topPadding
                leftPadding: parent.buttonHPadding
                rightPadding: parent.buttonHPadding
                topPadding: parent.buttonTopPadding
                onClicked: {
                    trace_act_doAutoZoom.invoke()
                }
            }
        }
    }

    // @param {double} extendRatio - extend the range within which x is considered to be contained. Value is a proportion of the are width.
    function workingValueInVisibleRangeX(x, extendRatio = 0) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        var extendValue = 0
        if (extendRatio != 0) {
            extendValue = Math.abs(graphItem.axisX.maxVisible - graphItem.axisX.minVisible) * extendRatio
        }
        var result = (x >= graphItem.axisX.minVisible - extendValue) && (x <= graphItem.axisX.maxVisible + extendValue)
        return result
    }

    function workingValueInVisibleRangeY(y, extendRatio = 0) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        var extendValue = 0
        if (extendRatio != 0) {
            extendValue = Math.abs(graphItem.axisY.maxVisible - graphItem.axisY.minVisible) * extendRatio
        }
        var result = (y >= graphItem.axisY.minVisible - extendValue) && (y <= graphItem.axisY.maxVisible + extendValue)
        return result
    }

    // NOTE: function assumes trace samples are distributed uniformly, no sparse arrays.
    // Not currently used because of difficulties with .csor curves.
    function graphPointAtWorkingValue(value) {
        var graphItem = graphRepeaterId.itemAt(trace_cfg_selectedIndex.value)
        if(graphItem) {
            var xAddress = ':result:Fiber_Optic:trace:points' + trace_cfg_selectedIndex.value +'_xData:doubleARRAY:'
            var yAddress = ':result:Fiber_Optic:trace:points' + trace_cfg_selectedIndex.value +'_yData:doubleARRAY:'
            var pointsX = platform.getElfData(xAddress)
            var pointsY = platform.getElfData(yAddress)

            var index = (value - graphItem.axisX.minValue) / (graphItem.axisX.maxValue  - graphItem.axisX.minValue) * pointsX.valueCount
            if (index < 0 || index > pointsX.valueCount) {
                console.log('Warning! graphPointAtWorkingValue could not find graph point at x ' + value)
            }
            else {
                var res = Qt.point(pointsX.at(index), pointsY.at(index))
                return res
            }
        }
        console.log('Warning! graphPointAtWorkingValue called on invalid graph')
        return Qt.point(0, 0)
    }

    /*
Find the invariant point for a zoom operation, in  screen coordinates 0-1
Invariant points rules:
x: - the single selected cursor, if it is on screen
   - the middle between the two selected cursors if they're both on screen
   - the middle of the visible part otherwise
y: - the intersection between the single selected cursor and the graph, if it is on screen
   - the value of the trace sample at the middle between the two selected cursors
   - the middle of the visible part otherwise

    NOTE: the heuristics employed can be further improved for two selected cursors: case when one of the selected cursors is close to the visible edge, a zoom in may take it outside. In such a case the following could be done: if a zoom operation is going to take one cursor outside then zoom center should be choses such that the cursor will continue to stay inside.
*/
    function findInvariantPoint() {
        // Target points will be considered within the visible area if they are outside with less than 1%.
        // This will prevent cases where the target point is right at the edge and will jump outside on zoom due to floating point precision.
        var extraMarginRatio = 0.01
        var x =  graphId.zoomRect.x + graphId.zoomRect.width / 2
        switch (trace_cfg_cursorSelected.value) {
        case trace_cfg_cursorSelected.enums.ecCursor_A:
            if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio)) {
                x = workingPointToZoomRect(Qt.point(cursorA.posValue, 0)).x
            }
            break;
        case trace_cfg_cursorSelected.enums.ecCursor_B:
            if (workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                x = workingPointToZoomRect(Qt.point(cursorB.posValue, 0)).x
            }
            break;
        case trace_cfg_cursorSelected.enums.ecCursor_AB:
            if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio) &&
                workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                x = workingPointToZoomRect(
                            Qt.point(Math.min(cursorA.posValue, cursorB.posValue) + (Math.max(cursorA.posValue, cursorB.posValue) - Math.min(cursorA.posValue, cursorB.posValue)) / 2, 0)).x
            }
            break;
        default:
            break;
        }

        var y = graphId.zoomRect.y + graphId.zoomRect.height / 2
        switch (trace_cfg_cursorSelected.value) {
        case trace_cfg_cursorSelected.enums.ecCursor_A:
            if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio)) {
                var workingIntersection = cursorA.yValues[cursorA.primaryPlotIndex]
                if (workingValueInVisibleRangeY(workingIntersection, extraMarginRatio)) {
                    y = workingPointToZoomRect(Qt.point(0, workingIntersection)).y
                }
            }
            break;
        case trace_cfg_cursorSelected.enums.ecCursor_B:
            if (workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                var workingIntersection = cursorB.yValues[cursorB.primaryPlotIndex]
                if (workingValueInVisibleRangeY(workingIntersection, extraMarginRatio)) {
                    y = workingPointToZoomRect(Qt.point(0, workingIntersection)).y
                }
            }
            break;
        case trace_cfg_cursorSelected.enums.ecCursor_AB:
            if (workingValueInVisibleRangeX(cursorA.posValue, extraMarginRatio) &&
                workingValueInVisibleRangeX(cursorB.posValue, extraMarginRatio)) {
                var workingIntersectionA = cursorA.yValues[cursorA.primaryPlotIndex]
                var workingIntersectionB = cursorB.yValues[cursorB.primaryPlotIndex]
                y = workingPointToZoomRect(
                            Qt.point(0, Math.min(workingIntersectionA, workingIntersectionB) + (Math.max(workingIntersectionA, workingIntersectionB) - Math.min(workingIntersectionA, workingIntersectionB)) / 2)).y
            }
            break;
        default:
            break;
        }

        var result = Qt.point(x, y)
        return result
    }


    function autoZoom(){
        rootId.zoomXValue = 1.0
        rootId.zoomYValue = 1.0
        var center = Qt.point(graphId.zoomRect.x + graphId.zoomRect.width / 2,
                              graphId.zoomRect.y + graphId.zoomRect.height /2)
        graphId.zoomRect.zoom(Qt.vector2d(rootId.zoomXValue, rootId.zoomYValue),
                              center)
    }

    function updateTraceZoomValuesFromUI() {
        zoomUpdateInProgress = true
        trace_cfg_zoomLeftPosFromUI.value = xAxis.minVisible
        trace_cfg_zoomTopPosFromUI.value = yAxis.maxVisible
        trace_cfg_zoomWidthFromUI.value = xAxis.visibleRange
        trace_cfg_zoomHeightFromUI.value = yAxis.visibleRange
        zoomUpdateInProgress = false

        trace_act_updateZoomFromUI.invoke()
    }

    function setZoomLevelChanged() {
        zoomLevelChanged(graphId.zoomRect.x,
                         graphId.zoomRect.y,
                         graphId.zoomRect.width,
                         graphId.zoomRect.height)
    }

    function updateCursorsPosition(x) {
        var currentCursor = rootId.lastUsedCursor
        if (currentCursor) {
            pressAndHoldTimer.start()
            trace_act_updateCursorsPosition.invokeWithStringArg(x)
        }
    }

    function updateCurrentCursors() {
        if (additionalCursors) {
            if (lastUsedCursor == cursorDragA) {
                if (lastUsedCursor.connectedCursor) {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_Aa
                }
                else {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_A
                }
            }
            else if (lastUsedCursor == cursorDragB) {
                if (lastUsedCursor.connectedCursor) {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_Bb
                }
                else {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_B
                }
            }
            else if (lastUsedCursor == cursorDragAdditionalA) {
                if (lastUsedCursor.connectedCursor) {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_Aa
                }
                else {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_a
                }
            }
            else if (lastUsedCursor == cursorDragAdditionalB) {
                if (lastUsedCursor.connectedCursor) {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_Bb
                }
                else {
                    trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_b
                }
            }
            else if (lastUsedCursor == cursorDragAdditionalC) {
                trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eCursor_c
            }
            else {
                trace_cfg_currentCursor.value = trace_cfg_currentCursor.enums.eNo_Cursor
            }
        }
    }
}
