<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\util;

/**
 * Base Class for dto objects that can be edited on client navigator. 
 * The DTO is sent to the client application and sent back to the server.
 * It computes a checksum to track object changes since the last fetch.
 *
 * @author Sylvain Desplat
 */
abstract class SMTRwDto extends SMTDto
{
    const CHECKSUM_PROPERTY = 'checkSum';
    
    /**
     * Dto checksum on properties to check modification when updating the configuration.
     *
     * @SMTAttributeInfosAnnotation(ignoreCheckSum=TRUE)
     * @var int
     */
    protected $checkSum;
    
    
    /**
     * @return int 
     */
    public function getCheckSum()
    {
        return $this->checkSum;
    }
    
    /**
     * Serialize as Json data (array) an object which is a property of the current instance.<br>
     * That method is recursively called by getJsonData();
     *
     * @param var $objectProperties array of defined object accessible non-static properties
     */
    function serializeObjectAsJsonData( &$objectProperties, $checkSum = TRUE )
    {
    	//compute checksum before sending the dto object
    	if ( $checkSum )
    	{
    		$objectProperties[self::CHECKSUM_PROPERTY] = $this->computeDtoCheckSum( $objectProperties );
    	}
    	else 
    	{
    		unset($objectProperties[self::CHECKSUM_PROPERTY]);
    	}
    	//serialize object
    	parent::serializeObjectAsJsonData( $objectProperties );
    }
    
    /**
     * Create the Dto from the given dto class name and a Json Dto string
     *
     * @param string $dtoClassName Dto class name
     * @param array $json_array
     * @return \app\util\$dto
     *
     * @throws InvalidArgumentException if the dto class doesn't exist.
     */
//     static function getInstance( $dtoClassName, array $json_array )
//     {
//     	return self::forge( $dtoClassName, $json_array );
//     }
    
    /**
     * cast stdClass object to a Dto. Recursively cast object properties if they are dto.
     *
     * @param string $dtoClassName the target dto class type
     * @param array $json_array
     * @param boolean $propertyNotFoundThrowException TRUE by default. Whether an exception must be thrown when a property is not found. 
     *
     * @return $dto Dto instance
     *
     * @throws InvalidArgumentException if the dto class doesn't exist or if a property couldn't be found.
     */    
    static function forge( $dtoClassName, array &$json_array, $propertyNotFoundThrowException=TRUE )
    {
        $dto = NULL;
    	if ( $json_array != NULL )
    	{
    		self::checkDtoClass( $dtoClassName );
    
    		$dto = new $dtoClassName();
    
    		foreach($json_array as $property => &$value)
    		{
    		    //always force property to have first character in lower case (used for properties sent by Otu)
    		    $dtoProperty = lcfirst($property);
    		    $propertyFound = property_exists( $dto, $dtoProperty);
    		    if ( !$propertyFound )
    		    {
    		        $dtoProperty = self::findPropertyNameCaseInsensitive($dto, $dtoProperty);
    		        $propertyFound = property_exists( $dto, $dtoProperty);
    		    }
    		    
    		    if ( $propertyFound )
    		    {
        			if ( is_array( $value ) )
        			{
        				$propertyAnnotation = new \ReflectionAnnotatedProperty( $dtoClassName, $dtoProperty );
        				if ( $propertyAnnotation->hasAnnotation( 'SMTAttributeInfosAnnotation' ) )
        				{
        					$attributeInfos = $propertyAnnotation->getAnnotation('SMTAttributeInfosAnnotation');
        					if ( isset ($attributeInfos->classname) )
        					{
        						//if the value is an array (test $attributeInfos->islist ) containing objects of class $attributeInfos->classname, loop on the objects
        						if ( $attributeInfos->islist)
        						{
        							foreach( $value as $key => &$listvalue)
        							{
        							    if ( $listvalue != NULL )
        							    {
        								    $listvalue = self::forge( $attributeInfos->classname, $listvalue, $propertyNotFoundThrowException );
        							    }
        							}
        						}
        						else if ( $value !== NULL )
        						{
        							$value = self::forge( $attributeInfos->classname, $value, $propertyNotFoundThrowException );
        						}
        					}
        				}
        			}

    				$dto->$dtoProperty = &$value;
    			}
    			else
    			{
    			    if ( $propertyNotFoundThrowException )
    			    {
    				    throw new \InvalidArgumentException( sprintf( MSG_ERROR_DTO_PROPERTY_NOT_FOUND, $property, $dtoClassName ) );
    			    }
    			    else
    			    {
    			        SMTLogger::getInstance()->trace( sprintf( MSG_ERROR_DTO_PROPERTY_NOT_FOUND, $property, $dtoClassName ) );
    			    }
    			}
    
    			unset($json_array[$property]);
    		}
    		unset($value);
    		$json_array = (unset) $json_array;
    	}
    	return $dto;
    }    
    
    /**
     * Check if Dto class exists.
     *
     * @param string $dtoClassName dto class name
     *
     * @throws InvalidArgumentException if the dto class doesn't exist.
     */
    private static function checkDtoClass( $dtoClassName )
    {
    	if ( !class_exists( $dtoClassName ) || !( new $dtoClassName() instanceof SMTDto ) )
    	{
    		throw new \InvalidArgumentException( sprintf( MSG_ERROR_DTO_NOT_FOUND, $dtoClassName ) );
    	}
    }    
    
    /**
     * Compute checksum on dto properties not marked as ignored and which are not objects.
     *
     * @return int the checksum of dto properties not marked as ignored and which are not objects.
     */
    function computeObjectDtoCheckSum()
    {
        return $this->computeDtoCheckSum( get_object_vars( $this ) );
    }
    
    
    /**
     * Compute checksum on dto properties (using already computed object properties given as parameter) not marked as ignored and which are not objects.
     * @param $objectProperties current object properties
     * 
     * @return int the checksum of dto properties not marked as ignored and which are not objects.
     */
    function computeDtoCheckSum( $objectProperties )
    {
    	$dtoPropertiesForCheckSum = SMTUtil::arrayCopy( $objectProperties );
    
    	foreach($objectProperties as $property => $value)
    	{
    		$propertyAnnotation = new \ReflectionAnnotatedProperty( $this->getDtoClassName(), $property );
    		if ( $propertyAnnotation->hasAnnotation( 'SMTAttributeInfosAnnotation' ) )
    		{
    			$attributeInfos = $propertyAnnotation->getAnnotation('SMTAttributeInfosAnnotation');
    			if ( $attributeInfos->ignoreCheckSum )
    			{
    				unset( $dtoPropertiesForCheckSum[$property] );
    			}
    			else
    			{ 
                    self::checkPropertiesForCheckSum( $property, $value, $dtoPropertiesForCheckSum );
    			}
    		}
    		else
    		{
    			self::checkPropertiesForCheckSum( $property, $value, $dtoPropertiesForCheckSum );
    		}    		
    		
    		if ( isset($dtoPropertiesForCheckSum[$property]) && is_array( $dtoPropertiesForCheckSum[$property] ) )
    		{
    			$dtoPropertiesForCheckSum[$property] = implode( $dtoPropertiesForCheckSum[$property] ).count($value);
    		}
    	}
    
    	return crc32 ( implode( $dtoPropertiesForCheckSum ) );
    }
    
    /**
     * Recurse in object properties arrays to exclude the properties which are Objects.
     * 
     * @param string $propertyKey
     * @param var $propertyValue
     * @param array $dtoPropertiesForCheckSum the array with properties to check. That array passed by reference is modified by the current function.
     * 
     */
    private static function checkPropertiesForCheckSum( $propertyKey, $propertyValue, &$dtoPropertiesForCheckSum )
    {
        if ( is_object( $propertyValue ) )
        {
            unset( $dtoPropertiesForCheckSum[ $propertyKey ] );
        }
        else if ( is_array( $propertyValue ) )
        {
        	foreach( $propertyValue as $arrayKey => $arrayValue )
        	{
        		self::checkPropertiesForCheckSum( $arrayKey, $arrayValue, $dtoPropertiesForCheckSum[$propertyKey] );
        	}
        }
    }
    
    /**
     * Search if the given property name case insensitive exists in the given dto. 
     * @param SMTDTO $dto
     * @param string $name
     * @return string the dto property name found or NULL if not found
     */
    private static function findPropertyNameCaseInsensitive($dto, $name)
    {
      $propertyNames = array_keys( get_object_vars($dto) );
      foreach( $propertyNames as $propertyName )
      {
        if ( strcasecmp( $name, $propertyName ) == 0 )
        {
          return $propertyName;
        }
      }
      return NULL;
    }
}
?>