<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

use app\services\monitoring\SMTOtuMonitoringTestThresholdDto;

/**
 * Monitoring test threshold buffer encoder/decoder used for switch buffer in OTDR acquisitions, tests...
 * 
 * @author Sylvain Desplat
 */
class SMTThresholdBufferEncoderDecoder
{
    const TOPAZ_THRESHOLD_COUNT = 5;
    const THRESHOLD_STRUCT_SIZE = 34; //17 * 2    
    const THRESHOLD_DIGITS = 2;  
    const THRESHOLD_VALUE_SIZE = 4;
    const THRESHOLD_VALUE_MONITORED_SIZE = 2;  
    const HIGHSENS_THRESHOLD_COUNT = 1;
    
    /**
     * @var string
     */
    private $thresholdsBufferWithHeader = NULL;
    /**
     * @var SMTOtuMonitoringTestThresholdDto
     */
    private $firstMarkerThreshold = NULL;
    /**
     * @var SMTOtuMonitoringTestThresholdDto
     */
    private $deltaFirstLastMarkerThreshold = NULL;
    
    /**
     * Monitoring thresholds for highsens monitoring
     * 
     * @var SMTOtuMonitoringTestThresholdDto
     */    
    private $monitoringThreshold = NULL;
    
    /**
     * @var array 
     */
    private $peakThresholds = array();        
    
    /**
     * @param string $thresholdsBufferWithHeader
     */
    function __construct( $thresholdsBufferWithHeader )
    {
        $this->thresholdsBufferWithHeader = $thresholdsBufferWithHeader;
    } 

    /**
     * Encode thresholds in a buffer string 
     * 
     * @param SMTOtuMonitoringTestThresholdDto $firstMarkerThreshold
     * @param SMTOtuMonitoringTestThresholdDto $deltaFirstLastMarkerThreshold
     * @param array $existingPeakThresholds
     * @return string
     */
    static function encodeThresholds( SMTOtuMonitoringTestThresholdDto $firstMarkerThreshold, 
    		                          SMTOtuMonitoringTestThresholdDto $deltaFirstLastMarkerThreshold, 
    		                          array &$existingPeaks,
    		                          $peakCount = 0)
    {
    	$buffer = self::encodeBufferThresholdHeader($peakCount, self::TOPAZ_THRESHOLD_COUNT);
        $buffer .= self::encodeThreshold( $firstMarkerThreshold );
        $buffer .= self::encodeThreshold( $deltaFirstLastMarkerThreshold );
        //new peak
        $newPeakThresholds = new SMTOtuMonitoringTestThresholdDto();
        $newPeakThresholds->setMonitored( FALSE );
        $buffer .= self::encodeThreshold( $newPeakThresholds );
        //fiber extension
        $fiberExtensionThresholds = new SMTOtuMonitoringTestThresholdDto();
        $fiberExtensionThresholds->setMonitored( FALSE );        
        $buffer .= self::encodeThreshold( $fiberExtensionThresholds );
        //ORL
        $orlThresholds = new SMTOtuMonitoringTestThresholdDto();
        $orlThresholds->setMonitored( FALSE );
        $buffer .= self::encodeThreshold( $orlThresholds );
        //existing peaks
        if ( ($existingPeaks!= NULL) && $peakCount > 0)
        {
        	foreach($existingPeaks as &$existingPeak)
        	{
        		$buffer .= self::encodeThreshold( $existingPeak->getPeakThresholds() );
        	}
        }
        
        return $buffer;
    }
    
    /**
     * Encode a threshold in a buffer string
     *
     * @param SMTOtuMonitoringTestThresholdDto $monitoringThreshold
     */
    static function encodeHighsenMonitoringThreshold( SMTOtuMonitoringTestThresholdDto $monitoringThreshold )
    {
    	$buffer = self::encodeBufferThresholdHeader(0, self::HIGHSENS_THRESHOLD_COUNT);
    	$buffer .= self::encodeThreshold( $monitoringThreshold);    	
    	return $buffer;
    }
    
    /**
     * Encode a threshold in a buffer string
     *
     * @param array $existingPeaks
     * @param int $peakCount
     */
    static function encodeHighsenPeaksMonitoringThreshold( array &$existingPeaks,
    		                                               $peakCount = 0 )
    {
    	$buffer = self::encodeBufferThresholdHeader($peakCount, 0);
    	//existing peaks
    	if ( ($existingPeaks!= NULL) && $peakCount > 0)
    	{
    		foreach($existingPeaks as &$existingPeak)
    		{
    			$buffer .= self::encodeThreshold( $existingPeak->getPeakThresholds() );
    		}
    	}
    	return $buffer;
    }
    
    /**
     * @return string
     */
    private static function encodeBufferThresholdHeader( $peakCount = 0, $thresholdCount = 1)
    {
    	$bufferSize = ( $peakCount + $thresholdCount) * self::THRESHOLD_STRUCT_SIZE;
        
        if ($bufferSize < 10)
        	$buffer = sprintf ("#1%d", $bufferSize);
        else if ($bufferSize <100)
        	$buffer = sprintf ("#2%d", $bufferSize);
        else if ($bufferSize < 1000)
        	$buffer = sprintf ("#3%d", $bufferSize);
        else if ($bufferSize < 10000)
        	$buffer = sprintf ("#4%d", $bufferSize);
        
        return $buffer;
    }
    
    /**
     * Check that minor threshold is really active: Minor threshold != Major threshold, else disable minor threshold flag
     * 
     * @param SMTOtuMonitoringTestThresholdDto $threshold
     */
    private static function checkMinorThresholdEnabledFlag(SMTOtuMonitoringTestThresholdDto $threshold)
    {
    	if ( ($threshold->getPositiveToMajorThreshold() == $threshold->getPositiveToMinorThreshold()) )
    	{
    		if ( $threshold->isMinorThresholdEnabled() )
    		{
    			$threshold->setMinorThresholdEnabled(FALSE);
    		}
    	}
    }
    
    /**
     * 
     * @param SMTOtuMonitoringTestThresholdDto $threshold
     * 
     * @return string Threshold buffer
     */
    private static function encodeThreshold (SMTOtuMonitoringTestThresholdDto $threshold )
    {
    	self::checkMinorThresholdEnabledFlag($threshold);

    	//minor threshold disabled
    	if ( !$threshold->isMinorThresholdEnabled() )
    	{
    		//modify DTO because it will be returned to client without being fetched again (not applied on OTU synchronously)
    		$threshold->setPositiveBackToNormalThreshold($threshold->getPositiveBackToMinorThreshold());
    		$threshold->setNegativeBackToNormalThreshold($threshold->getNegativeBackToMinorThreshold());
    		$threshold->setPositiveToMinorThreshold($threshold->getPositiveToMajorThreshold());
    		$threshold->setNegativeToMinorThreshold($threshold->getNegativeToMajorThreshold());    		
    	}    	
    	
        $buffer = self::convertInt16ToHex( round( $threshold->getNegativeToMinorThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getNegativeToMajorThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getNegativeBackToMinorThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getNegativeBackToNormalThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getPositiveToMinorThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getPositiveToMajorThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getPositiveBackToMinorThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= self::convertInt16ToHex( round( $threshold->getPositiveBackToNormalThreshold() * pow( 10, self::THRESHOLD_DIGITS ) ) );
        $buffer .= sprintf('%02d', $threshold->isMonitored()? SMTOtuMonitoringTestThresholdDto::MONITORED : SMTOtuMonitoringTestThresholdDto::NOT_MONITORED );
        
        return $buffer;
    }
    
    /**
     * convert signed 16 bits integer value to 16 bits hexadecimal values 
     * 
     * @param $intVal
     * @return string
     */
    static function convertInt16ToHex( $intVal )
    {
        $hex = sprintf('%04X', $intVal );
        return substr ($hex, strlen($hex) - 4, 4);//16 bits hexa values 0xFFFF
    }
    
    static function Int32ToHex($intVal)
    {
    	$firt = ($intVal & 0xFF000000) >> 24;
    	$second = ($intVal & 0x00FF0000) >> 16;
    	$third = ($intVal & 0x0000FF00) >> 8;
    	$forth = ($intVal & 0x000000FF) >> 0;
    	return sprintf('%02X',$firt).sprintf('%02X',$second).sprintf('%02X',$third).sprintf('%02X',$forth);
    }
    
    
    function hexToInt32($hexVal)
    {
    	$firtHex = substr($hexVal, 0, 2);
    	$secondHex = substr($hexVal, 2, 4);
    	$thirdHex = substr($hexVal, 4, 6);
    	$forthHex = substr($hexVal, 6, 8);
    	
    	$first = (int)hexdec($firtHex) & 0x000000FF << 24;
    	$second= (int)hexdec($secondHex) & 0x000000FF << 16;
    	$third= (int)hexdec($thirdHex) & 0x000000FF << 8;
    	$forth= (int)hexdec($forthHex) & 0x000000FF << 0;
    	
    	return $first + $second + $third + $forth;
    }
    
    /**
     * Decode the buffer threshold string into SMTOtuMonitoringTestThresholdDto dtos
     *
     * @param int $positionCount
     */
    function decodeThresholdsBuffer( $positionCount )
    {        
        //example: #3170 => $sizeOfBufferSize=3  $bufferSize=170
        $sizeOfBufferSize = intval( substr($this->thresholdsBufferWithHeader, 1, 1) );
        $bufferSize =  intval( substr($this->thresholdsBufferWithHeader, 2, $sizeOfBufferSize) );
        
        
    	$index = strlen( "#" ) + 1 + $sizeOfBufferSize;    	
    	$this->firstMarkerThreshold = self::decodeThresholdBuffer( substr( $this->thresholdsBufferWithHeader, $index, self::THRESHOLD_STRUCT_SIZE) );
    	$index += self::THRESHOLD_STRUCT_SIZE;
    	$this->deltaFirstLastMarkerThreshold = self::decodeThresholdBuffer( substr( $this->thresholdsBufferWithHeader, $index, self::THRESHOLD_STRUCT_SIZE) );
    	$index += self::THRESHOLD_STRUCT_SIZE * (self::TOPAZ_THRESHOLD_COUNT-1);

    	$this->peakThresholds = array();
    	$pos = 0;
  		while ( ($pos < $positionCount) && ($index < $bufferSize) )
    	{
    		$peakThreshold = self::decodeThresholdBuffer( substr( $this->thresholdsBufferWithHeader, $index, self::THRESHOLD_STRUCT_SIZE) );
    		array_push($this->peakThresholds, $peakThreshold);
    		$index += self::THRESHOLD_STRUCT_SIZE;    		
    		$pos++;
    	}    	
    }
    
    /**
     * Decode the buffer threshold string into SMTOtuMonitoringTestThresholdDto dtos
     *
     */
    function decodeHighsensMonitoringThresholdBuffer()
    {
    	//example: #3170 => $sizeOfBufferSize=3  $bufferSize=170
    	$sizeOfBufferSize = intval( substr($this->thresholdsBufferWithHeader, 1, 1) );
    	$bufferSize =  intval( substr($this->thresholdsBufferWithHeader, 2, $sizeOfBufferSize) );
    	    	
    	$index = strlen( "#" ) + 1 + $sizeOfBufferSize;
    	$this->monitoringThreshold= self::decodeThresholdBuffer( substr( $this->thresholdsBufferWithHeader, $index, self::THRESHOLD_STRUCT_SIZE) );
    }
    
    /**
     * Decode the buffer threshold string into SMTOtuMonitoringTestThresholdDto dtos
     *
     * @param int $positionCount
     */
    function decodeHighsensPeakThresholdsBuffer( $thresholdPeakBuffer, $positionCount )
    {
    	//example: #3170 => $sizeOfBufferSize=3  $bufferSize=170
    	$sizeOfBufferSize = intval( substr($thresholdPeakBuffer, 1, 1) );
    	$bufferSize =  intval( substr($thresholdPeakBuffer, 2, $sizeOfBufferSize) );
    	
    	
    	$index = strlen( "#" ) + 1 + $sizeOfBufferSize;   	
    	$this->peakThresholds = array();
    	$pos = 0;
    	while ( ($pos < $positionCount) && ($index < $bufferSize) )
    	{
    		$peakThreshold = self::decodeThresholdBuffer( substr( $thresholdPeakBuffer, $index, self::THRESHOLD_STRUCT_SIZE) );
    		array_push($this->peakThresholds, $peakThreshold);
    		$index += self::THRESHOLD_STRUCT_SIZE;
    		$pos++;
    	}
    }
    
    /**
     * @return \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getFirstMarkerThresholdsDb()
    {
        return $this->firstMarkerThreshold;
    }

    /**
     * @return \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getDeltaFirstLastMarkersThresholdsDb()
    {
    	return $this->deltaFirstLastMarkerThreshold;
    }    
    
    /**
     * Used for highsens: monitoring thresholds
     * 
     * @return \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getMonitoringThresholdsDb()
    {
    	return $this->monitoringThreshold;
    }
    
    /**
     * Return the hysteresis used in thresholds
     * 
     * @param float $defaultHysteresisValue
     * 
     * @return number the hysteresis used in thresholds
     */
    function getHysteresis( $defaultHysteresisValue)
    {
    	$hysteresis = $defaultHysteresisValue;
    	
    	if ( $this->monitoringThreshold != NULL)
    	{
    		$hysteresis= $this->monitoringThreshold->getPositiveToMinorThreshold() - $this->monitoringThreshold->getPositiveBackToNormalThreshold();
    	}
    	else if ( $this->deltaFirstLastMarkerThreshold!= NULL)
    	{
    		$hysteresis= $this->deltaFirstLastMarkerThreshold->getPositiveToMinorThreshold() - $this->deltaFirstLastMarkerThreshold->getPositiveBackToNormalThreshold();
    	}
    	
    	return $hysteresis;
    }
    
    
    /**
     * @return array of \app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    function getPeakThresholds()
	{
		return $this->peakThresholds;
    }
    
    /**
     * Decode threshold buffer values into the SMTOtuMonitoringTestThresholdDto
     *
     * @param string $thresholdBuffer
     * 
     * @return app\services\monitoring\SMTOtuMonitoringTestThresholdDto
     */
    private static function decodeThresholdBuffer( $thresholdBuffer )    
    {
        $threshold = new SMTOtuMonitoringTestThresholdDto();
        $index = 0;        
        
        $threshold->setNegativeToMinorThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setNegativeToMajorThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setNegativeBackToMinorThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setNegativeBackToNormalThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setPositiveToMinorThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setPositiveToMajorThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setPositiveBackToMinorThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );
        $threshold->setPositiveBackToNormalThreshold( self::convertHex16ToFloat( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index++, self::THRESHOLD_VALUE_SIZE ) ) / pow( 10, self::THRESHOLD_DIGITS ) );                
        $threshold->setMonitored( SMTOtuMonitoringTestThresholdDto::MONITORED ==  hexdec( substr( $thresholdBuffer, self::THRESHOLD_VALUE_SIZE * $index, self::THRESHOLD_VALUE_MONITORED_SIZE ) ) );
        
        $threshold->setMinorThresholdEnabled(TRUE);
        self::checkMinorThresholdEnabledFlag($threshold);
        
        return $threshold;
    }
  
    /**
     * Convert hexadecimal string value to float.
     *
     * @param $hexStringVal
     * @return float
     */
    static function convertHex16ToFloat( $hexStringVal )
    {
        $decValue = hexdec( $hexStringVal);
        if ( $decValue > hexdec("OxFFFF") / 2 )
        {
            $decValue = $decValue - hexdec("Ox00010000");
        }
        
    	return floatval( $decValue );
    }
    
}


?>