<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\maintenance;

use app\util\upload\SMTUploadStatus;

use app\sharedmemory\SMTMemoryManager;

use app\services\maintenance\SMTUpgradeDto;

use app\serviceshelper\SMTServiceHelper;

use app\http\SMTContext;

use app\util\SMTLogger;
use app\admin\SMTOtuType;
use app\parser\SMTOtuApi;
use app\util\SMTUtil;

/**
 * Handles upgrade function
 *
 * @author Sylvain Desplat
*/
class SMTUpgrade extends SMTServiceHelper
{    
    const OTU_CURRENT_UPGRADE_DIR = "/otu/release/current/";
    
    const OTU_UPGRADE_LOCK_FILE = "/tmp/otu_upgrade_lock";
    
    const OTU_UPGRADE_PROGRESS_FILE = "/otu/release/current/upgrade_status";
    
    const OTU_DUPLICATION_PROGRESS_FILE = "/otu/release/current/duplication_status";
    
    const OTU_RELEASE_NAME = "*otu-8kv2*.tar*";
    
    const COTU_RELEASE_NAME = "*5000*.tar*";
    
    const OTH_RELEASE_NAME = "fth*.tar*"; 

    const OTU_RELEASE_CURRENT_DIR = "/otu/release/current/";
    
    const OTU_PATCH_CURRENT_DIR = "/otu/release/patch/";
    
    const OTU_UPGRADE_VERSION_COMMAND = 'export DETACHED=1; sudo /sbin/start_upgrade_otu.sh --tar "%s" --upgradeversion';
    
    const OTU_UPGRADE_NORMAL_FS_COMMAND = 'sudo /sbin/start_upgrade_otu.sh --tar "%s" --short --nodet';
    
    const OTU_UPGRADE_RESCUE_FS_COMMAND = 'sudo /sbin/start_upgrade_otu.sh --tar "%s" --target rescue --index rescue --nomd5 --nodet';
    
    const DEFAULT_MAX_RELEASE_COUNT = 2;
    
    /**
     * Retrieve the name of the current upgrade archive
     * 
     * @return current upgrade or NULL if none is found
     */
    public static function getCurrentUpgradeName()
    {
        $currentUpgradeName = NULL;        
        $count = 0;
        
        //retrieve file name
        $resultArray = (SMTOtuType::isCOtu_EOTDRV2())? glob( self::OTU_CURRENT_UPGRADE_DIR.self::COTU_RELEASE_NAME) : ((SMTOtuType::isOTH())? glob( self::OTU_CURRENT_UPGRADE_DIR.self::OTH_RELEASE_NAME) : glob( self::OTU_CURRENT_UPGRADE_DIR.self::OTU_RELEASE_NAME ) );
        if ( $resultArray != FALSE && !empty( $resultArray ) )
        {
            $upgradeFiles = array();
            foreach ( $resultArray as $file )
            {
            	$upgradeFiles[ $file ] = filemtime( $file );
            	$count++;
            }
            
            //remove old releases
            if ( $count > self::DEFAULT_MAX_RELEASE_COUNT )
            {
                $filesCountToRemove =  $count - self::DEFAULT_MAX_RELEASE_COUNT;
                asort( $upgradeFiles, SORT_NUMERIC );
                for ( $i = 0; $i < $filesCountToRemove; $i++)
                {
                    reset( $upgradeFiles );
                	$fileName = key( $upgradeFiles );
                	unlink($fileName);
                	array_shift( $upgradeFiles );
                }
            }
                        
            arsort( $upgradeFiles, SORT_NUMERIC );
            reset( $upgradeFiles );
            $fileName = key( $upgradeFiles );            
            SMTLogger::getInstance()->trace("file:".$fileName." filemtime:".$upgradeFiles[$fileName], SMTLogger::DEBUG );
        	$currentUpgradeName = basename($fileName);
        }
        
        return $currentUpgradeName;
    }

    /**
     * Retrieve current upgrade info from APC cache
     *
     * @return SMTUpgradeDto or NULL if no upgrade is in progress
     */
    public static function getCurrentUpgradeInfoFromCache()
    {
    	$currentUpgradeInfo = NULL;
    	//retrieve current Upgrade from memory:
    	$dtos = SMTMemoryManager::fetchAll( SMTUpgradeDto::getClass() );
    	if ( is_array( $dtos ) && count( $dtos ) > 0 )
    	{
    		reset($dtos);
    		$first_key = key($dtos);
    		$currentUpgradeInfo = $dtos[$first_key];
    	}
    	//     	SMTLogger::getInstance()->trace( "SMTMeasureDataDto: ". var_export( $currentMeasureInfo, true ), SMTLogger::DEBUG );
    	return $currentUpgradeInfo;
    }         
    
    /**
     * Check whether an upgrade is in progress
     *
     * @return boolean
     */
    public static function checkNoUpgradeRunning()
    {
    	$noUpgradeRunning = TRUE;
    
    	if ( file_exists( self::OTU_UPGRADE_LOCK_FILE ))
    	{
    	    $noUpgradeRunning = FALSE;
    	}
    
    	return $noUpgradeRunning;
    }    
    
    /**
     * Retrieve the name and the status of the current upgrade/upload and refresh the status in the APC cache
     *
     * @return SMTUpgradeDto
     */
    public static function getCurrentUpgrade($fetchVersion = TRUE)
    {
		$upgradeDto = new SMTUpgradeDto();

		$currentUpgradeName = self::getCurrentUpgradeName();
		$inProgress = FALSE;
		$inProgressCache = FALSE;
		$cacheDto = self::getCurrentUpgradeInfoFromCache();		
		$currentUpgradeVersion = "";
		
		//check that an upload or upgrade is not in progress
		if ( $cacheDto != NULL )
		{
			$inProgressCache = $cacheDto->isUpgradeInProgressOnOtu();		
		}
		
		//check that an upload was not launched on OTU
		$inProgress = !SMTUploadStatus::checkNoUploadRunning();
		
		//check that an upgrade was not launched on OTU
		if ( !$inProgress )
		{
	        $inProgress = !self::checkNoUpgradeRunning();
	        //in cache, we found that an upgrade is in progress but we don't find the upgrade lock file,
	        //try a second time to let the upgrade script start or to handle the case of the cache not cleared after an upgrade: case of rescue? (very unlikely to happen )
	        if ( $inProgressCache != FALSE && !$inProgress )
	        {
	            sleep( 1 );
	            $inProgress = !self::checkNoUpgradeRunning();
	        }	        	        
		}				

		// fetch archive info if an upgrade is not in progress
		if ( !$inProgress )
		{
			if ( $fetchVersion )
			{
				$upgradeFullPath = self::OTU_CURRENT_UPGRADE_DIR.$currentUpgradeName;
				$upgradeVersionCommand = sprintf( self::OTU_UPGRADE_VERSION_COMMAND, $upgradeFullPath );
				
				$success=-1;
				$infoVersion= trim( exec ($upgradeVersionCommand, $output, $success) );
				if ( $success == 0 )
				{
					$currentUpgradeVersion = $infoVersion;
				}
				else
				{
					$currentUpgradeVersion = "";
				}
			}
		}
		else 
		{
			if ( $cacheDto != NULL )
			{
			$currentUpgradeName = $cacheDto->getCurrentUpgradeName();
			$currentUpgradeVersion = $cacheDto->getCurrentUpgradeVersion();
			}
		}
		
		$upgradeDto->setCurrentUpgradeName( $currentUpgradeName );
		$upgradeDto->setCurrentUpgradeVersion( $currentUpgradeVersion);
		$upgradeDto->setIsUpgradeInProgressOnOtu( $inProgress );	
		
		$upgradeDto->save();
    		
    	return $upgradeDto;
    }
    
    /**
     * Updates the upgrade dto progress with current upgrade log file content.
     * Updates the upgrade dto error status according to the information parsed in the log.
     * 
     * @param $upgradeDto SMTUpgradeDto
     * 
     */
    public static function getCurrentUpgradeProgressLog( SMTUpgradeDto $upgradeDto, SMTContext $context )
    {
        //check if an upgrade progress file is present:
        if ( file_exists( self::OTU_UPGRADE_PROGRESS_FILE ) )
        {
            $file = fopen( self::OTU_UPGRADE_PROGRESS_FILE,'r');
            $row = 0;
            if ( $file != FALSE)
            {
	            while( !feof( $file ) ) 
	            {
	            	$line = fgets( $file );
	            	//frist row contains the upgrade start date
	            	if ( $row++ == 0 )
	            	{
	            	    $upgradeDto->setLastUpgradeStartDate( $line );
	            	    $upgradeDto->addUpgradeProgressLog( trim($line) );
	            	}
	            	else
	            	{
	            		$upgradeDto->addUpgradeProgressLog( trim($line) );  	    
	            	}        	
	            }
	            fclose($file);
            }
            
            //retrieve last operation logged if upgrade is not in progress:
            if ( !$upgradeDto->isUpgradeInProgressOnOtu() )
            {
                $upgradeArray = $upgradeDto->getUpgradeProgressLogArray();
                if ( count( $upgradeArray ) > 0 )
                {                                 
                    $line = end( $upgradeArray );
                    if ( strpos( $line, "SUCCESS" ) !== FALSE )
                    {
                        $upgradeDto->setUpgradeError( SMTUpgradeDto::UPGRADE_SUCCESSFULL );
                    }
                    else if ( (strpos( $line, "FAILURE" ) !== FALSE) || (strpos( $line, "CANCELLED" ) !== FALSE) )
                    {
                        $upgradeDto->setUpgradeError( SMTUpgradeDto::UPGRADE_FAILED );
                        
                        // upgrade couldn't be started succesfully, cancel mode reboot
                        $context->setOTUStarting();
                        // set otu available
                        $context->setOTUAvailable();
                    }
                }
            }                
        }
    }
    
    /**
     * Updates the upgrade dto progress with current duplication log file content.
     * Updates the upgrade dto error status according to the information parsed in the log.
     *
     * @param $upgradeDto SMTUpgradeDto
     *
     */
    public static function getCurrentDuplicationProgressLog( SMTUpgradeDto $upgradeDto, SMTContext $context )
    {
    	//check if a duplication progress file is present:
        if ( (SMTOtuType::isCOtu_EOTDRV2() || SMTOtuType::isOTH() ) && file_exists( self::OTU_DUPLICATION_PROGRESS_FILE ) )
    	{
    		$file = fopen( self::OTU_DUPLICATION_PROGRESS_FILE,'r');
    		$row = 0;
    		if ( $file != FALSE )
    		{
	    		while( !feof( $file ) )
	    		{
	    			$line = fgets( $file );
	    			//frist row contains the duplication start date
	    			if ( $row++ == 0 )
	    			{
	    				$upgradeDto->addUpgradeProgressLog( "********************************************************************************************" );
	    				$upgradeDto->addUpgradeProgressLog( "Duplication" );
	    				$upgradeDto->addUpgradeProgressLog( trim($line) );
	    			}
	    			else
	    			{
	    				$upgradeDto->addUpgradeProgressLog( trim($line) );
	    			}
	    		}
	    		fclose($file);
    		}
    		
    	}
    }
    
    /**
     * Start file system upgrade
     * @param $context SMTContext 
     */
    public static function startFileSystemUpgrade( SMTContext $context )
    {
        $cacheDto = self::getCurrentUpgradeInfoFromCache();
        if ( $cacheDto == NULL )
        {
            $cacheDto = new SMTUpgradeDto();   
            $currentUpgradeName = self::getCurrentUpgradeName();
            $cacheDto->setCurrentUpgradeName( $currentUpgradeName );
        }
        $cacheDto->setIsUpgradeInProgressOnOtu( TRUE );        
        $cacheDto->save();
                
        $upgradeFullPath = self::OTU_CURRENT_UPGRADE_DIR.$cacheDto->getCurrentUpgradeName();        
        $upgradeCommand = sprintf( self::OTU_UPGRADE_NORMAL_FS_COMMAND, $upgradeFullPath );
        SMTLogger::getInstance()->trace( "Start OTU upgrade with command: ".$upgradeCommand, SMTLogger::PINFO );
        
        //remove current upgrade progress file before starting
        if ( file_exists( self::OTU_UPGRADE_PROGRESS_FILE ) )
        {
            unlink( self::OTU_UPGRADE_PROGRESS_FILE );
        }
        
        $context->setOTUUpgrade();
        
        $output = array();
        exec( $upgradeCommand, $output, $success);
                
        $cacheDto->setIsUpgradeInProgressOnOtu( FALSE );
        $cacheDto->setUpgradeError( SMTUpgradeDto::UPGRADE_REBOOT );
        $cacheDto->save();

        //process output
        SMTLogger::getInstance()->trace( print_r($output, TRUE) );
                
        $context->setOTUReboot();
    }
    
    /**
     * Start rescue system upgrade
     * 
     * @param $context SMTContext
     */
    public static function startRescueUpgrade( SMTContext $context )
    {
        $cacheDto = self::getCurrentUpgradeInfoFromCache();
        if ( $cacheDto == NULL )
        {
        	$cacheDto = new SMTUpgradeDto();
        	$currentUpgradeName = self::getCurrentUpgradeName();
        	$cacheDto->setCurrentUpgradeName( $currentUpgradeName );
        }
        $cacheDto->setIsUpgradeInProgressOnOtu( TRUE );
        $cacheDto->save();
                
        $upgradeFullPath = self::OTU_CURRENT_UPGRADE_DIR.$cacheDto->getCurrentUpgradeName();
        $upgradeCommand = sprintf( self::OTU_UPGRADE_RESCUE_FS_COMMAND, $upgradeFullPath );
        SMTLogger::getInstance()->trace( "Start OTU RESCUE upgrade with command: ".$upgradeCommand, SMTLogger::PINFO);
        
        //remove current upgrade progress file before starting
        if ( file_exists( self::OTU_UPGRADE_PROGRESS_FILE ) )
        {
        	unlink( self::OTU_UPGRADE_PROGRESS_FILE );
        }
        
        $context->setOTUUpgrade();
        
        $output = array();
        $result = exec( $upgradeCommand, $output, $success);
                        
        $cacheDto->setIsUpgradeInProgressOnOtu( FALSE );
        $cacheDto->setUpgradeError( SMTUpgradeDto::UPGRADE_REBOOT );
        $cacheDto->save();
        
        //process output
        SMTLogger::getInstance()->trace( print_r($output, TRUE) );        

        $context->setOTUReboot();
    }
    
    /**
     * Check upgrade archive content
     *
     * @param $context SMTContext
     */
    public static function checkUpgradeArchive( $releasePathFileName, SMTContext $context )
    {
    	$success = FALSE;
    	try
    	{
    		// on ne verifie l'archive qu'en enhanced security car la verification est longue
    		if (SMTLicence::isHighSecurityMode($context))
    		{
    			$socket = $context->getOtuSocket();
    			$socket->send( SMTOtuApi::checkSoftwareUpgrade($releasePathFileName),10000000, 120);
    		}
    		$success = TRUE;
    	}
    	//don't throw exception return error
    	catch ( \Exception $e )
    	{
    		$success = FALSE;
    		SMTLogger::getInstance()->traceException( $e );
    	}
    	return $success;
    }
}
?>