<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI Solutions
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright VIAVI Solutions 2022. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\http\SMTISerializable;
use app\util\SMTRwDto;
use app\util\SMTUtil;
use app\util\SMTInvalidValueException;

/**
 * Dto with DMUX infos (DMUX used as ROTAU)
 * 
 * @author Sylvain Desplat
 */
class SMTDmuxDto extends SMTRwDto implements SMTISerializable
{
    /**
     * DMUX index 1 to 36
     */
    protected $index = 0;
    /**
     * DMUX$index 48
     * @var string
     */
    protected $name = '';
    /**
     * @var number
     */
    protected $output = 0;
    /**
     * @var string
     */
    protected $grid = ''; //C50,C100,C200 SMTDWDMGridValues
		
	public function getName()
	{
		return $this->name;
	}
	
	public function getIndex()
	{
	    return $this->index;
	}
	
	public function getOutput()
	{
	    return $this->output;
	}
	
	public function getGrid()
	{
	    return $this->grid;
	}
	
	public function setName($name)
	{
	    $this->name = $name;
	}
	
	public function setIndex($index)
	{
	    if ( !isset($index) || $index === NULL || !SMTUtil::isInt( $index ) )
	    {
	        throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "dmux index", $index );
	    }
	    $this->index = SMTUtil::convertToInt( $index );
	}
	
	public function setOutput($output)
	{
	    if ( !isset($output) || $output === NULL || !SMTUtil::isInt( $output ) )
	    {
	        throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "dmux output", $output );
	    }
	    $this->output = SMTUtil::convertToInt( $output );
	}
	
	public function setGrid($grid)
	{
        $this->grid = $grid;
	}
	
	function isIndexValid()
	{
	    return $this->getIndex() > 0;
	}
	
	function isOutputValid()
	{
	    return $this->getOutput() > 0 && $this->getOutput() <= 96;
	}
	
	function isGridValid()
	{
	    return SMTDWDMGridValues::Is_Valid_Grid($this->getGrid());
	}
	
	function isValid()
	{
	    return $this->isIndexValid() && $this->isOutputValid() && $this->isGridValid();
	}
	
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    static function buildDmux($index, $name, $output, $grid)
    {
        $dto = new SMTDmuxDto();
        $dto->setIndex($index);
        $dto->setName($name);
        $dto->setOutput($output);
        $dto->setGrid($grid);        
        return $dto;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\setup\SMTSwitchDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }
    
    /**
     * Serialize object stored in session
     *
     * @return multitype:string
     */
    public function __sleep()
    {
    	return $this->getSerializableProperties();
    }
    
    /**
     * Returns the serializable properties of the object.
     *
     * @see \app\http\SMTISerializable::getSerializableProperties()
     */
    public function getSerializableProperties()
    {
    	$objectProperties = get_object_vars($this);
    	return array_keys( $objectProperties );
    }
    
    /**
     * Deserialize object.
     */
    public function __wakeup()
    {
    }    
}
?>