<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\parser;

use app\serviceshelper\maintenance\SMTLicenseList;

/**
 * Otu parser commands (commands imported from web interface otu_dico.php)
 * 
 * @author Sylvain Desplat
 */
class SMTOtuApi
{
    /**
     * OTU parser default constant parameters
     */
    const EMPTY_BUFFER = '#10';
    
    /**
     * Media 1: Email
     *
     * @var string
     */
    const MEDIA_1 = 'ALT1';
    
    /**
     * Media 2: SMS
     *
     * @var string
     */
    const MEDIA_2 = 'ALT2';
    
    /**
     * Media 3: SNMP
     *
     * @var string
     */
    const MEDIA_3 = 'ALT3';
    
    /**
     * internal modem for SMS
     * 
     * @var string
     */
    const MODEM_INTERNAL_1 = 'INT1';
        
    /**
     * Email media.
     *
     * @var string
     */
    const EMAIL_MEDIA = 'EMAIL';
    
    /**
     * Sms media.
     *
     * @var string
     */
    const SMS_MEDIA = 'SMS';

    /**
     * SNMP media.
     *
     * @var string
     */
    const SNMP_MEDIA = 'SNMP';
    
    /**
     * Only one SNMP config
     * @var integer
     */
    const SNMP_INDEX = 1;

    /**
     * Only one web service config
     * @var integer
     */
    const WEBSERVICE_INDEX = 1;
        
    /**
     * Number of retry in case of failure on an alert media.
     * Increase retry count because OTU try all the media once for an alarm and then switch to the next alarm; it will come back to the firt alarm after.
     * 
     * @var integer
     */
    const MEDIA_RETRY_COUNT = 50;
            
    /**
     * Delay between retry in case of failure on an alert media (600s)
     *
     * @var integer
     */
    const MEDIA_RETRY_DELAY = 600; //10mn
    
    const LOG_LEVEL_ERROR = 'ERR';
    const LOG_LEVEL_INFO = 'INFO';
    const LOG_LEVEL_DEBUG = 'DEBUG';
    
    /**
     * OTU PARSER RESULT STATUS
     */
    const RES_OK = 'OK';
    const RES_KO = 'KO';
    const RES_ON = 'ON';
    const RES_OFF = 'OFF';
    const RES_MISSCONF = 'MISSconf';
    const RES_MISSDET = 'MISSdet';
    const RES_SAME = 'SAME';
    const RES_DIFF = 'DIFF';
    const RES_NO = 'NO';
    const RES_NONE = 'NONE';
    const RES_WDM = 'WDM';
    const RES_UNKNOWN = '???';
    const RES_UNKNOWN_AF = 'Unknown AF';
    const RES_YES = 'YES';
    const RES_MOD = 'MOD';
    const RES_ALL = 'ALL';
    const RES_WEB = 'WEB';
    const RES_READY = 'READY';
    const RES_VALID = 'VALID';
    const FIREWALL_OFF_OLD = 'ENABLE';
    const FIREWALL_ON_OLD = 'DISABLE';
    const FIREWALL_OFF = 'DISABLE';
    const FIREWALL_ON =  'DEFAULT';
    const FIREWALL_CUSTOM = 'CUSTOM';
    const RES_OPT = 'OPT';
    const RES_SYST = 'SYST';
    const RES_BOTH = 'BOTH';
    const RES_DISABLED = 'DISABLED';
    const RES_ENABLED = 'ENABLED';
    const LEGACY_MODE = 'LEG';
    const HIGHSENS_MODE = 'HIGH';
    
    const RES_WAIT   ='WAIT';
    const RES_ENABLE = 'ENABLE';
    const RES_DISABLE = 'DISABLE';
    
    const OTU_MAX_ADDITIONAL_INFO_LENGTH = 30;
    
    /**
     * OTU PARSER COMMANDS
     */    
    //system
    const CMD_system_restart = 'OTU:SYSTEM:RESTART';
    const CMD_system_reboot = 'OTU:SYSTEM:REBOOT';
    const CMD_system_local = 'OTU:SYSTEM:COM:LOCAL?';
    const CMD_otu_ready = 'OTU:READY?';
    /**
     * @return string //STARTING or READY
     */
    static function isOTUReady()
    {
    	return self::CMD_otu_ready;
    }             
    
    const CMD_get_otu_dateTime = 'OTU:SYSTEM:LDATE?';
    /**
     * @return string //YYYY,MM,DD,HH24,mm,SS
     */
    static function getOtuDate()
    {
    	return self::CMD_otu_dateTime;    
    }
    const CMD_get_otu_dateTime_tz = 'OTU:SYSTEM:TZDATE?';
    /**
     * @return string //mm
     */
    static function getOtuDateTZ()
    {
        return self::CMD_get_otu_dateTime_tz;
    }
    /**
     * YYYY // MM // DD // HH24 // mm // SS
     */    
    const CMD_set_otu_dateTime = 'OTU:SYSTEM:LDATE %d,%d,%d,%d,%d,%d';
    /**
     * @param int $YYYY
     * @param int $MM
     * @param int $DD
     * @param int $HH24
     * @param int $mm
     * @param int $SS
     */
    static function setOtuDate( $YYYY, $MM, $DD, $HH24, $mm, $SS )
    {
    	return sprintf( self::CMD_otu_dateTime, $YYYY, $MM, $DD, $HH24, $mm, $SS );
    }    
    
    /**
     * YYYY // MM // DD // HH24 // mm // SS
     */
    const CMD_set_otu_dateTime_tz = 'OTU:SYSTEM:TZDATE %d,%d,%d,%d,%d,%d,%d';
    /**
     * @param int $YYYY
     * @param int $MM
     * @param int $DD
     * @param int $HH24
     * @param int $mm
     * @param int $SS
     */
    static function setOtuDateTz( $YYYY, $MM, $DD, $HH24, $mm, $SS, $TZ )
    {
        return sprintf( self::CMD_set_otu_dateTime_tz, $YYYY, $MM, $DD, $HH24, $mm, $SS, $TZ );
    }  
    //TYPE d'OTU (COTU,8KV2,EOTDRV2,MTS8KV2)
    const CMD_OTU_TYPE = 'OTU:HWABILITY? NAME';
    /**
     * @return string
     */
    static function getOTUType()
    {
    	return self::CMD_OTU_TYPE;
    }
    
    //modules configuration:
    const CMD_detect_config = 'OTU:MODU:DETECT?';
    const CMD_get_config = 'OTU:MODU:CONFIG?';
    /** 
     * @return string
     */
    static function getModuleConfig()
    {
    	return self::CMD_get_config;
    }    
    const CMD_config = 'OTU:MODULE:CONFIG';
        
    const CMD_get_otu_ser = 'OTU:SER?';
    /**
     * MOD2 or MOD1
     */
    const CMD_get_NMOD = 'OTU:MODU:CALSerialnumber? %s';
    /**
     * MOD2 or MOD1
     */    
    const CMD_get_NMOD_Detected = 'OTU:MODU:DETSerialnumber? %s';
//     /**
//      * MOD2 or MOD1
//      */
//     const CMD_get_MODName = 'OTU:MODU:CALName? %s';
//     /**
//      * MOD2 or MOD1
//      */
//     const CMD_get_MODName_Detected = 'OTU:MODU:DETName? %s';

    /**
     * MOD3, MOD2 or MOD1
     */
    const CMD_Is_MOD_Present_In_Cal = 'OTU:MODU:CALPresent? %s';
    /**
     * @param string $module MOD1, MOD2 or MOD3
     * @return string
     */
    static function getModulePresentInCal( $module )
    {
    	return sprintf( self::CMD_Is_MOD_Present_In_Cal, $module );
    }
    
    const CMD_Get_Min_Attenuation_Alarm_Threshold= 'OTU:MON:ATT:TMIN? %s';
    
    /**
     * @param string $module MOD1, MOD2 or MOD3
     * @return string  
     */
    static function getMinAttenuationAlarmThreshold($module)
    {
    	return sprintf( self::CMD_Get_Min_Attenuation_Alarm_Threshold, $module );
    }
    const CMD_Get_Min_Hysteresis_Attenuation_Alarm_Threshold= 'OTU:MON:ATT:HYSM? %s';
    
    /**
     * @param string $module MOD1, MOD2 or MOD3
     * @return string
     */
    static function getMinHysteresisAttenuationAlarmThreshold($module)
    {
    	return sprintf( self::CMD_Get_Min_Hysteresis_Attenuation_Alarm_Threshold, $module );
    }
    
    const CMD_Get_List_Hysteresis_Attenuation_Alarm_Threshold= 'OTU:MON:ATT:HYSL? %s';
    
    /**
     * @param string $module MOD1, MOD2 or MOD3
     * @return string
     */
    static function getHysteresisAttenuationAlarmThresholdList($module)
    {
    	return sprintf( self::CMD_Get_List_Hysteresis_Attenuation_Alarm_Threshold, $module );
    }
    
    const CMD_Get_Critical_Threshold='OTU:MON:CUT? %d';
    static function getGet_Critical_Threshold($testId)
    {
    	return sprintf( self::CMD_Get_Critical_Threshold, $testId);
    }
    
    /**
     * MOD3, MOD2 or MOD1
     */
    const CMD_Is_MOD_Present_In_Det = 'OTU:MODU:DETPresent? %s';
    /**
     * @param string $module MOD1, MOD2 or MOD3
     * @return string
     */
    static function getModulePresentInDet( $module )
    {
    	return sprintf( self::CMD_Is_MOD_Present_In_Det, $module );
    }
        
    /**
     * MOD3, MOD2 or MOD1
     */    
    const CMD_get_Funct = 'OTU:MODU:CALFU:LIST? %s';
    /**
     * @param string $module MOD1 or MOD2
     * @return string 
     */
    static function getModuleFunctions( $module )
    {
    	return sprintf( self::CMD_get_Funct, $module );
    }
    
//     /**
//      * MOD2 or MOD1, function, L1 or L2...  
//      */   
//     const CMD_get_Lambda = 'OTU:MODU:CALFR? %s,"%s",%s';      
    /**
     * MOD2 or MOD1
     */    
    const CMD_get_Funct_Detected = 'OTU:MODU:DETFU:LIST? %s';
//     /**
//      * MOD2 or MOD1, function, L1 or L2...
//      */
//     const CMD_get_Lambda_Detected = 'OTU:MODU:DETFR? %s,"%s",%s';
//     /**
//      * MOD2 or MOD1, function, L1 or L2...
//      */    
//     const CMD_get_Injection = 'OTU:MODU:DETLaser:INJT? %s,"%s",%s';   
    /**
     * MOD2 or MOD1, function command
     * @returns minGhz,maxGhz,spacingGhz
     */
    const CMD_get_Frequency_Setup = 'OTU:DWDM:PARAMETERS? %s';
    /**
     * @param string $module MOD1 or MOD2
     * @returns minGhz,maxGhz,spacingGhz
     */
    static function getModuleFrequencyParameters( $module )
    {
    	return sprintf( self::CMD_get_Frequency_Setup, $module );
    }   
    /**
     * MOD2 or MOD1, function command
     * @returns 1625,...
     */
    const CMD_get_Laser_List_SI = 'OTU:MODU:CALLaser:SI:LIST? %s,"%s"';
    /**
     * @param string $module MOD1 or MOD2
     * @param string $function
     * @return string 1625,...
     */
    static function getModuleWavelengths( $module, $function )
    {
    	return sprintf( self::CMD_get_Laser_List_SI, $module, $function );
    }    
    
    /**
     * MOD2 or MOD1, function
     */    
    const CMD_get_Laser_List_Detected_SI = 'OTU:MODU:DETLaser:SI:LIST? %s,"%s"';
    /**
     * MOD3, MOD2 or MOD1
     */
    const CMD_check_MOD = 'OTU:MODU:CHECK? %s';

    
    const CMD_get_MOD_Name = 'OTU:MODU:NAME? %s';
    /**
     * @param $moduleId MOD3, MOD2 or MOD1
     * @return string
     */
    static function getModuleName($moduleId)
    {
    	return sprintf( self::CMD_get_MOD_Name, $moduleId );
    }
    
    const CMD_check_module_compatibility_with_switch = 'OTU:MODU:SWITCH:COMP?';
    //modems configuration:
//     const CMD_modem_media_config = 'OTU:MED:GET? BACK';
//     const CMD_modem_speed = 'otu:modem:conf:spee? %s'; //INT1 ...
//     const CMD_modem_pre_init = 'otu:modem:conf:atid? %s'; //INT1 or EXT1
//     const CMD_modem_init = 'otu:modem:conf:atic? %s'; //INT1 or EXT1
//     const CMD_modem_connect = 'otu:modem:conf:atc? %s'; //INT1 ...
//     const CMD_modem_deconnect = 'otu:modem:conf:atd? %s'; //INT1 ...
//     const CMD_default_conf = 'OTU:MODEM:CONF:DEF %s';//INT1 retrieve default config   
//     const CMD_test_network_quality = 'OTU:PROD:MODEM:GSMQ? INT1';
    
    const CMD_modem_config = 'OTU:MODEM:CONFIG:TYPE? %s'; //INT1 or EXT1
    
    static function getModemConfig()
    {
    	return sprintf( self::CMD_modem_config, self::MODEM_INTERNAL_1 );
    }
    
    /**
     * INT1
     */
    const CMD_modem_media_ready = 'OTU:MODEM:READY? %s';
    
    static function getModemReady()
    {
    	return sprintf( self::CMD_modem_media_ready, self::MODEM_INTERNAL_1 );
    }    
    
    const CMD_get_saa_access_code = 'SYST:SAA?';
    /**
     * @return string
     */
    static function getSaaAccessCode()
    {
    	return self::CMD_get_saa_access_code;
    }    
    
    const CMD_saa_connect = 'SYST:SAA CONNECT';
    /**
     * @return string
     */
    static function getSaaConnect()
    {
    	return self::CMD_saa_connect;
    }    
    
    const CMD_saa_disconnect = 'SYST:SAA DISCONNECT';
    /**
     * @return string
     */
    static function getSaaDisconnect()
    {
    	return self::CMD_saa_disconnect;
    }    
    
//     const CMD_modem_config_user = 'OTU:MODEM:CONFIG:USER? EXT1';
//     const CMD_modem_config_password = 'OTU:MODEM:CONFIG:PASS? EXT1';
    
     //config update
//     const CMD_update_modem_detect_config = 'OTU:MODEM:CONF:TYPE %s,%s'; //INT1 ...,  NONE, PSTN, ISDN, GSM
//     const CMD_update_modem_speed = 'otu:modem:conf:spee %s,%s'; //INT1 ..., AUTO, V38400, V57600, V115200
//     const CMD_update_modem_init_0 = 'otu:modem:conf:atid %s,"%s"'; //INT 1or EXT1, init0
//     const CMD_update_modem_init = 'otu:modem:conf:atic %s,"%s"'; //INT1or EXT1, init
//     const CMD_update_modem_connect = 'otu:modem:conf:atc %s,"%s"'; //INT1 ..., connect_string
//     const CMD_update_modem_deconnect = 'otu:modem:conf:atd %s,"%s"'; //INT1 ...,deconnect_string
//     const CMD_update_modem_config_user = 'OTU:MODEM:CONFIG:USER %s,"%s"';//INT1 or EXT1, user name
//     const CMD_update_modem_config_password = 'OTU:MODEM:CONFIG:PASS %s,"%s"';//INT1 or EXT1, user password
    
    //modem port
//     const CMD_get_port = 'otu:modem:conf:port? %s';
    //retrive tsc login/password
//     const CMD_get_tsc_phone = 'otu:tsc:phone:get? %s';
    //get active ONMSi server
     const CMD_get_active_server = 'otu:tpz:server?';
    
    //start test modem
//     const CMD_start_test_modem = 'otu:modem:test %s,%s,%s,"%s",%s';          
    
    //TOPAZ configuration    
    const CMD_get_otu_BASE_TYPE = 'OTU:BASE?';//8KV2, 8KV1
	const CMD_get_otu_TYPE = 'OTU:CALIB:RFTY?'; //RTU, OTU, PON, ROTAU, OEM, TOPAZ, ONMSI, SMARTOTU        
	const CMD_set_otu_TYPE = 'OTU:CALIB:RFTY %s'; //OTU, ROTAU, OEM, ONMSI or SMARTOTU

	/**
	 * @param string $mode OTU, ROTAU, OEM, ONMSI or SMARTOTU
	 *
	 * @return string to change OTU mode (no longer SmartOtu)
	 */
	static function setOtuRFTYMode( $mode )
	{
		return sprintf( self::CMD_set_otu_TYPE, $mode );
	}
	
	/**
	 * @return string RTU, OTU, PON, ROTAU, OEM, TOPAZ, ONMSI or SMARTOTU
	 */
	static function getOtuRFTYMode()
	{
		return self::CMD_get_otu_TYPE;
	}
	
	const CMD_get_otu_INFO = "*idn?"; 
	
	/**
	 * @return string JDSU,OTU 8000e,04993,OTU(SmartOTU),V2.83
	 */
	static function getOtuInfo()
	{
		return self::CMD_get_otu_INFO;
	}
	
	const CMD_set_log_level = "OTU:TRACE %s";
	
	/**
	 * 
	 * @param String $logLevel (ERR, INFO, DEBUG)
	 * @return string
	 */
	static function changeLogLevel( $logLevel )
	{
	    return sprintf( self::CMD_set_log_level, $logLevel );
	}
	
	const CMD_check_software_upgrade = 'OTU:SOFTWARE:CHECK "%s"';
	
	/**
	 *
	 * @param String $releaseFileName
	 * @return string
	 */
	static function checkSoftwareUpgrade( $releaseFileName )
	{
		return sprintf( self::CMD_check_software_upgrade, $releaseFileName );
	}	
	
	
	const CMD_show_otu = 'OTU:LOCALIZE %s';
	/**
	 * @param string ON,OFF
	 */
	static function setOtuShow($onOff)
	{
		return sprintf(self::CMD_show_otu, $onOff);
	}	
	
	const CMD_get_show_otu = 'OTU:LOCALIZE?';	
	/**
	 * @return string ON,OFF
	 */
	static function getOtuShow()
	{
		return self::CMD_get_show_otu;
	}
	
	const CMD_get_ssh_port = 'otu:tpz:ssh:port?';
	const CMD_get_ssh_port_topaz = 'otu:tpz:ssh:tport?';
	const CMD_is_ssh_active = 'otu:tpz:ssh:act?';	
	const CMD_get_corba_port = 'otu:tpz:corba:port?';

	const CMD_set_ssh_port = 'otu:tpz:ssh:port %s'; // int
    const CMD_set_ssh_port_topaz = 'otu:tpz:ssh:tport %s'; //int
    const CMD_set_ssh_active = 'otu:tpz:ssh:act %s'; // TRUE/FALSE
    const CMD_set_corba_port = 'otu:tpz:corba:port %s'; // int
    
    //phone number
    const CMD_onms_phone_number = 'OTU:TSC:PHONE:GET? %s';

    //relay
    const CMD_relay = 'OTU:RELAY? %s';
    const CMD_update_relay = 'OTU:RELAY %s,%s';
    
    //test option relay
    const CMD_test_option_relay = 'OTU:EPOW:REL?';
  
    //mail info
    const CMD_get_domain = 'OTU:SYSTEM:COM:DOMAIN?';
    const CMD_get_site = 'OTU:SYSTEM:COM:SITE?';    
    /**
     * Email index
     */
//     const CMD_get_email_conf = 'OTU:ALE:EMAIL:GET? %d';
    /**
     * #config//ETH or MOD//login//password//@ from//SmtpServer//@ to//subject
     */
//     const CMD_set_email_conf = 'OTU:ALE:EMAIL:SET %d,%s,"%s","%s","%s","%s","%s","%s"';
    
    /**
     * Email index
     */
    const CMD_get_email_conf_with_attach_file = 'OTU:ALE:EMAIL:CONFIG:GET? %d';
    /**
     * #config//ETH or MOD//login//password//@ from//SmtpServer//smtp port//security protocol//@ to//subject//attachment//filter//port buffer
     */
    const CMD_set_email_conf_with_attach_file = 'OTU:ALE:EMAIL:CONFIG:SET %d,%s,"%s","%s","%s","%s",%d,%s,"%s","%s",%s,%s,%s';
    
    static function getEmailWithAttachedFileCommand( $index ) //index from 1 to 16
    {
    	return sprintf( self::CMD_get_email_conf_with_attach_file, $index );
    }
    /**
     * 
     * @param integer $index
     * @param string $login
     * @param string $password
     * @param string $from
     * @param string $smtServer
     * @param integer $smtPort
     * @param string $smtSecurityProtocol
     * @param string $to
     * @param string $subject
     * @param boolean $attach
     * @param boolean $filter
     * @param string $portBuffer
     * @return string
     */
    static function setEmailWithAttachedFileCommand( $index, $login, $password, $from, $smtServer, $smtPort, $smtSecurityProtocol, $to, $subject, $attach, $filter, $portBuffer ) //index from 1 to 16
    {
        $attachValue = $attach? self::RES_YES : self::RES_NO;
        $filterValue = $filter? self::RES_YES : self::RES_NO;
    	return sprintf( self::CMD_set_email_conf_with_attach_file, $index, "ETH", $login, $password, $from, $smtServer, $smtPort, $smtSecurityProtocol, $to, $subject, $attachValue, $filterValue, $portBuffer);
    }
    
    /**
     * SNMP index (1 uniquement)
     */
    const CMD_get_snmp_conf = 'OTU:ALE:SNMP:GET? %d';
    /**
     * #config=1//SNMPServerAddress//SNMPCommunity//port SNMP
     */
    const CMD_set_snmp_conf = 'OTU:ALE:SNMP:SET %d,"%s","%s",%d';
    
    static function getSnmpCommand($snmpIndex)
    {
    	return sprintf( self::CMD_get_snmp_conf, $snmpIndex );
    }
    /**
     *
     * @param integer $snmpIndex
     * @param string $serverAddress
     * @param string $community
     * @param integer $port
     * 
     * @return string
     */
    static function setSnmpCommand( $snmpIndex, $serverAddress, $community, $port )
    {
    	return sprintf( self::CMD_set_snmp_conf, $snmpIndex, $serverAddress, $community, $port);
    }
    
    /**
     * #config=1//SNMPServerAddress//SNMPCommunity//port SNMP//message
     */
    const CMD_test_snmp = 'OTU:ALE:SNMP:TEST %d,"%s","%s",%d,"%s"';
    
    
    /**
     * Web service index (1 uniquement)
     */
    const CMD_get_web_service_conf = 'OTU:ALE:WEBS:GET? %d';
    /**
     * #config=1//WebServiceUrl//AlarmType
     */
    const CMD_set_webservice_conf = 'OTU:ALE:WEBS:SET %d,"%s",%s';
    
    static function getWebServiceCommand()
    {
    	return sprintf( self::CMD_get_web_service_conf, self::WEBSERVICE_INDEX );
    }
    /**
     *
     * @param string  $webServiceUrl
     * @param integer $alarmType
     *
     * @return string
     */
    static function setWebServiceCommand( $webServiceUrl, $alarmType )
    {
    	return sprintf( self::CMD_set_webservice_conf, self::WEBSERVICE_INDEX, $webServiceUrl, $alarmType);
    }
    
    
    /**
     *
     * @param string $serverAddress
     * @param string $community
     * @param integer $port
     * @param string $message //30 characters
     *
     * @return string
     */
    static function testSnmpCommand( $serverAddress, $community, $port, $message )
    {
    	return sprintf( self::CMD_test_snmp, self::SNMP_INDEX, $serverAddress, $community, $port, $message);
    }
    
    const CMD_set_domain = 'OTU:SYSTEM:COM:DOMAIN "%s"';
    const CMD_set_site = 'OTU:SYSTEM:COM:SITE "%s"';
    /**
     * ALT2 or ALT1
     */
    const CMD_get_media = 'OTU:MEDIA:SNMP:GET? %s';
    
    static function getEmailMedia()
    {
    	return sprintf( self::CMD_get_media, self::MEDIA_1 );
    }
    
    static function getSmsMedia()
    {
    	return sprintf( self::CMD_get_media, self::MEDIA_2 );
    }    
    
    static function getSNMPMedia()
    {
    	return sprintf( self::CMD_get_media, self::MEDIA_3 );
    }
    
    /**
     * Set email max retry
     *  
     * @var integer
     */
    const CMD_set_email_retry = 'OTU:ALE:EMA:RETR %d';
    
    /**
     * Set sms max retry
     *
     * @var integer
     */
    const CMD_set_sms_retry = 'OTU:ALE:SMS:RETR %d';

    /**
     * Set snmp max retry
     *
     * @var integer
     */
    const CMD_set_snmp_retry = 'OTU:ALE:SNMP:RETR %d';    
    
    static function setEmailMaxRetry( $maxRetry )
    {
    	return sprintf( self::CMD_set_email_retry, $maxRetry );
    }
    
    static function setSmsMaxRetry( $maxRetry )
    {
    	return sprintf( self::CMD_set_sms_retry, $maxRetry );
    }
    
    static function setSNMPMaxRetry( $maxRetry )
    {
    	return sprintf( self::CMD_set_snmp_retry, $maxRetry );
    }
    
    /**
     * Set media retry period
     *
     * @var integer
     */
    const CMD_set_media_retry_period = 'OTU:MEDIA:PER %d';
    
    static function setMediaRetryPeriod( $retryPeriod )
    {
    	return sprintf( self::CMD_set_media_retry_period, $retryPeriod );
    }
    
    /**
     * ALT1 or ALT2 or ALT3 // EMAIL or SMS or SNMP // ETH0 or PPP0 // NONE or INT1 // NONE or ALL // NONE or ALL // NONE or ALL // retry count // delay between retry  
     * Warning: the last 2 parameters are not used and must be set by setMediaRetryPeriod and setxxxxMaxRetry
     */
    const CMD_set_media = 'OTU:MEDIA:SNMP:SET %s,%s,%s,%s,%s,%s,%s,%d,%d';

    /**
     * 
     * @param boolean $on whether email media must be activated or not
     * 
     * @return string
     */
    static function setEmailMedia( $on )
    {
        $emailStatusCommand = NULL;
        if ( $on )
        {
            $emailStatusCommand = sprintf( self::CMD_set_media, self::MEDIA_1, self::EMAIL_MEDIA, 'ETH0', self::RES_NONE, self::RES_NONE, self::RES_ALL, self::RES_NONE,10, self::MEDIA_RETRY_DELAY );//the last 2 parameters are not used
        }
        else
        {
            $emailStatusCommand = sprintf( self::CMD_set_media, self::MEDIA_1, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, 10, self::MEDIA_RETRY_DELAY  );//the last 2 parameters are not used
        }
    	
    	return $emailStatusCommand;
    }    
    
    /**
     *
     * @param boolean $on whether sms media must be activated or not
     *
     * @return string
     */    
    static function setSmsMedia( $on )
    {
        $smsStatusCommand = NULL;
        if ( $on )
        {
        	$smsStatusCommand = sprintf( self::CMD_set_media, self::MEDIA_2, self::SMS_MEDIA, 'PPP0', self::MODEM_INTERNAL_1, self::RES_ALL, self::RES_NONE, self::RES_NONE, 10, self::MEDIA_RETRY_DELAY );//the last 2 parameters are not used
        }
        else
        {
        	$smsStatusCommand = sprintf( self::CMD_set_media, self::MEDIA_2, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, 10, self::MEDIA_RETRY_DELAY );//the last 2 parameters are not used
        }        
        
    	return $smsStatusCommand;
    }    
    
    /**
     *
     * @param boolean $on whether SNMP media must be activated or not
     *
     * @return string
     */
    static function setSNMPMedia( $on )
    {
    	$snmpStatusCommand = NULL;
    	if ( $on )
    	{
    		$snmpStatusCommand = sprintf( self::CMD_set_media, self::MEDIA_3, self::SNMP_MEDIA, 'ETH0', self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_ALL, 10, self::MEDIA_RETRY_DELAY  );//the last 2 parameters are not used
    	}
    	else
    	{
    		$snmpStatusCommand = sprintf( self::CMD_set_media, self::MEDIA_3, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, self::RES_NONE, 10, self::MEDIA_RETRY_DELAY  );//the last 2 parameters are not used
    	}
    	 
    	return $snmpStatusCommand;
    }    
    
    
    const CMD_get_heartbeat ='OTU:HEARTBEAT:GET?';
    
    static function isSNMPHearbeat()
    {
    	return self::CMD_get_heartbeat;
    }
    
    const CMD_start_heartbeat ='OTU:HEARTBEAT:START';
    const CMD_stop_heartbeat ='OTU:HEARTBEAT:STOP';
    
    /**
     * 
     * @param boolean $on
     * @return string
     */
    static function setSNMPHearbeat( $on )
    {
    	return $on? self::CMD_start_heartbeat : self::CMD_stop_heartbeat;
    }    
    
    const CMD_get_heartbeat_period ='OTU:HEARTBEAT:PERIOD?';
    /**
     *
     * @return string
     */
    static function getSNMPHearbeatPeriod()
    {
    	return self::CMD_get_heartbeat_period;
    }
    
    const CMD_set_heartbeat_period ='OTU:HEARTBEAT:PERIOD %d';    
    /**
     *
     * @param integer $period
     * @return string
     */
    static function setSNMPHearbeatPeriod( $period )
    {
    	return sprintf( self::CMD_set_heartbeat_period, $period);
    }
    
    /**
     * #config//ETH or MOD// message //login//password//@ from//SmtpServer//smtp port//smtp security//@ to//subject
     */
    const CMD_test_email = 'OTU:ALE:EMAIL:CONF:TEST %d,%s,"%s","%s","%s","%s","%s",%d,%s,"%s","%s" ';
       
    const CMD_get_host_name = 'OTU:SYSTEM:HOST?';
    const CMD_set_host_name = 'OTU:SYSTEM:HOST "%s"';
    
    /**
     * #config Sms index
     */
    const CMD_get_sms_conf = 'OTU:ALE:SMS:GET? %d';
    /**
     * #config//phone number
     */
    const CMD_set_sms_conf = 'OTU:ALE:SMS:SET %d,"%s"';    
    /**
     * #config//phone number//message
     */
//     const CMD_test_sms = 'OTU:ALE:SMS:TEST %d,"%s","%s"';
    const CMD_test_modem_sms = 'OTU:PROD:MODEM:SMS %s,"%s","%s"';//[INT1,EXT1], tel number, sms message
    
   //media
   const CMD_media_test = 'OTU:ALARM:TEST %s';
   
    //IP    
    const CMD_get_ethernet_interface = 'OTU:SYSTEM:COM:INTE?';
    //ETH0
    const CMD_get_ip = 'OTU:SYSTEM:COM:IP? %s';
    //ETH0
    const CMD_get_subnet = 'OTU:SYSTEM:COM:SUBNET? %s';
    const CMD_get_gateway = 'OTU:SYSTEM:COM:NGATEWAY? %s';
    const CMD_get_dns = 'OTU:SYSTEM:COM:DNS?';
    /**
     * TSC1 or TSC2
     * @param int index = 1 or 2
     */
    const CMD_get_onms_address = 'OTU:TSC:GET? TSC%d';
    static function getOnmsiServerAddress( $index )
    {
    	return sprintf( self::CMD_get_onms_address, $index );
    } 

    /**
     * TSC1 or TSC2
     * @param int index = 1 or 2
     */   
    const CMD_get_onms_name = 'OTU:TSC:NGET? TSC%d';
    static function getOnmsiServerName( $index )
    {
    	return sprintf( self::CMD_get_onms_name, $index );
    } 
    
    const CMD_get_network_Interface_Count = 'OTU:SYSTEM:COM:NBINterface?';
    /**
     * LOC,FIX,DHCP,DYN
     */
    const CMD_get_otu_mode = 'OTU:SYST:COM:NMODE? %s'; //ETH0/ETH1
    
    const CMD_get_mtu = 'OTU:SYST:COM:MTU? %s';// ETH0/ETH1
    const CMD_set_mtu = 'OTU:SYST:COM:MTU %s,%d';// ETH0/ETH1
    
    const CMD_get_autoneg = 'OTU:SYST:COM:NEGO? %s';// ETH0/ETH1
    const CMD_set_autoneg = 'OTU:SYST:COM:NEGO %s,%s,%s';// ETH0/ETH1, ,
    
    /**
     * LOC,FIX,DHCP or DYN
     */    
    const CMD_set_otu_mode = 'OTU:SYSTEM:COM:NMODE %s,%s';
    const CMD_get_backup_route = 'OTU:SYSTEM:COM:BACKUP:GATE?';
    const CMD_set_ip = 'OTU:SYSTEM:COM:IP %s,%d,%d,%d,%d';//10,33,16,192
    const CMD_set_subnet = 'OTU:SYSTEM:COM:SUBNET %s,%d,%d,%d,%d';//255,255,252,0
    const CMD_set_gateway = 'OTU:SYSTEM:COM:NGATEWAY %s,%d,%d,%d,%d';
    const CMD_set_dns = 'OTU:SYSTEM:COM:DNS %d,%d,%d,%d';
    /**
     * TSC1 or TSC2 //@IP1 //@IP2 // @IP3
     */    
    const CMD_set_backup_route = 'OTU:SYSTEM:COM:BACKUP:GATE %d,%d,%d,%d';
    const CMD_is_backup_route_active = 'OTU:SYSTEM:COM:BACK:USEG?';
    const CMD_activate_backup_route = 'OTU:SYSTEM:COM:BACK:USEG %s';
    /**
     * TSC1 or TSC2
     */    
//     const CMD_set_tsc_on = 'OTU:TSC:ON %s';

    //IPV6
    const CMD_get_ipv6 = 'OTU:SYSTEM:COM:IPV6:IP? %s';
    const CMD_get_gatewayv6 = 'OTU:SYSTEM:COM:IPV6:NGATEWAY? %s';
    const CMD_get_dnsv6 = 'OTU:SYSTEM:COM:IPV6:DNS?';
    const CMD_get_linkipv6 = 'OTU:SYSTEM:COM:IPV6:LINKIP? %s';
    const CMD_get_sitev6 = 'OTU:SYSTEM:COM:IPV6:SITE?';
    const CMD_set_ipv6 = 'OTU:SYSTEM:COM:IPV6:IP %s,"%s"';
    const CMD_set_gatewayv6 = 'OTU:SYSTEM:COM:IPV6:NGATEWAY %s,"%s"';
    const CMD_set_dnsv6 = 'OTU:SYSTEM:COM:IPV6:DNS "%s"';
    const CMD_set_sitev6 = 'OTU:SYSTEM:COM:IPV6:SITE "%s"';
    const CMD_get_otu_mode_ipv6 = 'OTU:SYSTEM:COM:IPV6:NMODE? %s';
    
    const CMD_set_ipv6_prefix = 'OTU:SYSTEM:COM:IPV6:NPRE %s,%d';
    const CMD_get_ipv6_prefix = 'OTU:SYSTEM:COM:IPV6:NPRE? %s';

    /**
     * FIX,DHCP or OFF
     */
    const CMD_set_otu_mode_ipv6 = 'OTU:SYSTEM:COM:IPV6:NMODE %s,%s';
    
    //OTAU
    const CMD_add_otau = 'OTU:SWITCH:CONFIG:ADD? %s,"%s","%s"'; //device:OTU,IPR / GLOB switch string / rotau address ip
    const CMD_update_rotau_in_config = 'OTU:SWITCH:CONFIG:ROT? %s,"%s","%s"';//specific command update the ROTAU config in the OTU
    const CMD_remove_otau = 'OTU:SWITCH:CONFIG:DEL %s'; 
    const CMD_set_osu_address = 'OTU:SWITCH:OSUS:ADDRESS %s';
    const CMD_set_osx_address = 'OTU:SWITCH:OSXJ:ADDRESS %s';
    const CMD_scan_osu = 'OTU:SWITCH:OSUJ:SCAN?';   // on ne gere plus les OSUs Dicon
    const CMD_get_osuj_desc = 'OTU:SWITCH:OSUJ:DESC? %s';
    const CMD_is_osu_present = 'OTU:SWITCH:OSUS:PRESENT? %s,%s';//N_Osus,ALL/JDSU/DICON     
    const CMD_get_osu_det_type = 'OTU:SWITCH:OSUS:DETType?';
    const CMD_get_osu_conf_type = 'OTU:SWITCH:OSUS:CONFType?';
    
    const CMD_get_osu_desc = 'OTU:SWITCH:OSUS:DESC? %s';
    const CMD_get_osuj_idn = 'OTU:SWITCH:DEBUG:JDSU:IDN? %d';
    const CMD_get_osuj_select = 'OTU:SWITCH:DEBUG:SELECT EXT';
    const CMD_get_osu_ports = 'OTU:SWITCH:OSUS:PORT? %s';    //OSU1/OSU2/...OSU31
    const CMD_get_bigtopo = 'OTU:SWITCH:CONF:BIGT? %d';    //0/1... // Macroswitchid
    const CMD_set_bigtopo = 'OTU:SWITCH:CONF:BIGT %d, %s';    //0/1... forMacroswitchid, TRUE/FALSE
    const CMD_get_Nb_Port_Distributeur = 'OTU:SWITCH:CONFIG:DIST?';
    const CMD_set_Nb_Port_Distributeur = 'OTU:SWITCH:CONFIG:DIST %d';
    const CMD_Has_Fixed_Port_Distributeur = 'OTU:SWITCH:CONFIG:FIXDist?';
    const CMD_show_osx = 'OTU:SWITCH:OSUJ:SHOW %d ';    //0/1... osx address
    const CMD_power_on_off_osx = 'OTU:SWITCH:OSXJ:CONN %s ';//DISABLED/ENABLED
    const CMD_get_power_on_off_osx = 'OTU:SWITCH:OSXJ:CONN?';//DISABLED/ENABLED
    
    const CMD_switch_config_list = 'OTU:SWITCH:CONFIG:LIST?';         
    const CMD_switch_config_desc = 'OTU:SWITCH:CONFIG:DESC? %d,%s'; //0,GLOB//INT/OSU1/OSU2 ...  OSU31
    const CMD_switch_config_port_count = 'OTU:SWITCH:CONFIG:PORTCount?';
    const CMD_switch_detect_list = 'OTU:SWITCH:DETECT:LIST?'; 
    const CMD_switch_detect_desc = 'OTU:SWITCH:DETECT:DESC? %d,%s'; //0,GLOB//INT/OSU1/OSU2 ... OSU31
    const CMD_switch_config_SerialNum = 'OTU:SWITCH:CONFIG:SER? %d,%s'; //0,INT/OSU1/OSU2 ... OSU31
    const CMD_switch_detect_SerialNum = 'OTU:SWITCH:DETECT:SER? %d,%s'; //0,INT/OSU1/OSU2 ... OSU31
    const CMD_switch_detect_calib_SerialNum = 'OTU:SWITCH:CONFIG:DETECT:SER? %d,%s'; //0,INT/OSU1/OSU2 ... ... OSU31//Detection from calibration
    const CMD_get_switch_ports = 'OTU:SWITCH:CONF:PORT? %d';    //0/1/2/3/...31
    const CMD_get_switch_jdsu_idn = 'OTU:SWITCH:OSXJdsu:IDN? %d';
    const CMD_get_switch_jdsu_idn_scan = 'OTU:SWITCH:OSXJdsu:SIDN? %d';    
    const CMD_switch_detect_Serial_Number_List = 'OTU:SWITCH:OSUS:DETSERial?';
    const CMD_switch_config_Serial_Number_List = 'OTU:SWITCH:OSUS:CONFSERial?';
    const CMD_switch_osx_detect_Serial_Number_List = 'OTU:SWITCH:OSXJdsu:DETSERial?';
    const CMD_switch_osx_config_Serial_Number_List = 'OTU:SWITCH:OSXJdsu:CONFSERial?';
    
    //SWITCH MOLEX
    const CMD_get_switch_molex_idn = 'OTU:SWITCH:MOLEX:IDN? %d';
    const CMD_get_switch_molex_available = 'OTU:SWITCH:MOLEX:AVAIlable?'; //TRUE/FALSE
    const CMD_get_switch_uos_available = 'OTU:SWITCH:UOSP:AVAIlable?'; //TRUE/FALSE
    const CMD_get_switch_config_type = 'OTU:SWITCH:CONFIG:TYPE? %d'; //OSX_JDSU, MOLEX
    const CMD_Has_Internal_Switch_License = 'OTU:SWITCH:LIC:INTValid?';
    const CMD_Get_Internal_Switch_Total_Ports = 'OTU:SWITCH:CONFIG:INTOutputs?';
        
    //OTAU NEW FUNCTIONS to handle big configurations of ROTAUs
    const CMD_compare_switches = 'OTU:SWITCH:COMPAR:MACRO? %d';//0/1... for Macroswitchid, returns SAME,DIFF,MISSCONF,MISSDET
    const CMD_compare_switches_result = 'OTU:SWITCH:COMPAR:RESULT? %d';//0/1... for Macroswitchid, returns SAME,DIFF,MISSCONF,MISSDET
    const CMD_switch_config_macro = 'OTU:SWITCH:CONFIG:MACRO? %d';//0/1... for Macroswitchid, returns ON,MISSCONF
    const CMD_switch_detect_macro = 'OTU:SWITCH:DETECT:MACRO? %d';//0/1... for Macroswitchid, returns ON,MISSDET
    const CMD_switch_detect_result = 'OTU:SWITCH:DETECT:RESULT? %d';//0/1... for Macroswitchid, returns ON,MISSDET
    const CMD_switch_detect_osu_homogeneous = 'OTU:SWITCH:DETECT:OSUIdentiques?';// returns SAME,DIFF
    
    //******************
    // ROTAU
    //**********************
    const CMD_is_otu_ROTAU = 'OTU:ROTAU?';      
    const CMD_verify_ROTAU = 'OTU:SWITCH:DETECT:IPROTAU:NVER? "%s"';//IP address: verify that an OTU is a ROTAU   
    const CMD_get_rotau_sn_conf = 'OTU:SWITCH:CONFIG:SERIAL? %d,GLOB'; //1,2,...
    const CMD_get_rotau_sn_det = 'OTU:SWITCH:DET:SERIAL? %d,GLOB'; //1,2,...
    const CMD_switch_detect_calib_desc = 'OTU:SWITCH:CONFIG:DETECT:DESC? %d,%s';//0...,GLOB
    const CMD_switch_detect_add_chaine_ROTAU = 'OTU:SWITCH:DETECT:IPROTAU:NADD "%s"';//String IP ROTAU
    const CMD_switch_config_add_chaine_ROTAU = 'OTU:SWITCH:CONFIG:NROTAUADD? %s,"%s","%s"';// IP_ROTAU, String switch code, String IP
    const CMD_switch_detect_remove_chaine_ROTAU = 'OTU:SWITCH:DETECT:IPROTAU:NDEL "%s"';//String IP ROTAU
    
    //****************
    // DMUX
    //****************
    const CMD_get_dmux_list = 'OTU:WDM:LISTe?';
    const CMD_get_dmux = 'OTU:WDM:GET? %d';// 1 DMUX position
    const CMD_remove_dmux = 'OTU:WDM:DEL %d';// 1 DMUX position
    const CMD_add_dmux = 'OTU:WDM:ADD? "%s",%d,%s';// 1 DMUX position, Name, Outputs, Channel
    const CMD_update_dmux = 'OTU:WDM:CHANGE %d,"%s",%d,%s';// 1 DMUX position, Name, Outputs, Channel
    
    //****************
    //* measurement
    //***************
    const CMD_pulse_SI = 'OTU:MODULE:CALOTds:SI:LPULSE? %s,"%s"';//MOD1 or MOD2, Function
    /**
     * @param string $module MOD1 or MOD2
     * @param string $function  
     * @return string pulses in ns
     */
    static function getModulePulses( $module, $function )
    {
    	return sprintf( self::CMD_pulse_SI, $module, $function );
    }
    const CMD_range_SI = 'OTU:MODULE:CALOTds:SI:FRANGE? %s,"%s",%d';//MOD1 or MOD2, Function, pulse
    /**
     * @param string $module MOD1 or MOD2
     * @param string $function
     * @param number $pulse in ns
     * @return string ranges in km
     */
    static function getModuleRanges( $module, $function, $pulse )
    {
    	return sprintf( self::CMD_range_SI, $module, $function, $pulse );
    }    
    const CMD_resolution_SI = 'OTU:MODULE:CALOTds:SI:FRESOLUTION? %s,"%s",%d,%F';//MOD1 or MOD2, Function, pulse, range    
    /**
     * @param string $module MOD1 or MOD2
     * @param string $function
     * @param number $pulse
     * @param float $range
     * 
     * @return string (int) resolutions in cm
     */
    static function getModuleResolutions( $module, $function, $pulse, $range )
    {
    	return sprintf( self::CMD_resolution_SI, $module, $function, $pulse, $range );
    }    
    const CMD_ind_ref = 'OTU:INDEX:DEFAULT? '; 
    
    /**
     * //module, switches number, switch ASCII buffer// auto Config (AUTO, MANUAL) // pulse //range KM // duration s // refraction index // wavelength // resolution // function// short acq
     */
    const CMD_acquisition_otdr_SI = 'OTU:MEALINK:SI:SWN:CONFIG? %s,%d,%s,%s,%d,%F,%d,%F,%d,%d,"%s",%s';
    /**
     * Execute a measurement on demand command
     *
     * @param string $module
     * @param integer $switchCount 
     * @param string $switchBuffer
     * @param string $autoConfigMode //AUTO, MANUAL
     * @param integer $pulseNs
     * @param float $rangeKM
     * @param integer $durationSec
     * @param float $refractionIndex
     * @param integer $wavelengthNm
     * @param integer $resolutionCm
     * @param string $function
     * @param boolean $shortAcq
     * 
     * @return string
     */
    static function getExecuteMeasurementCommand( $module, $switchCount, $switchBuffer, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq )
    {
    	return sprintf( self::CMD_acquisition_otdr_SI, $module, $switchCount, $switchBuffer, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq? self::RES_YES : self::RES_NO);
    }
    
    /**
     * //module, switches number, switch ASCII buffer// auto Config (AUTO, MANUAL) // pulse //range KM // duration s // refraction index // wavelength // resolution // function// short acq// frequency
     */
    const CMD_acquisition_DWDM_SI= 'OTU:MEALINK:SI:SWN:DWDM? %s,%d,%s,%s,%d,%F,%d,%F,%d,%d,"%s",%s,%d';
    /**
     * Execute a dwdm measurement on demand command
     *
     * @param string $module
     * @param integer $switchCount
     * @param string $switchBuffer
     * @param string $autoConfigMode //AUTO, MANUAL
     * @param integer $pulseNs
     * @param float $rangeKM
     * @param integer $durationSec
     * @param float $refractionIndex
     * @param integer $wavelengthNm
     * @param integer $resolutionCm
     * @param string $function
     * @param boolean $shortAcq
     * @param int $frequency
     *
     * @return string
     */
    static function getExecuteDwdmMeasurementCommand( $module, $switchCount, $switchBuffer, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq, $frequency )
    {
    	return sprintf( self::CMD_acquisition_DWDM_SI, $module, $switchCount, $switchBuffer, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq? self::RES_YES : self::RES_NO, $frequency);
    }
    
    /**
     * link Id // auto Config (AUTO, MANUAL) // pulse //range KM // duration s // refraction index // wavelength // resolution // function // short acq
     */   
    const CMD_acquisition_on_link_SI = 'OTU:MEASURE:SI:SMARTCONFIG? %d,%s,%d,%F,%d,%F,%d,%d,"%s",%s';//numero de liaison, programmation autoconfig, valeur du pulse, Valeur de la portee, duree acq, Indice de refraction, Valeur du laser, Valeur de la resolution, Nom de la fonction, short acq                       
    /**
     * Execute a measurement on a link command
     * 
     * @param integer $linkId
     * @param string $autoConfigMode //AUTO, MANUAL
     * @param integer $pulseNs
     * @param float $rangeKM
     * @param integer $durationSec
     * @param float $refractionIndex
     * @param integer $wavelengthNm
     * @param integer $resolutionCm
     * @param string $function
     * @param boolean $shortAcq
     * 
     * @return string
     */
    static function getExecuteMeasurementOnLinkCommand( $linkId, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq)
    {
    	return sprintf( self::CMD_acquisition_on_link_SI, $linkId, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq? self::RES_YES : self::RES_NO);
    }
    
    /**
     * link Id // auto Config (AUTO, MANUAL) // pulse //range KM // duration s // refraction index // wavelength // resolution // function // short acq
     */
    const CMD_acquisition_on_link_DWDM = 'OTU:MEASURE:SI:DWDM? %d,%s,%d,%F,%d,%F,%d,%d,"%s",%s,%d';//numero de liaison, programmation autoconfig, valeur du pulse, Valeur de la portee, duree acq, Indice de refraction, Valeur du laser, Valeur de la resolution, Nom de la fonction, short acq
    /**
     * Execute a measurement on a link command
     *
     * @param integer $linkId
     * @param string $autoConfigMode //AUTO, MANUAL
     * @param integer $pulseNs
     * @param float $rangeKM
     * @param integer $durationSec
     * @param float $refractionIndex
     * @param integer $wavelengthNm
     * @param integer $resolutionCm
     * @param string $function
     * @param boolean $shortAcq
     * @param int     $frequency
     *
     * @return string
     */
    static function getExecuteMeasurementOnLinkDwdmCommand( $linkId, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq, $frequency)
    {
    	return sprintf( self::CMD_acquisition_on_link_DWDM, $linkId, $autoConfigMode, $pulseNs, $rangeKM, $durationSec, $refractionIndex, $wavelengthNm, $resolutionCm, $function, $shortAcq? self::RES_YES : self::RES_NO, $frequency);
    }
        
    const CMD_meas_add_status = 'OTU:MEAS:ADD:STATUS?';
    /**
     * Get the status of the addition of the measure added with CMD_acquisition_on_link_SI command
     *
     * @return string
     */
    static function getMeasurementAddStatus( )
    {
    	return self::CMD_meas_add_status;
    }    

    const CMD_meas_status = 'OTU:MEAS:STATUS?';
    /**
     * Get the status of the running measure launched with CMD_acquisition_on_link_SI command
     *
     * @return string
     */
    static function getMeasurementStatus( )
    {
    	return self::CMD_meas_status;
    }
    
    const CMD_stop_measure = 'OTU:MEAS:STOP';
    /**
     * Stop the current running measure command
     *
     * @return string
     */
    static function getStopMeasurement( )
    {
    	return self::CMD_stop_measure;
    }
    
    const CMD_meas_som_eom = 'OTU:MEAS:CURS?';
    /**
     * Get the SOM and EOM of the measurement on demand
     *
     * @return string
     */
    static function getMeasurementCursors( )
    {
    	return self::CMD_meas_som_eom;
    }
    
    //****************
    //* Links
    //***************
    
    /**
     * linkId
     */
    const CMD_link_swn_get = 'OTU:LINK:SWN:GET? %d';
    /**
     * Get link utf8 config command
     *
     * @param number $linkId
     *
     * @return string
     */
    static function getlinkUtf8Command( $linkId )
    {
    	return sprintf( self::CMD_link_swn_get, $linkId );
    }
    
    const CMD_link_list = 'OTU:LINK:LIST?';
    
    /**
     * links list command
     *
     * @return string
     */
    static function getLinkListCommand()
    {
    	return self::CMD_link_list;
    }
    
    const CMD_link_list_per_port = 'OTU:LINK:PLIST? %d, %d';
    
    /**
     * links list per port command
     *
     * @param number $portFrom
     * @param number $portTo
     *
     * @return string
     */
    static function getLinkListPerPortCommand($portFrom, $portTo)
    {
    	return sprintf(self::CMD_link_list_per_port, $portFrom, $portTo);
    }

    const CMD_links_ports = 'OTU:LINK:PORTs? %d, %d';
    
    /**
     * links port per port range command
     *
     * @param number $portFrom
     * @param number $portTo
     *
     * @return string
     */
    static function getLinkPortListPerPortCommand($portFrom, $portTo)
    {
    	return sprintf(self::CMD_links_ports, $portFrom, $portTo);
    }
    
    const CMD_Virtual_Switch_Port_Count = 'OTU:VSW:PORT:COUNT?';
            
    /**
     * Virtual switch port count
     *
     * @return int
     */
    static function getVirtualSwitchPortCount()
    {
     return self::CMD_Virtual_Switch_Port_Count;   
    }
    
    const CMD_link_change_gps_file = 'OTU:LINK:GPS:FILE %d,"%s"';
    /**
     * Update link GPS file name
     *
     * @param number $linkId
     * @param string $gpsFileName
     * @return string
     */
    static function updateGpsFileNameCommand( $linkId,$gpsFileName )
    {
    	return sprintf( self::CMD_link_change_gps_file, $linkId, $gpsFileName );
    }
    
    const CMD_get_map_url = 'OTU:SMARTOTU:MAPURL?';
    /**
     * Update link GPS file name
     *
     * @param number $linkId
     * @param string $gpsFileName
     * @return string
     */
    static function getMapUrlCommand()
    {
    	return self::CMD_get_map_url;
    }
    
    const CMD_link_delete_gps_file = 'OTU:LINK:GPS:DELete %d';
    /**
     * Dissociate geographical file
     *
     * @param number $linkId
     * 
     * @return string
     */
    static function deleteGpsFileCommand( $linkId )
    {
    	return sprintf( self::CMD_link_delete_gps_file, $linkId );
    }    
    
    /**
     * linkId
     */    
    const CMD_link_last_mon = 'OTU:LINK:LASTMONITORING? %d';
    /**
     * Get link last monitoring date command
     *
     * @param number $linkId
     * @return string
     */
    static function getLinkLastMonitoringDateCommand( $linkId )
    {
    	return sprintf( self::CMD_link_last_mon, $linkId );
    }    
    
    /**
     * linkId
     */
    const CMD_link_available = 'OTU:LINK:AVA? %d';
    /**
     * Get link available/ not available command
     * @param number $linkId
     * @param boolean $available
     * @return string
     */
    static function getLinkAvailableCommand( $linkId )
    {
    	return sprintf( self::CMD_link_available, $linkId );
    }    
    

    /**
     * linkId//availability [KO, OK]
     */
    const CMD_set_link_available = 'OTU:LINK:AVA %d,%s';

    /**
     * Set link available/ not available command
     * @param number $linkId
     * @param boolean $available
     * @return string
     */
    static function linkAvailableCommand( $linkId, $available )
    {
    	return sprintf( self::CMD_set_link_available, $linkId, $available);
    }
        
    /**
     * linkId//link name
     */
    const CMD_link_rename = 'OTU:LINK:CHAN:NAME %d,"%s"';
    
    /**
     * rename link command
     * @param number $linkId
     * @param string $newlinkName
     * @return string
     */
    static function linkRenameCommand( $linkId, $newlinkName )
    {
    	return sprintf( self::CMD_link_rename, $linkId, $newlinkName);
    }
        
    /**
     * linkId//link name
     */
    const CMD_link_utf8_rename = 'OTU:LINK:UTF8:CHAN:NAME %d,"%s"';
    
    /**
     * rename link command
     * @param number $linkId
     * @param string $newlinkName
     * @return string
     */
    static function linkUtf8RenameCommand( $linkId, $newlinkName )
    {
    	return sprintf( self::CMD_link_utf8_rename, $linkId, $newlinkName);
    }
    

    /**
     * utf8 link name //link name //availability [KO, OK] //timeslot [NONE,CI10MN,CP10MN,CI5MN,CP5MN,C135,C246] // [MOD1,MOD2]// traffic [NO,YES]// switch# //switch buffer
     */
    const CMD_link_swn_add = 'OTU:LINK:SWN:ADD? "%s","%s",%s,%s,%s,%s,%d,%s';
    
    /**
     * Build add link command
     *
     * @param string $utf8LinkName
     * @param string $linkName
     * @param number $moduleRunningNumber //1 or 2
     * @param number $switchCount
     * @param string $switchBuffer
     *
     * @return string
     */
    static function addLinkUtf8Command( $utf8LinkName, $linkName, $moduleRunningNumber, $switchCount, $switchBuffer )
    {
    	return sprintf( self::CMD_link_swn_add, $utf8LinkName, $linkName, self::RES_OK, self::RES_NONE, self::RES_MOD.$moduleRunningNumber, self::RES_YES, $switchCount, $switchBuffer);
    }    
    /**
     * @param number linkId
     */
    const CMD_delete_link ='OTU:LINK:DEL %d';
    /**
     * Delete a link command
     * @param number $linkId
     * @return string command
     */
    static function deleteLinkCommand($linkId)
    {
    	return sprintf( self::CMD_delete_link, $linkId );
    }    
            
    //****************
    //* tests
    //***************
    /**
     * linkId
     */
    const CMD_list_test_mon = 'OTU:LINK:TEST:MON? %d';
    /**
     * Get the list of tests identifiers on the link command
     *
     * @param number $linkId
     *
     * @return string
     */
    static function getTestOnLinkCommand( $linkId )
    {
    	return sprintf( self::CMD_list_test_mon, $linkId );
    }    
    
    const CMD_mon_list = 'OTU:MON:LIST?';
    
    /**
     * Monitoring tests list command
     *
     * @return string
     */
    static function getMonitoringTestsListCommand()
    {
    	return self::CMD_mon_list;
    }    
    
    /**
     * Monitoring test parameters read from FO command
     * 
     * @param number $testId
     */    
    const CMD_info_test_mon = 'OTU:MON:INF:DET? %d';
    
    /**
     * Symetric to OTU:TOPAZ:MON:PEAK:Add command
     * 
     * @param number $testId     
     */
    const CMD_test_mon_desc = 'OTU:HIGHSENS:MON:GET? %d';
    /**
     * Get test description command
     *
     * @param number $testId
     *
     * @return string
     */
    static function getTestDescriptionCommand( $testId )
    {
    	return sprintf( self::CMD_test_mon_desc, $testId );
    }    
        
    /**
     * testId
     */    
    const CMD_test_mon_available = 'OTU:MON:AVA? %d';
    /**
     * Get test available/ not available command
     *
     * @param number $testId
     * 
     * @return string
     */
    static function getTestAvailableCommand( $testId )
    {
    	return sprintf( self::CMD_test_mon_available, $testId );
    }
    
    /**
     * testId//availability [KO, OK]
     */
    const CMD_set_test_mon_available = 'OTU:MON:AVA %d,%s';    

    /**
     * Set test available/ not available command
     * 
     * @param number $testId
     * @param boolean $available
     * @return string
     */
    static function testAvailableCommand( $testId, $available )
    {
    	return sprintf( self::CMD_set_test_mon_available, $testId, $available);
    }
    
    /**
     * testId
     */
    const CMD_test_mon_info = 'OTU:TOPAZ:MON:INFO? %d';
    
    /**
     * Get test mode and available/ not available command
     *
     * @param number $testId
     *
     * @return string
     */
    static function getTestInfoCommand( $testId )
    {
    	return sprintf( self::CMD_test_mon_info, $testId );
    }
    
    /**
     *  [STOP, SEQU, RAND]
     */
    const CMD_get_mon_seq_status = 'OTU:MON:SEQ?';
    
    /**
     * Get monitoring sequensor status command
     *
     * @return string [STOP, SEQU, RAND]
     */
    static function getMonitoringSequensorStatus()
    {
    	return self::CMD_get_mon_seq_status;
    }    
    
    /**
     *  [STOP, SEQU, RAND]
     */
    const CMD_set_mon_seq_status = 'OTU:MON:SEQ %s';
    
    /**
     * Get monitoring sequensor status command
     *
     * @param string [STOP, SEQU, RAND]
     */
    static function setMonitoringSequensorStatus( $status )
    {
    	return sprintf(self::CMD_set_mon_seq_status, $status );
    }
    
    
    
    /**
     *  [WAIT, ENABLE, DISABLE]
     */
    const CMD_get_mon_seq_suspend = 'OTU:SUSPEND?';
    
    /**
     * Get monitoring sequensor suspend status command
     *
     * @return string [WAIT, ENABLE, DISABLE]
     */
    static function getMonitoringSequensorSuspend()
    {
    	return self::CMD_get_mon_seq_suspend;
    }
    
    /**
     *  [WAIT, ENABLE, DISABLE]
     */
    const CMD_set_mon_seq_suspend = 'OTU:SUSPEND %s';
    
    /**
     * Set monitoring sequensor suspend status 
     *
     * @param boolean
     */
    static function setMonitoringSequensorSuspend( $suspend )
    {
    	return sprintf(self::CMD_set_mon_seq_suspend, $suspend? self::RES_ENABLE : self::RES_DISABLE);
    }
    
    /**
     * testId
     */    
    const CMD_execute_test_mon = 'OTU:MON:EXEC? %d,%s';
    /**
     * Execute test on demand command
     *
     * @param number $testId
     * @param string $mode //ALL or WEB to send an alarm or not
     * @return string
     */    
    static function getExecuteTestCommand( $testId, $mode )
    {
    	return sprintf( self::CMD_execute_test_mon, $testId, $mode );
    }
    
    const CMD_test_mon_status = 'OTU:MON:STATUS?';
    /**
     * Get the status of the monitoring test launched with CMD_execute_test_mon command
     *
     * @return string
     */
    static function getTestStatus( )
    {
    	return sprintf( self::CMD_test_mon_status );
    }
        
    /**
     * testId
     */    
    const CMD_test_mon_level_alarm = 'OTU:MON:LEVEL? %d';
    /**
     * Get test alarm level status command
     *
     * @param number $testId
     * @return string
     */
    static function getTestAlarmLevelCommand( $testId )
    {
    	return sprintf( self::CMD_test_mon_level_alarm, $testId );
    }    
    
    /**
     * test start date(s)// period mn// linkId// test available [KO, OK]// loc always after (NO, ONELoc, ALLLoc])// trace name // first marker m// last marker m// peak shift en point// type monitoring de pic// number of thresholds //threshold buffer// peaks positions buffer // peak names
     * 
     * @return testId
     */
    const CMD_test_add = 'OTU:TOPAZ:MON:PEAK:ADD? %d,%d,%d,%s,%s,"%s",%F,%F,%d,%s,%d,%s,%s,"%s"';        
    /**
     * Build add test command
     * 
     * @param integer $linkId
     * @param string $traceOtdrName
     * @param float $firstMakerM
     * @param float $lastMarkerM
     * @param int $peakShift
     * @param string $monitoringType
     * @param integer $thresholdsCount
     * @param string $thresholdsBuffer
     * @param string $peaksPositionsBuffer
     * @parma string $peakNames
     * 
     * @return string
     */
    static function addTestCommand( $linkId, $traceOtdrName, $firstMakerM, $lastMarkerM, $peakShift, $monitoringType, $thresholdsCount, $thresholdsBuffer, $peaksPositionsBuffer, $peakNames)
    {
    	return sprintf( self::CMD_test_add, 0, 0, $linkId, self::RES_OK, self::RES_NO, $traceOtdrName, $firstMakerM, $lastMarkerM, $peakShift, $monitoringType, $thresholdsCount, $thresholdsBuffer, $peaksPositionsBuffer,$peakNames);
    }
    
    /**
     * test start date(s)// period mn// linkId// test available [KO, OK]// loc always after (NO, ONELoc, ALLLoc])// trace name // test mode // first marker m// last marker m// peak shift en point// type monitoring de pic// number of thresholds //threshold buffer// peaks positions buffer // peak names
     *
     * @return testId
     */
    const CMD_highsens_test_add = 'OTU:HIGHSENS:MON:ADD? %d,%d,%d,%s,%s,"%s",%s,%.2F,%.2F,%.2F,%s,%s,%s,%d,%s,%s,%s,%d,"%s"';
    /**
     * Build add test command
     *
     * @param integer $linkId
     * @param string $traceOtdrName
     * @param float $SOM
     * @param float $EOM
     * @param int $peakShift
     * @param string $peakMonitoringType
     * @param integer $peakCount
     * @param string $thresholdsBuffer
     * @param string $peaksThresholdsBuffer
     * @param string $peaksPositionsBuffer
     * @parma string $peakNames
     *
     * @return string
     */
    static function addHighsensTestCommand( $linkId, $testMode, $traceOtdrName, $SOM, $EOM, $peakShift, $peakMonitoringType, $peakCount, $thresholdsBuffer, $peaksThresholdsBuffer, $peaksPositionsBuffer, $peakNames)
    {
    	return sprintf( self::CMD_highsens_test_add, 0, 0, $linkId, self::RES_OK, self::RES_NO, $traceOtdrName, $testMode, $SOM, $SOM, $EOM, $thresholdsBuffer, self::EMPTY_BUFFER,self::EMPTY_BUFFER,$peakCount,$peaksThresholdsBuffer,$peaksPositionsBuffer,$peakMonitoringType,$peakShift,$peakNames);
    }
    
    /**
     * test Id, test start date(s)// period mn// linkId// test available [KO, OK]// loc always after (NO, ONELoc, ALLLoc])// trace name // first marker m// last marker m// peak shift en point// type monitoring de pic// number of thresholds //threshold buffer// peaks positions buffer // peak names
     *
     * @return testId
     */
    const CMD_highsens_test_update = 'OTU:HIGHSENS:MON:CHAN %d,%d,%d,%d,%s,%s,"%s",%.2F,%.2F,%.2F,%s,%s,%s,%d,%s,%s,%s,%d,"%s"';
    /**
     * Build change test command
     *
     * @param integer $testId
     * @param integer $linkId
     * @param string $traceOtdrName
     * @param float $SOM
     * @param float $EOM
     * @param int $peakShift
     * @param string $peakMonitoringType
     * @param integer $peakCount
     * @param string $thresholdsBuffer
     * @param string $peaksThresholdsBuffer
     * @param string $peaksPositionsBuffer
     * @parma string $peakNames
     *
     * @return string
     */
    static function updateHighsensTestCommand( $testId, $linkId, $traceOtdrName, $SOM, $EOM, $peakShift, $peakMonitoringType, $peakCount, $thresholdsBuffer, $peaksThresholdsBuffer, $peaksPositionsBuffer, $peakNames, $monitoringAllowed)
    {
    	return sprintf( self::CMD_highsens_test_update, $testId,0, 0, $linkId, $monitoringAllowed? self::RES_OK : self::RES_KO, self::RES_NO, $traceOtdrName, $SOM, $SOM, $EOM, $thresholdsBuffer, self::EMPTY_BUFFER,self::EMPTY_BUFFER,$peakCount,$peaksThresholdsBuffer,$peaksPositionsBuffer,$peakMonitoringType,$peakShift,$peakNames);
    }
    
    /**
     * add test localisation:
     * testId// startOfMonitoring m// buffer start of monitoring threshold// peak shift en point// trace name1// start of trace zone1// trace name2// start of trace zone2// trace 3// start of trace zone3// number of peaks //peaks threshold buffer// peaks positions buffer
     */
    const CMD_localisation_test_add = 'OTU:TOPAZ:MON:PEAK:LOC:ADD %d,%F,%d,%s,"%s",%F,"%s",%F,"%s",%F,%d,%s,%s';
    /**
     * Build the add test command
     *
     * @param integer $testId
     * @param float $startOfMonitoring
     * @param int $peakShift
     * @param string $traceOtdrName 
     * @param integer $peaksCount
     * @param string $peaksThreasholdsBuffer
     * @param string $peaksPositionsBuffer
     *
     * @return string 
     */
    static function addLocalisationTestCommand( $testId, $startOfMonitoring, $peakShift, $traceOtdrName, $peaksCount, $peaksThreasholdsBuffer, $peaksPositionsBuffer )
    {
    	return sprintf( self::CMD_localisation_test_add, $testId, $startOfMonitoring, $peakShift, self::EMPTY_BUFFER, $traceOtdrName, 0, "", 0, "", 0, $peaksCount, $peaksThreasholdsBuffer, $peaksPositionsBuffer);
    }
    
    /**
     *
     */
    const CMD_get_last_test_add_status ='OTU:TEST:ADD:STATUS?';
    /**
     * Get last test addition status command.
     * 
     * @return string command
     */
    static function getLastTestAddStatusCommand()
    {
    	return sprintf( self::CMD_get_last_test_add_status );
    }    
    
    /**
     * testId // test start date(s)// period mn// linkId// test available [KO, OK]// loc always after (NO, ONELoc, ALLLoc])// trace name // first marker m// last marker m// peak shift en points// type de monitoring de pic// number of thresholds //threshold buffer// peaks positions buffer // trace LOC name // start of monitoring // peak names 
     */
    const CMD_test_update = 'OTU:TOPAZ:MON:PEAK:CHAN %d,%d,%d,%d,%s,%s,"%s",%F,%F,%d,%s,%d,%s,%s,"%s",%F,"%s"';
    
    /**
     * Build the update test command
     * 
     * @param integer $testId
     * @param integer $linkId
     * @param string $traceOtdrName
     * @param float $firstMakerM and start of monitoring for localization
     * @param float $lastMarkerM
     * @param int $peakShift
     * @param string $monitoringType
     * @param integer $thresholdsCount
     * @param string $thresholdsBuffer
     * @param string $peaksPositionsBuffer
     * @param string $peakNames
     * 
     * @return string
     */
    static function updateTestCommand( $testId, $linkId, $traceOtdrName, $firstMakerM, $lastMarkerM, $peakShift, $monitoringType, $thresholdsCount, $thresholdsBuffer, $peaksPositionsBuffer, $peakNames, $monitoringAllowed)
    {
    	return sprintf( self::CMD_test_update, $testId, 0, 0, $linkId, $monitoringAllowed? self::RES_OK : self::RES_KO, self::RES_NO, $traceOtdrName, $firstMakerM, $lastMarkerM, $peakShift, $monitoringType, $thresholdsCount, $thresholdsBuffer, $peaksPositionsBuffer, $traceOtdrName, $firstMakerM, $peakNames);
    }    
        
    /**
     * testId
     */
    const CMD_delete_test ='OTU:MON:DEL %d';    
    /**
     * Delete a monitoring test command
     * @param number $testId
     * @return string command
     */
    static function deleteTestCommand($testId)
    {
        return sprintf( self::CMD_delete_test, $testId );    
    }
            
    const CMD_set_missing_peak_threshold_points_det = 'OTU:TOPAZ:MONITORING:DELTAPOSPEAK %d,DET,%d';
    
    /**
     * Set max number of points of peak shift
     * 
     * @param number $testId
     * @param number $nbPoints
     * 
     * @return string command
     */
    static function setMissingPeakThresholdPointsDet($testId, $nbPoints)
    {
    	return sprintf( self::CMD_set_missing_peak_threshold_points_det, $testId, $nbPoints );
    }
    
    const CMD_get_missing_peak_threshold_points_det = 'OTU:TOPAZ:MONITORING:DELTAPOSPEAK? %d,DET';
    
    /**
     * Get max number of points of peak shift
     *
     * @param number $testId
     *
     * @return string command
     */
    static function getMissingPeakThresholdPointsDet($testId)
    {
    	return sprintf( self::CMD_get_missing_peak_threshold_points_det, $testId );
    }
    
    const CMD_set_missing_peak_threshold_points_loc = 'OTU:TOPAZ:MONITORING:DELTAPOSPEAK %d,LOC1,%d';
    /**
     * Set max number of points of peak shift
     *
     * @param number $testId
     * @param number $nbPoints
     *
     * @return string command
     */
    static function setMissingPeakThresholdPointsLoc($testId, $nbPoints)
    {
    	return sprintf( self::CMD_set_missing_peak_threshold_points_loc, $testId, $nbPoints );
    }
   
    const CMD_set_peak_name= 'OTU:TOPAZ:MONITORING:PEAK:NAME %d,%f,"%s"';
    /**
     * Set peak name
     *
     * @param number $testId
     * @param float  $position
     * @param string $name
     *
     * @return string command
     */
    static function setPeakName($testId, $position, $name)
    {
    	return sprintf( self::CMD_set_peak_name, $testId, $position, $name);
    }
    
    const CMD_get_peak_name= 'OTU:TOPAZ:MONITORING:PEAK:NAME? %d';
    /**
     * Get peak name
     *
     * @param number $testId
     *
     * @return string command
     */
    static function getPeaksName($testId)
    {
    	return sprintf( self::CMD_get_peak_name, $testId);
    }
    
    /**
     * @var float in dB
     */
    const CMD_set_critical_thresholds = "OTU:CAL:CRIT %d";
    
    /**
     * Set attenuation critical thresholds for all the monitoring test command
     * @param float in dB
     * @return string command
     */
    static function setAttenuationCriticalThresholds( $criticalThresholdDb )
    {
    	return sprintf( self::CMD_set_critical_thresholds, $criticalThresholdDb );
    }
    
    /**
     * @var string acknowledge string: alarm file name
     */
    const CMD_acknowledge_alarm='OTU:ALARM:ACKN? "%s"';
    /**
     * Acknowledge successful OTU alarm processing command
     * @param string $acknowledge alarm file name
     * @return string command
     */
    static function acknowledgeAlarmCommand( $acknowledge )
    {
    	return sprintf( self::CMD_acknowledge_alarm, $acknowledge );
    }          
    
    /**
     * type
     * testId
     * peak cNotPeak=-1
     */
    const CMD_delete_monitoring_test_alarms ='OTU:ALARM:OPT:CLEAR %s,%d,%d';
    
    /**
     * Delete monitoring test alarms command
     *
     * @param string $type INJ, ATT, 
     * @param number $testId
     * @param number $peak cNotPeak=-1
     * 
     * @return string command
     */
    static function deleteMonitoringTestAlarmsCommand($type, $testId, $peak = -1)
    {
    	return sprintf( self::CMD_delete_monitoring_test_alarms, $type, $testId, $peak );
    }
    
    /**
     * testId
     */
    const CMD_clear_system_alarm ='OTU:ALARM:SYST:CLEAR %d,"%s"';
    
    /**
     * Delete monitoring test alarms command
     *
     * @param number $specificProblemId
     * @param string $otuAdditionalInfoId
     *
     * @return string command
     */
    static function clearSystemAlarmCommand( $specificProblemId, $otuAdditionalInfoId )
    {
        //chunk string to 30 chars
        $otuAdditionalInfoId = ( strlen($otuAdditionalInfoId) > self::OTU_MAX_ADDITIONAL_INFO_LENGTH ) ? substr($otuAdditionalInfoId, 0, self::OTU_MAX_ADDITIONAL_INFO_LENGTH) : $otuAdditionalInfoId;
    	return sprintf( self::CMD_clear_system_alarm, $specificProblemId, $otuAdditionalInfoId );
    }
    
    /**
     * Clear all alarms
     */
    const CMD_clear_all_alarms ='OTU:ALARM:INIT';
    
    /**
     * Clear all alarms on OTU
     * @return string command
     */
    static function clearAllAlarmsCommand()
    {
    	return self::CMD_clear_all_alarms;
    }

    /**
     * Backup OTU monitoring files and configuration (alerts and media)
     */
    const CMD_backup_otu ='OTU:SMARTOTU:BACKUP?';
    
    /**
     * Backup OTU monitoring files and configuration (alerts and media) Command
     * @return string command
     */
    static function getSmartOtuBackupCommand()
    {
    	return self::CMD_backup_otu;
    }    
    
    /**
     * Backup OTU monitoring files and configuration (alerts and media) and SmartOTU password file
     */
    const CMD_restore_otu ='OTU:SMARTOTU:RESTORE "%s"';
    
    /**
     * Restore OTU backup monitoring files, configuration (alerts and media) and SmartOTU password file Command
     * 
     * @return string command
     */
    static function smartOtuRestoreCommand( $archiveFullPath )
    {
    	return sprintf( self::CMD_restore_otu, $archiveFullPath );
    }
    
    
    /**
     * Check OTU backup integrity
     */
    const CMD_check_otu_backup ='OTU:SMARTOTU:CHECKBackup "%s"';
    
    /**
     * Check OTU backup integrity Command
     *
     * @return string command
     */
    static function getSmartOtuCheckBackupCommand( $archiveFullPath )
    {
    	return sprintf( self::CMD_check_otu_backup, $archiveFullPath );
    }
    
    //****************
    //* admin
    //***************     
    /**
     * BOTH // BASE or SLIC2// OTU
     */  
    const CMD_otu_port = 'MODULE:FUNCTION:PORT? %s,%s,"%s"';    
    const CMD_change_unit = 'OTU:SYST:DUNI %s';
    const CMD_get_unit = 'OTU:SYST:DUNI? ';
    const CMD_get_otu_name = 'otu:name?';
    const CMD_set_otu_name = 'otu:name "%s"';
    const CMD_get_otu_lang = 'OTU:LANG?';
    const CMD_set_otu_lang = 'OTU:LANG %s';
    
    //****************
    //* alarms
    //***************
    const CMD_get_alarm = 'OTU:ALARM:GET? "%s"';
    /**
     * Alarm get command
     *
     * @param string $alarmFileName 
     *
     * @return string
     */
    static function getAlarmCommand( $alarmFileName )
    {
    	return sprintf( self::CMD_get_alarm, $alarmFileName );
    }
        
    const CMD_alarm_list = 'OTU:ALARM:LIST? %s,%s';      
    /**
     * Alarm list command
     *
     * @param string $severityMin //INFO,CLEAR,WARNING,MINOR,MAJOR,CRITICAL
     * @param string $resourceType //UNIT, COM, FIBER, PERF
     * @return string
     */
    static function getAlarmsListCommand( $severityMin, $resourceType )
    {
    	return sprintf( self::CMD_alarm_list, $severityMin, $resourceType );
    }    
    
    //****************************************
    //* 
    //* AUTOTEST 
    //* 
    //*/
    
    const CMD_ENABLE_AUTOTEST = 'OTU:ETST %s';
    
    const CMD_IS_AUTOTEST_ENABLED = 'OTU:ETST?';    
    
    /**
     * Whether autotest is enabled
     */
    static function isAutotestEnabledCommand()
    {
    	return self::CMD_IS_AUTOTEST_ENABLED;
    }
    
    /**
     * Set autotest enabled/disabled
     * 
     * @param $enable boolean
     */
    static function enableAutotestCommand( $enable )
    {
    	return sprintf( self::CMD_ENABLE_AUTOTEST, $enable? self::RES_ON : self::RES_OFF );
    }
    
    const CMD_SET_AUTOTEST_PERIOD_H = 'OTU:PTST %d'; //[6 : 720]
    
    const CMD_GET_AUTOTEST_PERIOD_H = 'OTU:PTST?';
    
    /**
     * Whether autotest is enabled
     */
    static function getAutotestPeriodInHourCommand()
    {
    	return self::CMD_GET_AUTOTEST_PERIOD_H;
    }
    
    /**
     * Set autotest enabled/disabled
     *
     * @param $periodH integer [6 : 720]
     */
    static function setAutotestPeriodInHourCommand( $periodH )
    {
    	return sprintf( self::CMD_SET_AUTOTEST_PERIOD_H, $periodH );
    }
    
    const CMD_SET_AUTOTEST_DATE_M = 'OTU:TPZ:AUTO:DATE %d';
    
    const CMD_GET_AUTOTEST_DATE_M = 'OTU:TPZ:AUTO:DATE?';
    
    /**
     * Get autotest date command in minutes
     */
    static function getAutotestDateInMinuteCommand()
    {
    	return self::CMD_GET_AUTOTEST_DATE_M;
    }
    
    /**
     * Set autotest date command in minutes
     *
     * @param $dateM integer [0 : 1439]
     */
    static function setAutotestDateInMinuteCommand( $dateM )
    {
    	return sprintf( self::CMD_SET_AUTOTEST_DATE_M, $dateM );
    }
    
    const CMD_START_MANUAL_AUTOTEST = "OTU:TPZ:AUTO:RUN";
    
    /**
     * Start manual autotest
     *
     */
    static function startManualAutotestCommand()
    {
    	return self::CMD_START_MANUAL_AUTOTEST;
    }
    
    const CMD_GET_MONITORING_TEST_PORT = "OTU:TOPAZ:MON:PORT? %d";
    /**
     * Get Monitoring test port
     *
     * @param $testId integer [0 : 1024]
     */
    static function getMonitoringTestPortCommand( $testId )
    {
    	return sprintf( self::CMD_GET_MONITORING_TEST_PORT, $testId);
    }
    
    const CMD_SET_FIREWALL = "OTU:SYSTEM:FIREWALL %s";
    const CMD_GET_FIREWALL = "OTU:SYSTEM:FIREWALL?";
    
    /**
     * Get firewall statuscommand
     */
    static function getFirewallCommand()
    {
    	return self::CMD_GET_FIREWALL;
    }
    
    /**
     * Set firewall status 
     *
     * @param $on
     * @param $custom
     */
    static function setFirewallCommand( $on, $custom )
    {
    	if ( $on ) 
    	{ 
    	    return sprintf( self::CMD_SET_FIREWALL, self::FIREWALL_ON );
    	}
    	else if ( $custom )
    	{
    	    return sprintf( self::CMD_SET_FIREWALL, self::FIREWALL_CUSTOM );
    	}
    	else
    	{
    	    return sprintf( self::CMD_SET_FIREWALL, self::FIREWALL_OFF );
    	}
    }
    
    const CMD_RESET_FIREWALL_CONTENT = "OTU:SYSTEM:EFIREWALL:CUSTReset";    
    /**
     * Get firewall reset custom firewall content
     */
    static function getFirewallResetCustomContentCommand()
    {
    	return self::CMD_RESET_FIREWALL_CONTENT;
    }
    
    const CMD_SET_FIREWALL_OLD = "OTU:SYSTEM:WWW %s";
    const CMD_GET_FIREWALL_OLD = "OTU:SYSTEM:WWW?";
    
    /**
     * Get firewall statuscommand
     */
    static function getFirewallCommandOld()
    {
    	return self::CMD_GET_FIREWALL_OLD;
    }
    
    /**
     * Set autotest date command in minutes
     *
     * @param $on
     */
    static function setFirewallCommandOld( $on )
    {
    	return sprintf( self::CMD_SET_FIREWALL_OLD, $on? self::FIREWALL_ON_OLD : self::FIREWALL_OFF_OLD );
    }
    
    const CMD_otu_get_option_list = "OTU:SYSTEM:OPTION:LIST?";
    /**
     * OTU COMMAND
     *
     * @return string //HIGHSEC,ACCESS,PEAK,DUAL_OS...
     */
    static function getOtu_Option_List()
    {
    	return self::CMD_otu_get_option_list;
    }
    
    const CMD_otu_is_option_status = "OTU:SYST:OPTION:STATUS? %s";
    
    /**
     * OTU COMMAND
     *
     * @param $option Option OTU key
     * 
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function getOtu_Option_Status($option)
    {
    	return sprintf(self::CMD_otu_is_option_status, $option);
    }
    
    const CMD_otu_get_PAYG_internal_switch_lic_count = "OTU:SWITCH:LIC:INTCount?";
    
    /**
     * OTU COMMAND
     *
     * @return integer number of PYG license count for internal switch
     */
    static function getOtu_PAYG_Internal_Switch_Lic_Count()
    {
        return self::CMD_otu_get_PAYG_internal_switch_lic_count;
    }
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_High_Security_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_HIGH_SEC);
    }
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Ssh_Key_Root_Access_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_SSH_KEY_ACCESS);
    }
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Peak_Monitoring_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_PEAK_MONITORING);
    } 
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Dual_Os_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_DUAL_OS_MONITORING);
    } 
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Ext_Temp_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_EXT_TEMP_RANGE_MONITORING);
    } 
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Test_Pon_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_TEST_PON_MONITORING);
    } 
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Saa_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_SAA_L2_MONITORING);
    } 
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Tapping_Monitoring_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_TAPPING_MONITORING);
    } 
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Light_Source_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_LIGHT_SOURCE);
    }
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Rest_Api_Option()
    {
    	return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_REST_API);
    }
    
    /**
     * OTU COMMAND
     *
     * @return string //IMPOSSIBLE POSSIBLE AVAILABLE EXPIRED
     */
    static function isOtu_Vpn_Option()
    {
        return sprintf(self::CMD_otu_is_option_status, SMTLicenseList::OTU_OPTION_VPN);
    }
    /**
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     * WARNING ISU COMMANDS
     */
    const CMD_get_OTU_LICENSE = "ISU:LIC:LIST? \"OTU\"";
    const CMD_get_Sensing_LICENSE = "ISU:LIC:LIST? \"Sensing\"";
    const CMD_get_Otdr_LICENSE = "ISU:LIC:LIST? \"Otdr\"";
    const CMD_get_SAA_LICENSE = "ISU:LIC:LIST? \"Anywhere\"";
    const CMD_get_OTAU_LICENSE = "ISU:LIC:LIST? \"Otau\"";
    
    /**
     * WARNING ISU COMMAND
     * 
     * @return string 
     */
    static function getOtuLicensesFromIsu()
    {
    	return self::CMD_get_OTU_LICENSE;
    }    
    /**
     * WARNING ISU COMMAND
     * 
     * @return string 
     */
    static function getSensingLicensesFromIsu()
    {
    	return self::CMD_get_Sensing_LICENSE;
    }
    /**
     * WARNING ISU COMMAND
     *
     * @return string
     */
    static function getOtauLicensesFromIsu()
    {
        return self::CMD_get_OTAU_LICENSE;
    }
    
    /**
     * WARNING ISU COMMAND
     *
     * @return string
     */
    static function getSAALicensesFromIsu()
    {
    	return self::CMD_get_SAA_LICENSE;
    }
    
    /**
     * WARNING ISU COMMAND
     *
     * @return string
     */
    static function getOtdrLicensesFromIsu()
    {
        return self::CMD_get_Otdr_LICENSE;
    }
    
    const CMD_isu_is_license_status = "PROCESS:LICENSE:STATUS? \"OTU\",\"%s\"";
    /**
     * WARNING ISU COMMAND
     *
     * @return string //VALID or INVALID
     */
     static function isHigh_Security_Option()
     {
     	return sprintf(self::CMD_isu_is_license_status, SMTLicenseList::ENHANCEDSECURITY);
    }
    /**
    * WARNING ISU COMMAND
    * 
    * @return string //VALID or INVALID
    */
    static function isSsh_Key_Root_Access_Option()
    {
    	return sprintf(self::CMD_isu_is_license_status, SMTLicenseList::SSH_KEY_ACCESS);
    }
    
    /**
     * WARNING ISU COMMAND
     *
     * @return string //VALID or INVALID
     */
    static function isPeak_Monitoring_Option()
    {
    	return sprintf(self::CMD_isu_is_license_status, SMTLicenseList::PEAK_MON);
    } 
    
    /**
     * WARNING ISU COMMAND
     *
     * @return string //VALID or INVALID
     */
    static function isTapping_Monitoring_Option()
    {
    	return sprintf(self::CMD_isu_is_license_status, SMTLicenseList::PEAK_MON);
    } 
    
    const CMD_isu_license_import = "ISU:LIC:IMPORT DISK, \"%s\"";
    
    /**
     * WARNING ISU COMMAND
     *
     * @param string //license file name
     */
    static function setSsh_License_Option($licenseFileName)
    {
    	return sprintf(self::CMD_isu_license_import, $licenseFileName);
    }
    
    const CMD_isu_get_payg_option_value= "isu:lic:payg? \"Otau\",\"%s\"";
    /**
     * WARNING ISU COMMAND
     *
     * @param $option Option PAYG key
     *
     * @return string
     */
    static function getOtu_Option_Value($option)
    {
        return sprintf(self::CMD_isu_get_payg_option_value, $option);
    }
        
    //Hard disk
    const CMD_get_harddisk_status = 'OTU:HDISK:STATUS?';
    static function get_harddisk_status()
    {
    	return self::CMD_get_harddisk_status;
    }
    const CMD_is_Backup_Hard_Disk_Option = 'OTU:HDBackup:OPTion?';
    static function get_is_Backup_Hard_Disk_Option()
    {
    	return self::CMD_is_Backup_Hard_Disk_Option;
    }
    const CMD_reset_harddisk = 'OTU:HDISK:RESET';
    static function get_reset_harddisk()
    {
    	return self::CMD_reset_harddisk;
    }
    
    // ssh strong/weak
    const CMD_Get_Ssh_Personalization='OTU:SYST:COM:SSH?';
    static function getSsh_Personalization()
    {
    	return self::CMD_Get_Ssh_Personalization;
    }        
    
    const CMD_get_otu_mon_mode= 'OTU:TOPAZ:MON:MODe?';
    static function get_otu_mon_mode()
    {
    	return self::CMD_get_otu_mon_mode;
    }
    
    const CMD_set_otu_mon_mode= "OTU:TOPAZ:MON:MODe %s";
    static function set_otu_mon_mode($mode)
    {
    	return sprintf( self::CMD_set_otu_mon_mode, $mode);
    }
    
    const CMD_create_ufom_archive="OTU:UFOM:ARCH:CSVSOR?";
    static function create_ufom_archive()
    {
        return self::CMD_create_ufom_archive;
    }
    
    /**
     *
     */
    const CMD_upgrade_UOS = 'OTU:SWITCH:DEBUG:UOSP:UPGRADE %d';
    
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    //
    // API
    //
    ////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
    const CMD_activate_vpn ='OTU:API:CONF:VPN:ENABLE TRUE';
    /**
     * Activate vpn command Command
     * @return string command
     */
    static function getActivateVpnCommand()
    {
        return self::CMD_activate_vpn;
    }
    
    const CMD_disable_vpn ='OTU:API:CONF:VPN:ENABLE FALSE';
    /**
     * Disable vpn Command
     * @return string command
     */
    static function getDisableVpnCommand()
    {
        return self::CMD_disable_vpn;
    }
    
    const CMD_vpn_file ='OTU:API:CONF:VPN:FILE?';
    /**
     * Vpn configuration file name Command
     * @return string command
     */
    static function getVpnFileCommand()
    {
        return self::CMD_vpn_file;
    } 
    
    const CMD_is_ntp_enabled= "OTU:API:CONF:NTP:ENABLE?";
    function getNtpActive()
    {
    	return self::CMD_is_ntp_enabled;
    }
    
    const CMD_get_ntp_sync= "OTU:API:CONF:NTP:SYNC";
    function getNtpSync()
    {
    	return self::CMD_get_ntp_sync;
    }
    
    const CMD_Api_get_otu_Main_INFO= 'OTU:API:CONF:SUMary?';
    /**
     * @return string "04993","OTU 8000E","OTDR","4","1561628574","PEAK_MON SMART_ACCESS_ANYWHERE","V18.05"
     */
    static function getOtuMainInfo()
    {
    	return self::CMD_Api_get_otu_Main_INFO;
    }
    
    const CMD_Api_update_otu_date = "OTU:API:CONF:DATE %d";
    static function updateOtuDate($timestamp)
    {
    	return sprintf(self::CMD_Api_update_otu_date, $timestamp);
    }
    
    const CMD_Api_setSnmpConfig = "OTU:API:CONF:SNMP:CONF %d,\"%s\",\"%s\",%s";
    static function setSnmpConfig($id, $address, $community, $port)
    {
    	return sprintf(self::CMD_Api_setSnmpConfig, $id, $address, $community, $port);
    }
    
    const CMD_Api_getSnmpConfig = "OTU:API:CONF:SNMP:CONF? %d";
    static function getSnmpConfig($id)
    {
    	return sprintf(self::CMD_Api_getSnmpConfig, $id);
    }
    
    const CMD_Api_setSnmpImAlive = "OTU:API:CONF:SNMP:IMAlive %d";
    static function setSnmpImAlive($periodMin)
    {
    	return sprintf(self::CMD_Api_setSnmpImAlive, $periodMin);
    }
    
    const CMD_Api_getSnmpImAlive = 'OTU:API:CONF:SNMP:IMAlive?';
    static function getSnmpImAlive()
    {
    	return self::CMD_Api_getSnmpImAlive;
    }    
    
    const CMD_Api_testSnmpConfig = "OTU:API:CONF:SNMP:TEST %d";
    static function testSnmpConfig($id)
    {
    	return sprintf(self::CMD_Api_testSnmpConfig, $id);
    }
    
    const CMD_Api_getPort = "OTU:API:PORT:CONF? %d";
    static function getPortConfig($port)
    {
    	return sprintf(self::CMD_Api_getPort, $port);
    }
    
    const CMD_Api_updatePort = "OTU:API:PORT:CONF %d,\"%s\",%s";
    static function updatePortConfig($port, $name, $monitored)
    {
    	return sprintf(self::CMD_Api_updatePort, $port, $name, $monitored? 'TRUE' : 'FALSE');
    }
    const CMD_Api_getModule_Info = 'OTU:API:CONF:MOD:OPT?';
    static function getModuleInfo()
    {
    	return self::CMD_Api_getModule_Info;
    }
    const CMD_Api_getIpv4 = 'OTU:API:CONF:ETH:IPV4?';
    static function getIpv4Config()
    {
    	return self::CMD_Api_getIpv4;
    }
    const CMD_Api_getIpv6 = 'OTU:API:CONF:ETH:IPV6?';
    static function getIpv6Config()
    {
    	return self::CMD_Api_getIpv6;
    }
    const CMD_Api_getPortAlarmStatus = "OTU:API:ALARM:OPT:GET? %d";
    static function getPortAlarmStatus($port)
    {
    	return sprintf(self::CMD_Api_getPortAlarmStatus, $port);
    }
    
    const CMD_Api_getReplayAlarm = "OTU:API:ALARM:OPT:REPLAY %d";
    static function replayAlarm($port)
    {
    	return sprintf(self::CMD_Api_getReplayAlarm, $port);
    }
    
    const CMD_Api_getClearAlarm = "OTU:API:ALARM:OPT:CLEAR %d";
    static function clearAlarm($port)
    {
    	return sprintf(self::CMD_Api_getClearAlarm, $port);
    }
    
    const CMD_Api_getClearAllAlarms = "OTU:API:ALARM:CLRALL";
    static function clearAllAlarms()
    {
    	return self::CMD_Api_getClearAllAlarms;
    }
    
    const CMD_Api_getToken = "OTU:API:TOKEN?";
    static function getApiToken()
    {
    	return self::CMD_Api_getToken;
    }
    
    const CMD_Api_checkToken = "OTU:API:TOKEN \"%s\"";
    static function getApiCheckToken($token)
    {
    	return sprintf(self::CMD_Api_checkToken, $token);
    }
    
    const CMD_Api_changeSmartOTUCredential= "OTU:API:USER:SMARTOTU \"%s\",\"%s\",\"%s\",\"%s\"";
    static function changeSmartOTUCredential($currentLogin, $currentPassword, $newLogin, $newPassword)
    {
    	return sprintf(self::CMD_Api_changeSmartOTUCredential, $currentLogin, $currentPassword, $newLogin, $newPassword);
    }
    
    const CMD_Api_changeSmartOTUGuestCredential= "OTU:API:USER:GUEST \"%s\",\"%s\"";
    static function changeSmartOTUGuestCredential($newLogin, $newPassword)
    {
    	return sprintf(self::CMD_Api_changeSmartOTUGuestCredential, $newLogin, $newPassword);
    }
    
    const CMD_Api_SmartOTUGuestUser= "OTU:API:USER:ENAGuest %s";
    static function enableSmartOTUGuestUser($enable)
    {
    	return sprintf(self::CMD_Api_changeSmartOTUCredential, $enable? 'TRUE' : 'FALSE');
    }
    
    const CMD_Api_HasSmartOTUGuestUser= 'OTU:API:USER:ENAGuest?';
    static function hasSmartOTUGuestUser()
    {
    	return self::CMD_Api_HasSmartOTUGuestUser;
    }
}


?>