<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\http;

use app\serviceshelper\otdr\SMTBaseOtdrTraceFileUtil;

use app\util\SMTLogger;

use app\util\SMTIOException;

class SMTSessionDirectoryUtil
{
    /**
     * Directory for session files (otdr traces)
     * @var unknown
     */
	const CUSTOM_SMARTOTU_SESSION_DIR = "/otu/smartOTU/session_";
	/**
	 * Default life time for session files before being deleted
	 * 
	 * @var int
	 */
	const DEFAULT_MAX_LIFE_TIME_FOR_SESSION_FILES = 1800; //30 min
	
	/**
	 * OTDR default file name on session directory
	 * 
	 * @param $portNumber
	 * @param $operationId 
	 * @var string
	 */
	const DEFAULT_OTDR_FILE_NAME = "measure_%s_%s.sor";
	
	/**
	 * OTDR default file name on session directory
	 *
	 * @param $portNumber
	 * @param $operationId
	 * @var string
	 */
	const BASE_OTDR_FILE_NAME = "measure_%s_%s%s";
	
	/**
	 * Maximum number of OTDR files stored in session: 10
	 * Those files can be used as reference of new monitoring test
	 *
	 * @var int
	 */
	const DEFAULT_MAX_OTDR_FILES_IN_SESSION = 10;
	/**
	 * Directory for temporary files
	 * @var string
	 */
	const SMART_OTU_TEMP_DIR = '/otu/smartOTU/tmp/';
	/**
	 * Maximum number of temporary files stored: 15
	 *
	 * @var int
	 */	
	const DEFAULT_MAX_FILES_IN_TEMP_DIR = 15;
	
/**
 * Build otdr file name stored on SmartOtu session directory
 * @param int $portNumber The port number where the acquisition was done.
 * @param string $operationId The operation id of the measure
 * @param string $smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION
 */
	static function buildOtdrFileName( $portNumber, $operationId, $smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION)
{
	return sprintf(self::BASE_OTDR_FILE_NAME, $portNumber, $operationId, $smartAcqTraceExtension);
}	
	
/**
 * Returns the last acquisition trace full path for the current session and from SmartOTU directory.
 *
 * @param $portNumber The port number where the acquisition was done.
 * 
 * @return string The acquisition file path in SmartOtu directory
 */
static function getLastOTDRTraceFileNameFromSession( $portNumber, $smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION)
{
	$acquisitionTrace = NULL;
	$sessionDirectory = self::createSessionDirectoryIfNeeded();
	foreach ( glob( $sessionDirectory.self::buildOtdrFileName( $portNumber, "*", $smartAcqTraceExtension) ) as $file )
	{
		$acquisitionTrace = $file;
		break;
	}
	 
	if ( $acquisitionTrace == NULL )
	{
		throw new SMTIOException( SMTIOException::FILE_NOT_FOUND, $sessionDirectory.sprintf(self::DEFAULT_OTDR_FILE_NAME, $portNumber) );
	}

	return $acquisitionTrace;
}

/**
 * Returns the acquisition trace full path for the current session and from SmartOTU directory.
 *
 * @param $otdrShortFileName The otdr short file name
 *
 * @return string The acquisition file path in SmartOtu directory
 * @throws SMTIOException if file was not found
 */
static function getOTDRTraceFileNameFromSession( $otdrShortFileName )
{
	$acquisitionTrace = NULL;
	$sessionDirectory = self::createSessionDirectoryIfNeeded();
	
	SMTLogger::getInstance()->trace("sessionDirectory + file name: ".$sessionDirectory.$otdrShortFileName);
	
	foreach ( glob( $sessionDirectory.$otdrShortFileName ) as $file )
	{
		$acquisitionTrace = $file;
		break;
	}

	if ( $acquisitionTrace == NULL )
	{
		throw new SMTIOException( SMTIOException::FILE_NOT_FOUND, $sessionDirectory.$otdrShortFileName );
	}

	return $acquisitionTrace;
}

/**
 * Backup the given OTDR file to SmartOTU directory by user session.
 *
 * @param string $filePath
 * @return string The new file path in SmartOtu directory
 */
public static function backupOTDRTraceToSessionDirectory( $filePath, $portNumber, $operationId, $smartAcq = FALSE )
{
    self::removeSessionsDirectoriesIfNeeded();
    
	$sessionDirectory = self::createSessionDirectoryIfNeeded();
    self::checkOTDRFilesCountInSession();
    
    $smartAcqTraceExtension = $smartAcq? SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($filePath) : SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION;
	
    $newFilePath = $sessionDirectory.self::buildOtdrFileName( $portNumber, $operationId, $smartAcqTraceExtension) ;

	if ( !copy( $filePath, $newFilePath ) )
	{
		throw new SMTIOException( SMTIOException::COPY_FAILURE, 'from '.$filePath.' to '.$newFilePath );
	}

	return $newFilePath;
}

/**
 * Search if the last acquisition of the current test is a localisation
 * 
 * @param integer $testId
 * @param integer $portNumber
 * 
 * @return boolean
 */
public static function hasLastAcquisitionLocalisation( $testId, $portNumber )
{
	$hasLastAcquisitionLoc = FALSE;
	$filePath = SMTBaseOtdrTraceFileUtil::buildLastMonitoringDetectionTraceNameOnOtu($testId, $portNumber);
	$filePathLoc = SMTBaseOtdrTraceFileUtil::buildLastMonitoringLocalisationTraceNameOnOtu($testId, $portNumber);
	
	//CSOR
	$filePath = SMTBaseOtdrTraceFileUtil::replaceExtension($filePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_CSOR_EXTENSION);
	if ( !file_exists( $filePath) )
	{
		//MSOR
		$filePath = SMTBaseOtdrTraceFileUtil::replaceExtension($filePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_MSOR_EXTENSION);
		if ( !file_exists( $filePath) )
		{
			//SOR
			$filePath = SMTBaseOtdrTraceFileUtil::replaceExtension($filePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
		}
	}
	//CSOR
	$filePathLoc = SMTBaseOtdrTraceFileUtil::replaceExtension($filePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_CSOR_EXTENSION);
	if ( !file_exists( $filePathLoc) )
	{
		//MSOR
		$filePathLoc = SMTBaseOtdrTraceFileUtil::replaceExtension($filePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_MSOR_EXTENSION);
		if ( !file_exists( $filePathLoc) )
		{
			//SOR
			$filePathLoc = SMTBaseOtdrTraceFileUtil::replaceExtension($filePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
		}
	}
	
	if ( ($filePath == NULL) || !file_exists( $filePath) )
	{
		if ( $filePathLoc!= NULL && file_exists( $filePathLoc) )
		{
			$hasLastAcquisitionLoc= TRUE;
		}
	}
	else
	{
		if ( ($filePathLoc!= NULL) && file_exists( $filePathLoc) )
		{
			// on doit prendre le fichier le plus recent
			$hasLastAcquisitionLoc= (filemtime($filePathLoc) >= filemtime($filePath) );
		}
	}
	return $hasLastAcquisitionLoc;
}

/**
 * Backup the last monitoring OTDR file (SOR,MSOR,CSOR) of the given test to SmartOTU directory by user session.
 *
 * @param integer $testId
 * @param boolean smartAcq FALSE to limit the search to sor file
 * 
 * @return string The new file path in SmartOtu directory
 */
public static function backupLastMonitoringTraceToSessionDirectory( $testId, $portNumber, &$isLoc, $smartAcq = TRUE )
{
	self::removeSessionsDirectoriesIfNeeded();
	self::checkOTDRFilesCountInSession();
	
	$filePath = SMTBaseOtdrTraceFileUtil::buildLastMonitoringDetectionTraceNameOnOtu($testId, $portNumber);
	$filePathLoc = SMTBaseOtdrTraceFileUtil::buildLastMonitoringLocalisationTraceNameOnOtu($testId, $portNumber);	
	$newFilePath = SMTBaseOtdrTraceFileUtil::buildLastMonitoringDetectionTraceNameOnSmartOtu($testId, $portNumber) ;
	$newFilePathLoc = $newFilePath;
	$isLoc = FALSE;
	
	if ( $smartAcq )
	{
		//CSOR
		$filePath = SMTBaseOtdrTraceFileUtil::replaceExtension($filePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_CSOR_EXTENSION);
		if ( file_exists( $filePath) )
		{
			$newFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($newFilePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_CSOR_EXTENSION);
		}
		else
		{
			//MSOR
			$filePath = SMTBaseOtdrTraceFileUtil::replaceExtension($filePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_MSOR_EXTENSION);
			if ( file_exists( $filePath) )
			{
				$newFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($newFilePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_MSOR_EXTENSION);
			}
			//SOR
			else
			{
				$filePath = SMTBaseOtdrTraceFileUtil::replaceExtension($filePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
				$newFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($newFilePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
			}
		}
		
		
		//CSOR
		$filePathLoc= SMTBaseOtdrTraceFileUtil::replaceExtension($filePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_CSOR_EXTENSION);
		if ( file_exists( $filePathLoc) )
		{
			$newFilePathLoc= SMTBaseOtdrTraceFileUtil::replaceExtension($newFilePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_CSOR_EXTENSION);
		}
		else
		{
			//MSOR
			$filePathLoc = SMTBaseOtdrTraceFileUtil::replaceExtension($filePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_MSOR_EXTENSION);
			if ( file_exists( $filePathLoc) )
			{
				$newFilePathLoc= SMTBaseOtdrTraceFileUtil::replaceExtension($newFilePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_MSOR_EXTENSION);
			}
			//SOR
			else
			{
				$filePathLoc = SMTBaseOtdrTraceFileUtil::replaceExtension($filePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
				$newFilePathLoc= SMTBaseOtdrTraceFileUtil::replaceExtension($newFilePathLoc, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
			}
		}
	}

	if ( $filePath== NULL || !file_exists( $filePath) )
	{
		if ( $filePathLoc == NULL || !file_exists( $filePathLoc) )
		{
			SMTLogger::getInstance()->trace( sprintf( "Failed to retrieve last monitoring OTDR acquisition trace: %s ", $filePath), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
			throw new SMTIOException( SMTIOException::FILE_NOT_FOUND, $filePath);
		}
		else
		{
			$isLoc = TRUE;
		}
	}
	else
	{
		if ( $filePathLoc!= NULL && file_exists( $filePathLoc) )
		{
			// on doit prendre le fichier le plus recent
			$isLoc = (filemtime($filePathLoc) >= filemtime($filePath) ); 
		}
	}

	if ( $isLoc )
	{
		SMTLogger::getInstance()->trace( sprintf( "copy last monitoring LOC OTDR acquisition trace %s to %s", $filePathLoc, $newFilePathLoc), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
		
		if ( !copy( $filePathLoc, $newFilePathLoc) )
		{
			throw new SMTIOException( SMTIOException::COPY_FAILURE, $newFilePathLoc);
		}
		return $newFilePathLoc;
	}
	else
	{
		SMTLogger::getInstance()->trace( sprintf( "copy last monitoring DET OTDR acquisition trace %s to %s", $filePath, $newFilePath), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
		
		if ( !copy( $filePath, $newFilePath ) )
		{
			throw new SMTIOException( SMTIOException::COPY_FAILURE, $newFilePath );
		}
	}

	return $newFilePath;
}

/**
 * Touch files in session directory if it exists and if:
 * current time - last update time > DEFAULT_MAX_LIFE_TIME_FOR_SESSION_FILES / 2.
 * 
 * @return $lastUpdateTime update time of the session directories
 */
static function touchSessionDirectoryFiles( $lastUpdateTime )
{    
    if ( time() - $lastUpdateTime > (self::DEFAULT_MAX_LIFE_TIME_FOR_SESSION_FILES / 2) )
    {
        $lastUpdateTime = time();
        $id = session_id();
        //if a session exists
        if ( $id != "")
        {
            $sessionDirectory = self::CUSTOM_SMARTOTU_SESSION_DIR.$id;
            
            //if session directory exists
            if ( is_dir( $sessionDirectory ) )
            {
                $otdrFiles = array();
                foreach ( glob( $sessionDirectory."/*.*") as $file )
                {
                    $otdrFiles[$file] = filemtime( $file );
                }
                	
                asort( $otdrFiles, SORT_NUMERIC );
                
                //touch with fake date (current date - 1s) to keep file timestamp order
                $newTime = time() - count($otdrFiles);
                foreach ( $otdrFiles as $fileName=>$lastUpdate)
                {
                    //SMTLogger::getInstance()->trace(sprintf("touch file %s, with value %s",$fileName,$lastUpdate), SMTLogger::DEBUG);
                    touch( $fileName, $newTime++ );
                }                
            }
        }
    }
    return $lastUpdateTime;
}

/**
 * Create directory by user session on disc if it doesn't exist.
 *
 * @return the session directory or NULL if no session is active.
 */
public static function createSessionDirectoryIfNeeded()
{
	$sessionDirectory = NULL;
	$id = session_id();
	if ( $id != "")
	{
		$sessionDirectory = self::CUSTOM_SMARTOTU_SESSION_DIR.$id;
		if (!is_dir( $sessionDirectory ) )
		{
			mkdir( $sessionDirectory, 0777);
		}
	}
	return $sessionDirectory."/";
}

/**
 * No more than 10 acquisitions stored:
 *  - if there are already 10 acquisitions, remove the oldest ones
 */
private static function checkOTDRFilesCountInSession()
{
	$sessionDirectory = NULL;
	$id = session_id();
	if ( $id != "")
	{
		$sessionDirectory = self::CUSTOM_SMARTOTU_SESSION_DIR.$id;
		if ( is_dir( $sessionDirectory ) )
		{
		    $otdrFileCount = 0;
		    $otdrFiles = array();
		    foreach ( glob( $sessionDirectory."/*.*") as $file )
		    {
		        $otdrFiles[$file] = filemtime( $file );
		        $otdrFileCount++;
		    }
		    
		    if ( $otdrFileCount >= self::DEFAULT_MAX_OTDR_FILES_IN_SESSION )
		    {
		        $filesCountToRemove =  $otdrFileCount - ( self::DEFAULT_MAX_OTDR_FILES_IN_SESSION  -1 );
		        asort( $otdrFiles, SORT_NUMERIC );
		        
		        for ( $i = 0; $i < $filesCountToRemove; $i++)
		        {
		            reset($otdrFiles);
		            $fileName = key( $otdrFiles );		        	
		        	unlink($fileName);		      
		        	array_shift( $otdrFiles );
		        }
		    }
		}
	}
}

/**
 * Remove session directories on disc if sessions have expired.
 * That method is intended to be called by PHP garbage collector through SMTFileSessionHandler.
 *
 * @param $maxlifetime integer max session life time (by default, set to DEFAULT_MAX_LIFE_TIME_FOR_SESSION_FILES = 30min)
 */
public static function removeSessionsDirectoriesIfNeeded( $maxlifetime = self::DEFAULT_MAX_LIFE_TIME_FOR_SESSION_FILES )
{
    $sessionFiles = glob(self::CUSTOM_SMARTOTU_SESSION_DIR."*/*.*");
	foreach ( $sessionFiles as $file )
	{
		if ( file_exists($file) && ( filemtime($file) + $maxlifetime < time() ) )
		{
			unlink($file);
		}
	}
	
	$sessionDirs = glob(self::CUSTOM_SMARTOTU_SESSION_DIR."*");	
	foreach ( $sessionDirs as $dir )
	{
		if ( is_dir( $dir ) )
		{
			$files = scandir($dir);
			if ( count( $files ) <= 2 ) // . and ..
			{
				//remove dir
				rmdir( $dir );
			}
		}
	}
}

/**
 * Remove the given session directory on disc.
 * That method is intended to be called by PHP session handler through SMTFileSessionHandler.
 *
 * @param $sessionId string the identifier of the session being removed.
 */
static function removeSessionDirectoryIfNeeded( $sessionId )
{
	$dir = self::CUSTOM_SMARTOTU_SESSION_DIR.$sessionId;
	
	if ( is_dir( $dir ) )
	{
	    $files = scandir($dir);
	    
    	foreach ( $files as $file )
    	{
    	    if ( $file != "." && $file != ".." )
    	    {    	    		
        	    if ( file_exists( $dir."/".$file) )
        	    {
    			    unlink( $dir."/".$file );
    	        }
    	    }
    	}
    	
		//remove dir
		rmdir( $dir );
	}		
}

/**
 * No more than 15 files stored in temp directory:
 *  - if there are already 15 files, remove the oldest ones
 */
public static function checkFilesCountInTempDir()
{
	$tempDirectory = self::SMART_OTU_TEMP_DIR;
	
	if (!is_dir( $tempDirectory ) )
	{
		mkdir( $tempDirectory, 0777);
	}		
			
	$tempFileCount = 0;
	$tempFiles = array();
	foreach ( glob( $tempDirectory."/*.*") as $file )
	{
		$tempFiles[$file] = filemtime( $file );
		$tempFileCount++;
	}

	if ( $tempFileCount >= self::DEFAULT_MAX_FILES_IN_TEMP_DIR )
	{
		$filesCountToRemove =  $tempFileCount - ( self::DEFAULT_MAX_FILES_IN_TEMP_DIR  -1 );
		asort( $tempFiles, SORT_NUMERIC );

		for ( $i = 0; $i < $filesCountToRemove; $i++)
		{
    		reset($tempFiles);
    		$fileName = key( $tempFiles );
    		unlink($fileName);
    		array_shift( $tempFiles );
		}
	}
}

}

?>