<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright VIAVI Solutions 2018. All rights reserved.
// *********************************************************
namespace app\admin;    

use app\http\SMTContext;
use app\serviceshelper\maintenance\SMTLicence;
use app\util\SMTLogger;
use app\parser\SMTOtuApi;
use app\serviceshelper\maintenance\SMTLicenseList;


/**
 * Handle the OTU mode: 
 *  - retrieve it from settings
 *  - store it in session variable
 * 
 * @author Sylvain Desplat
 *
 */
class SMTOtuMonitoringOptions
{        
    const OTU_MONITORING_OPTION  = "otumonitoringoption";
    const TAPPING     = "TAPPING";
    const LEGACY      = "LEGACY";
    const HIGHSENS    = "HIGHSENS";
    const NONE        = "";
    
    const DEFAULT_OTU_MONITORING_OPTION = self::HIGHSENS;
    
    
    /**
     * The supported OTU modes keys.
     *
     *
     * @return string The OTU mode key.
     */
    public static function getSupportedMonitoringOptions()
    {
    	return array(self::TAPPING, self::LEGACY, self::HIGHSENS, self::NONE );
    }    
    
    /**
     * Whether there is an OTU monitoring option TAPPING
     * @return boolean Whether there is an OTU monitoring option TAPPING
     */
    public static function hasTapping(SMTContext $context)
    {
    	return ( strpos( self::getOtuMonitoringOptions($context), self::TAPPING ) !== false);
    }

    /**
     * Whether there is an OTU monitoring option HIGHSENS
     * @return boolean Whether there is an OTU monitoring option HIGHSENS
     */
    public static function hasHighsens(SMTContext $context)
    {
    	return ( strpos( self::getOtuMonitoringOptions($context), self::HIGHSENS) !== false );
    }
    
    /**
     * Whether there is an OTU monitoring option LEGACY monitoring
     * @return boolean Whether there is an OTU monitoring option LEGACY
     */
    public static function hasLegacy(SMTContext $context)
    {
    	return ( strpos( self::getOtuMonitoringOptions($context), self::LEGACY) !== false);
    }
    
    /**
     * Whether there is no OTU monitoring option
     * @return boolean there is no OTU monitoring option
     */
    public static function isNone(SMTContext $context)
    {
    	return ( strcasecmp( self::getOtuMonitoringOptions($context), self::NONE) == 0 );
    }
    
    /**
     * Whether the string value contains an existing OTU mode
     *
     * @param string $option
     * @return boolean
     */
    public static function isValid( $option )
    {
    	if ( strpos( $option , self::TAPPING ) !== false|| 
    	     strpos( $option , self::LEGACY) !== false||
    		 strpos( $option , self::HIGHSENS) !== false||
    		 strcasecmp( $option , self::NONE) == 0
    	   )
    	{
    		return true;
    	}
    	return false;
    }
    
    /**
     * Set OTU option in session
     *
     * @param string $option the OTU Option
     */
    public static function setOtuMonitoringOptions( $option )
    {
    	if ( self::isValid( $option) )
    	{
    		$_SESSION[self::OTU_MONITORING_OPTION] = $option;
    	}
    }
    
    /**
     * Reset OTU option in session
     *
     */
    public static function resetOtuMonitoringOptions( )
    {
    	unset($_SESSION[self::OTU_MONITORING_OPTION]);
    	
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_DUAL_OS_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_EXT_TEMP_RANGE_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_HIGH_SEC]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_LIGHT_SOURCE]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_PEAK_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_REST_API]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_SAA_L2_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_SSH_KEY_ACCESS]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_TAPPING_MONITORING]);
    	unset($_SESSION[SMTLicenseList::OTU_OPTION_TEST_PON_MONITORING]);
    	
    }
    
    /**
     * Return the OTU option
     * 
     * @return string
     */
    public static function getOtuMonitoringOptions(SMTContext $context, $forceRetrieve = FALSE)
    {
        //update session current language
    	if ( $forceRetrieve || (!isset($_SESSION[self::OTU_MONITORING_OPTION] ) || $_SESSION[self::OTU_MONITORING_OPTION] == NULL ) )
        { 
        	$options = self::retrieveOtuMonitoringOptions($context);
            if ( self::isValid( $options ) )
            {
                self::setOtuMonitoringOptions( $options );
            }
        }
        
        $options = isset($_SESSION[self::OTU_MONITORING_OPTION])?$_SESSION[self::OTU_MONITORING_OPTION]: self::DEFAULT_OTU_MONITORING_OPTION;
        
        SMTLogger::getInstance()->trace("getOtuMonitoringOptions: ".$options);
        
        if ( self::isValid( $options ) )
        {
            return $options;
        }
        
        return self::DEFAULT_OTU_MONITORING_OPTION;
    }
    
    /**
     * Retrieve the OTU option from settings
     *
     */
    private static function retrieveOtuMonitoringOptions(SMTContext $context)
    {    	
    	$isTapping = SMTLicence::isTappingMonitoringOption($context);
   		$options = self::HIGHSENS;
   		if ( $isTapping )
   		{
   			$options .= ','.self::TAPPING;
   		}
   		
   		// Replace default HIGHSENS mode by LEGACY
   		if ( !SMTOtuType::isOTH() ) //no legacy mode on OTH
   		{
    		try
    		{
    			//retrieve monitoring mode
    			$mode = trim( $context->getOtuSocket()->sendReceive( SMTOtuApi::get_otu_mon_mode( ) ) );
    			if ( strpos( $mode, SMTOtuApi::LEGACY_MODE) !== false )
    			{
    				$options= self::LEGACY;
    			}
    		}
    		catch ( \Exception $e )
    		{
    			//don't throw exception here, either SmartOTU will not be initialized
    			SMTLogger::getInstance()->traceException($e);
    		}
   		}

    	    	
    	SMTLogger::getInstance()->trace("OTU monitoring options: ".$options);
    	
    	return $options;
    }
}
?>