"""
Module containing functions for user files rest api
"""
import os
import json
import xml.etree.ElementTree as ET
try:
    from urllib import quote  # Python 2.X
except ImportError:
    from urllib.parse import quote  # Python 3+

from marshmallow import Schema, fields
from webargs.bottleparser import use_kwargs

import PyPDF2
from bottle import request, response, static_file, Bottle

from rest_api.api.job_manager.plugin import JobManagerPlugin
from rest_api.api.job_manager import save_job_artifacts
from .schemas import UploadArgs

user_files = Bottle() # pylint: disable=C0103


METADATA_KEY = 'metadata'


@user_files.get('/api/filemgr/v1/userfiles')
def file_list():
    """Endpoint to list the files and directories of the base directory
    ---
    get:
      tags:
        - user files
      description: Endpoint to list the files and directories of the base directory
      responses:
        200:
          description: success
          schema: {'$ref': '#/definitions/UserDirectory'}
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member

    return get_dir_json(base_dir, base_dir)


@user_files.get('/api/filemgr/v1/userfiles/<file_path:path>')
def file_list_subdir(file_path):
    """
    Endpoint to either list the files and directories in the file_path or to
    return the file to the client
    ---
    get:
      tags:
        - user files
      description: >
                     Endpoint to either list the files and directories in the
                     file_path or to return the file to the client
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
        400:
          description: file or directory not found
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member

    rel_path = strip_basepath(file_path, base_dir)
    full_path = os.path.join(base_dir, rel_path)
    if os.path.isdir(full_path):
        return get_dir_json(full_path, base_dir)
    if os.path.isfile(full_path):
        metadata = get_metadata(file_path, base_dir)
        static_file_response = static_file(file_path, root=base_dir)

        for name, value in metadata.items():
            static_file_response.add_header(name, value)

        return static_file_response

    response.status = 400
    return {}

@user_files.delete('/api/filemgr/v1/userfiles/<file_path:path>')
def delete_file(file_path):
    """Endpoint for deleting a file from the instrument
    ---
    delete:
      tags:
        - user files
      description: Endpoint for deleting a file from the instrument
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
        404:
          description: invalid file path
        500:
          description: could not delete file
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member

    full_path = os.path.join(base_dir, file_path)
    if os.path.isfile(full_path):
        os.remove(full_path)
        response.status = 200
    else:
        response.status = 404


@user_files.post('/api/testdata/v1/userfiles', apply=JobManagerPlugin())
@user_files.post('/api/signature/v1/userfiles', apply=JobManagerPlugin())
@use_kwargs(UploadArgs(strict=True), locations=('files', 'form'))
def save_item(filedata, job_manager):
    """Endpoint for uploading an additional piece of testdata to the instrument
       like a signature capture, bar code , etc.
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload testdata to the instrument
      consumes:
        - multipart/form-data
      parameters:
        - in: formData
          schema: UploadArgs
      responses:
        200:
          description: success
    """
    reports_dir = request.app.config.get('user_files.reports_dir') #pylint: disable=no-member

    filedata.save(reports_dir, overwrite=True)

    filepath = os.path.join(reports_dir, filedata.filename)

    job = job_manager.load_job(1)

    if not job_manager.active_state:
        return

    #mobiletech sends the metadata as a 'files' type (while you might expect it
    #as 'forms', so we had to uses request.files to access it and convert it to a python object

    metadatafile = request.files.get(METADATA_KEY) #pylint: disable=no-member
    metadata = json.loads(metadatafile.file.read().decode('utf-8'))
    test_data = save_image_to_pdf(
        filepath,
        job.logo,
        get_type_of_file(metadata),
        job,
    )

    job_manager.add_test_data(test_data)

def strip_metadata_type_name(_type):
    """
    Function to strip the name of the type of image file being sent

    args:
        _type: full 'type' sent as part of metadata eg. "MISC.Signature"
    returns:
        stripped type field eg. "Signature"
    """
    return _type.split('.').pop()

def get_type_of_file(metadata):
    """
    Function to return the 'type' field from the incoming metadata

    args:
        metadata: python dict of metadata
    returns:
        'type' field of metadata eg. MISC.Signature
    """
    return strip_metadata_type_name(metadata['type']) if 'type' in metadata else 'Unknown'

@user_files.post('/api/filemgr/v1/userfiles', apply=JobManagerPlugin())
@use_kwargs(UploadArgs(strict=True), locations=('files', 'form'))
def save_file(filedata, job_manager):
    """Endpoint for uploading a file to the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload a file to the instrument
      consumes:
        - multipart/form-data
      parameters:
        - in: formData
          schema: UploadArgs
      responses:
        200:
          description: success
    """
    if not file_is_picture(filedata.filename):
        upload_dir = request.app.config.get('user_files.upload_dir') #pylint: disable=no-member

        filedata.save(upload_dir, overwrite=True)
        return

    reports_dir = request.app.config.get('user_files.reports_dir') #pylint: disable=no-member
    filedata.save(reports_dir, overwrite=True)
    filepath = os.path.join(reports_dir, filedata.filename)

    job = job_manager.load_job(1)

    if not job_manager.active_state:
        return

    test_data = save_image_to_pdf(filepath, job.logo, 'Picture', job)
    job_manager.add_test_data(test_data)

def file_is_picture(filepath):
    """Function to determine if the file is a picture
    """
    _, file_ext = os.path.splitext(filepath)
    return file_ext.lower() in ('.jpg', '.png')


def save_image_to_pdf(image_path, logo_path, image_type, job):
    """Function to save the image to a pdf

    Args:
        image_path (str): the filepath of the image
        logo_path (str): the filepath of the customer logo
        image_type (str): the type of image being saved
        job (Job): the active job object

    Returns:
        test_data (dict): the dictionary of the test data for the file
    """
    root, _ = os.path.splitext(image_path)
    pdf_file_path = root + '.pdf'

    save_job_artifacts.save_image_to_pdf(
        logo_path=logo_path,
        image=image_path,
        file_path=pdf_file_path,
        job=job,
        image_type=image_type,
    )

    test_data = {
        'test_type': image_type,
        'verdict': 'NotApplicable',
        'file_path': pdf_file_path,
        'reference_info': [],
        'sub_type_info': [
                {
                    "key": "testLabel",
                    "value": ""
                }
            ]
    }

    os.remove(image_path)

    return test_data

def get_dir_json(path, base_dir):
    """Function to create a dictionary representing the directory
    the swagger schema is below
    ---
    type:
      enum:
        - D
      type: string
      description: >
                      the type response for this object for directories this
                      will be D
    name:
      type: string
      description: The name of the directory on the file system
    fileUri:
      type: string
      description: the relative path to the directory from the base directory
    files:
        items:
          $ref: '#/definitions/UserFile'
        type: array
        description: list of UseFile objects in this directory
    subDirs:
      items:
        $ref: '#/definitions/UserDirectory'
      type: array
      description: list of UserDirectory for each subdirectory
    """
    res = {'type':'MISC.Directory'}
    dir_name = os.path.basename(os.path.normpath(path))
    dir_path_rel = os.path.relpath(path, base_dir)
    append_base = lambda c: os.path.join(path, c)

    content_paths_abs_file = map(append_base, os.listdir(path))
    content_paths_abs_dir = map(append_base, os.listdir(path))
    
    res['name'] = dir_name
    res['fileUri'] = dir_path_rel
        
    #Directory
    try:
        dirs = filter(os.path.isdir, content_paths_abs_dir)
        res['subDirs'] = [get_dir_json(d, base_dir) for d in  dirs]
    except:
        res['subDirs'] = []
    
    #File
    try:
        files = filter(os.path.isfile, content_paths_abs_file)
        res['files'] = [filepath_to_dict(f, base_dir) for f in  files]
    except:
        res['files'] = []
    
    return res


def filepath_to_dict(filepath, base_dir):
    """Function to create a dictionary representing the file path
    the swagger schema is below
    ---
    type:
      enum:
        - F
      type: string
      description: the type of this object for files this will be F
    name:
      type: string
      description: the name of the file
    path:
      type: string
      description: the relative path to the file from the base directory
    mtime:
      type: string
      description: time of most recent content modification fo rthis file
    size:
      type: number
      description: 'the size of the file, in bytes'
    """
    
    res = {'type': 'MISC.File'}
    fstats = os.stat(filepath)
    file_name = os.path.basename(filepath)
    file_path_rel = os.path.relpath(filepath, base_dir)
    res['name'] = file_name
    res['modifiedTime'] = fstats.st_mtime
    res['size'] = fstats.st_size
    res['fileUri'] = quote(file_path_rel)
    return res


def strip_basepath(file_path, base_path):
    """Strip any directories matching the base directories from the file path

    This funciton is so that if the user supplies an absolute path that matches
    the base path the api will return the file
    """
    file_path_dirs = file_path.split('/')
    base_path_dirs = base_path.split('/')

    for dir_name in base_path_dirs:
        if dir_name in file_path_dirs and file_path_dirs.index(dir_name) == 0:
            file_path_dirs.remove(dir_name)

    return '/'.join(file_path_dirs)


def get_metadata(filename, root):
    """Function to attempt to extract metadata from a pdf file for use in
    StrataSync

    Args:
        filename (str): name or path of the file
        root (str): root path of the file for lookups

    Retruns:
        metadata (dict): key value pairs of the metadata extracted from the
            files
    """
    root = os.path.abspath(root) + os.sep
    filename = os.path.abspath(os.path.join(root, filename.strip('/\\')))

    if not os.path.isfile(filename):
        return {}

    with open(filename, 'rb') as work_file:
        try:
            pdf_reader = PyPDF2.PdfFileReader(work_file)
            pdf_keywords = pdf_reader.documentInfo.get('/Keywords')
        except PyPDF2.utils.PdfReadError:
            pdf_keywords = None

    if not pdf_keywords:
        return {}

    xml_root = ET.fromstring(pdf_keywords)
    xml_metadata = {child.tag: child.text for child in xml_root}

    metadata = get_stratasync_repr(xml_metadata)

    return metadata


def get_stratasync_repr(xml_metadata):
    """Function to convert from the xml representation of the metadata
    to something that stratasync wants

    Args:
        xml_metadata (dict): dictionary representing the metadata pulled from
            xml

    Returns:
        metadata (dict): dictionary of metadata in stratasync format
    """
    keys_xml_to_stratasync = {
        'Type': 'type',
        'Verdict': 'result',
        'TechnicianId': 'techId',
        'CustomerName': 'customerName',
        'TestLocation': 'loc',
        'Workorder': 'wo',
        'Comments': 'comments',
        'AcqDate': 'fiberAcqDate',
        'FiberNumber': 'fiberNumber',
        'TestLocationB': 'locB',
        'Wavelengths': 'fiberWavelengths',
        'WorkflowId': 'workflowId',
        'TotalLoss': 'fiberTotalLoss',
        'TotalLength': 'fiberTotalLength',
        'LinkOrl': 'fiberLinkOrl',
        'TestDirection': 'fiberDirection',
        'CableId': 'cableId',
        'FiberId': 'fiberId',
        'FiberLabel': 'fiberLabel',
        'ContractorId': 'contractorId',
    }

    xml_metatdata_ss_keys = {keys_xml_to_stratasync.get(key, key): value
                             for key, value in xml_metadata.items()}

    info = {key: value
            for key, value in xml_metatdata_ss_keys.items()
            if key not in ('type', 'techId')}

    metadata = {key: value
                for key, value in xml_metatdata_ss_keys.items()
                if key in ('type', 'techId')}
    metadata['info'] = info

    return metadata
