"""
Module that contains Fiber's adaptations to schemas for the rest api
"""
from marshmallow import Schema, fields
from marshmallow.validate import OneOf, Length

from .schemas_viavi import FilePathSchema, ArtifactSchema, ReferenceInfoSchema
from .schemas_viavi import SubTypeInfoSchema, ResultsSchema

UI_STRING_VALIDATOR = Length(max=60)


def make_job_schema(config_schema=None):
    """Job Schema factory function to create a job schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """

    class JobSchema(Schema):
        """
        Schema for the job object

        """
        job_number = fields.Str(
            load_from='jobNumber',
            dump_to='jobNumber',
            validate=UI_STRING_VALIDATOR,
            missing='--',
            description='string representation of the job number')
        customer_name = fields.Str(
            required=True,
            load_from='customerName',
            dump_to='customerName',
            validate=UI_STRING_VALIDATOR,
            description='the name of the end customer for the job')
        technician_id = fields.Str(
            load_from='technicianId',
            dump_to='technicianId',
            validate=UI_STRING_VALIDATOR,
            missing='--',
            description="the technician's identification number")
        test_location = fields.Str(
            load_from='testLocation',
            dump_to='testLocation',
            missing='--',
            validate=UI_STRING_VALIDATOR,
            description='the start location for the test being performed')
        logo = fields.Str(
            required=False,
            load_from='logo',
            dump_to='logo',
            missing="",
            description='the path to the logo that will be added to a report')
        cable_id = fields.Str(
            required=False,
            load_from='cableId',
            dump_to='cableId',
            missing="",
            description='the Cable Id for the Job being tested')
        test_location_b = fields.Str(
            required=False,
            load_from='testLocationB',
            dump_to='testLocationB',
            missing="",
            description='the end location for the test being performed')
        fiber_id = fields.Str(
            required=False,
            load_from='fiberId',
            dump_to='fiberId',
            missing="",
            description='the Fiber Id for the Job being tested')
        job_comments = fields.Str(
            required=False,
            load_from='comments',
            dump_to='comments',
            missing="",
            description='comments to be applied to the Job being tested')
        contractor_id = fields.Str(
            required=False,
            load_from='contractorId',
            dump_to='contractorId',
            missing="",
            description='the contractor Id for the Job being tested')
        test_data = fields.Nested(
            'TestDataSchemaFiber',
            required=False,
            missing=[],
            load_from='testData',
            dump_to='testData',
            many=True,
            description='list containing zero or more test_data dictionaires')
        test_plan = fields.Nested(
            make_planned_test_schema(config_schema),
            load_from='testPlan',
            dump_to='testPlan',
            many=True,
            description='list containing zero or more planned tests for the job')
        workflow_id = fields.Integer(
            required=False,
            missing=0,
            load_from="workflowId",
            dump_to="workflowId",
            description='integer, id of the workflow')
        wo_attribs = fields.Dict(
            required=False,
            missing={},
            load_from='woAttribs',
            dump_to='woAttribs',
            many=True,
            description='work order attributes from StrataSync Southbound CDM')
        cdm_string = fields.Str(
            required=False,
            load_from='cdmString',
            dump_to='cdmString',
            description='the cdm string')

    return JobSchema

class TestDataSchemaFiber(Schema):
    """
    Schema for test data entries

    """
    test_type = fields.Str(
        required=True,
        load_from='testType',
        dump_to='testType',
        description='the type of test that created this test data')
    file_path = fields.Str(
        required=True,
        load_from='filePath',
        dump_to='filePath',
        description='the path to the underlying file for this test data entry')
    verdict = fields.Str(required=True,
                         load_from='verdict',
                         dump_to='verdict',
                         validate=OneOf(
                             ('Pass', 'Fail', 'Marginal', 'NotApplicable', 'Skipped')
                         ),
                         description='the verdict of the test')
    comments = fields.Str(
        required=False,
        load_from='comments',
        dump_to='comments',
        description='any comments on the particular test item')
    reference_info = fields.Nested(
        'ReferenceInfoSchema',
        required=False,
        many=True,
        load_from='referenceInfo',
        dump_to='referenceInfo',
        description=(
            'list of zero or more dictionaries containing '
            'reference info differentiating this test data '
            'from others like it'))
    results = fields.Nested(
        'ResultsSchema',
        required=False,
        many=True,
        load_from='results',
        dump_to='results',
        description=(
            'list of zero or more dictionaries containing '
            'summary result values'))
    sub_type_info = fields.Nested(
        'SubTypeInfoSchema',
        required=False,
        many=True,
        load_from='subTypeInfo',
        dump_to='subTypeInfo',
        description=(
            'list of zero or more dictionaries containing '
            'configInfo that can differentiate this test data '
            'when the referenceInfo fields are identical '
            '(useful for manualStep testtype)'))

def make_planned_test_schema(config_schema):
    """Planned test Schema factory function to create a planned test schema class
    based on the config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    def make_config(config_schema):
        if not config_schema:
            return fields.Dict(allow_none=True)
        return fields.Nested(
            config_schema,
            allow_none=True,
            required=False,
            load_from='config',
            dump_to='config',
            description='the configuration of the particular test to be launched'
        )

    class PlannedTestSchema(Schema):
        """
        Schema for tests that are planned to be performed as part of a
        test plan

        """
        test_type = fields.Str(required=True,
                               load_from='testType',
                               dump_to='testType',
                               description='the type of test required')
        test_label = fields.Str(required=False,
                                load_from='testLabel',
                                dump_to='testLabel',
                                description='The Label to be displayed for the Test Type')
        status = fields.Str(
            missing='To Do',
            load_from='status',
            dump_to='status',
            validate=OneOf(
                ('To Do',
                 'Pass',
                 'Fail',
                 'Marginal',
                 'Complete',
                 'Skipped')),
            description='the status of the test')
        reference_info = fields.Nested(
            'ReferenceInfoSchema',
            required=False,
            many=True,
            load_from='referenceInfo',
            dump_to='referenceInfo',
            missing=[],
            description=(
                'list of zero or more dictionaries containin'
                'reference info differentiationg this test data '
                'from others like it'))
        reference_info_str = fields.Method(
            serialize='make_reference_info_str',
            dump_to='referenceInfoStr',
            description=(
                'the string representation'
                ' of the reference information '
                'can be used to display on user interfaces '
                'ignored on POST and PUT requests'))
        procedures = fields.Str(
            required=False,
            missing='',
            load_from='procedures',
            dump_to='procedures',
            description='path to method and procedures for this planned test')
        test_data = fields.Nested(
            'TestDataSchemaFiber',
            missing=[],
            many=True,
            load_from='testData',
            dump_to='testData',
            description='list containing zero or more test_data dictionaires')
        config = make_config(config_schema)
        sub_type_info = fields.Nested(
            'SubTypeInfoSchema',
            missing=[],
            required=False,
            many=True,
            load_from='subTypeInfo',
            dump_to='subTypeInfo',
            description=(
                'list of zero or more dictionaries containing '
                'configInfo that can differentiate this test data '
                'when the referenceInfo fields are identical '
                'useful for manualStep testtype'))
        workflow_id=fields.Integer(
            required=False,
            missing=0,
            load_from="workflowId",
            dump_to="workflowId",
            description='integer, workflowId of the planned test')


        def make_reference_info_str(self, obj):  # pylint: disable=no-self-use
            """Method to create the reference information string based on the
            reference_info list in the PlannedTest object

            Args:
                obj (PlannedTest object): the required object input for method
                    fields

            Returns:
                reference_info_str (str): the string representation of the
                    reference info
            """
            reference_info_strings = ('{}: {}'.format(item['key'], item['value'])
                                      for item in obj.reference_info)
            reference_info_str = ' , '.join(reference_info_strings)
            return reference_info_str

    return PlannedTestSchema


# class ResultsSchema(Schema):
#     """
#     Schema for result values for the test
#     each test may have zero or more results
#
#     """
#     key = fields.Str(required=True,
#                      load_from='key',
#                      dump_to='key',
#                      validate=UI_STRING_VALIDATOR,
#                      description='the field name for this result')
#     value = fields.Str(
#         required=True,
#         load_from='value',
#         dump_to='value',
#         validate=UI_STRING_VALIDATOR,
#         description='the entered value for this particular field')
#
#
# class SubTypeInfoSchema(Schema):
#     """
#     Schema for configuration information for the test type
#     used as a secondary differentiator after any refInfo content
#     each test may have zero or more sub-type information
#     entries to differentiate it from other tests of the same type
#     with the same reference info
#     """
#     key = fields.Str(required=True,
#                      load_from='key',
#                      dump_to='key',
#                      validate=UI_STRING_VALIDATOR,
#                      description='the field name for this reference')
#     value = fields.Str(
#         required=True,
#         load_from='value',
#         dump_to='value',
#         validate=UI_STRING_VALIDATOR,
#         description='the entered value for this particular field')
