'''Module containing the DataStore class for working with rest api database
'''
import json

import os
import tinydb
import logging

log = logging.getLogger(__name__)

DEFAULT_JOB_INIT_6_4 = ('{"jobNumber":"--","customerName":"--",'
                        '"technicianId":"--","testLocation":"--",'
                        '"logo":"","workflowId":0}')

class DataStore(object):
    '''
    Class to manage the datastore currently implemented as a file on the file system
    '''
    def __init__(self, job_manager_file):
        dir_path = os.path.dirname(job_manager_file)
        if not os.path.isdir(dir_path):
            os.makedirs(dir_path)
        try:
            self.db = tinydb.TinyDB(job_manager_file) # pylint: disable=C0103
        except ValueError:
            log.warning("Unable to parse DB file - starting new DB")
            open(job_manager_file, 'w').close()
            self.db = tinydb.TinyDB(job_manager_file) # pylint: disable=C0103

        self.jobs = self.db.table('jobs')
        self.active = self.db.table('active')
        self.job_item_active_index = self.db.table('job_item_active_index')

        if not self.jobs:
            default_job = json.loads(DEFAULT_JOB_INIT_6_4)
            self.jobs.insert(default_job)

        if not self.active:
            self.active.insert({'active_job': None})

        if not self.job_item_active_index:
            self.job_item_active_index.insert({'active_job_item_index': None})

        self.job_manager_file = job_manager_file

    def get_jobs_eids(self):
        """
        Method to return a list of all the element ids in the jobs table
        of the datastore

        Returns:
            (list): the list of element ids (int)
        """
        return [element.eid for element in self.jobs.all()]

    def read_job(self, element_id):
        '''
        Method to read the job from the datastore

        Returns: the dictionary representation of the job or None
            if the job does not exist
        '''
        return self.jobs.get(eid=element_id)

    def write_job(self, job_details):
        """
        Method to job record in the datastore with new information

        Note: currently the implementation is hardcoded to eid = 1

        Args:
            job_details (dict): the dictionary with the job information to
            write into the datastore
        """
        self.jobs.update(job_details, eids=[1])

    def delete_job(self):
        """
        Method to delete a job from the datastore

        Note: currently the implementation is hardcoded to delete job 1
        if it exists and do nothing otherwise
        """
        if self.jobs.contains(eids=[1]):
            self.jobs.remove(eids=[1])

    def read_active(self):
        '''
        Method to read the job that is currently active

        Returns:
            Dictionary containing the active job or None
        '''
        return self.active.get(eid=1)

    def set_active(self, active_job_eid):
        '''
        Method to set the active job

        Args:
            active_job_eid (int): the element id of the active job or
                None
        '''
        self.active.update({'active_job': active_job_eid}, eids=[1])

    def read_active_job_item_index(self, active_job_eid):
        '''
        Method to read the job item index that is currently active

        Args:
            active_job_eid (int): the element id of the active job or
                None

        Returns:
            Dictionary containing the active job item index or None
        '''
        return self.job_item_active_index.get(eid=active_job_eid)

    def set_active_job_item_index(self, active_job_eid, job_item_active_index):
        '''
        Method to set the active job item index

        Args:
            active_job_eid (int): the element id of the active job or
                None
            job_item_active_index (int): the index of the active job-item of the active job
        '''
        self.job_item_active_index.update({'active_job_item_index': job_item_active_index}, eids=[active_job_eid])

