"""
Module containing endpoints for the job manager api
"""
import os.path
import json
import logging

from webargs.bottleparser import use_args
from bottle import response, request, Bottle

from .save_job_artifacts import create_artifacts
from .schemas_viavi import FilePathSchema
from .schemas_viavi import TestDataSchema
from .schemas_viavi import ArtifactSchema
from .plugin import JobManagerPlugin, SideEffect

app = Bottle() #pylint: disable=invalid-name

#Inject job_manger object into the view function for each route
app.install(JobManagerPlugin())
app.install(SideEffect())


def get_job_schema(request): # pylint: disable=redefined-outer-name
    """Function to get a job schema class from the application configuration

    See https://webargs.readthedocs.io/en/latest/advanced.html#schema-factories
    for information

    args:
        request: Bottle request object containing the configuration information
            for the app
    returns:
        a job schema instance for `use_args` to process
    """
    job_schema = request.app.config["rest_api.product_specific"].job_schema
    return job_schema(strict=True)

@app.get('/api/v1/cdm/workflow')
def cdm_workflow(job_manager):
    """
    Workflow endpoint
    ---
    get:
        workflow bloc on cdm bloc
    """
    job = job_manager.load_job(1)
    workflow = job.get_cdm_workflow()

    return workflow

@app.get('/api/v1/jobs')
@app.get('/api/v1/jobs/')
def job_summaries(job_manager):
    """
    Job summaries endpoint
    ---
    get:
        tags:
          - jobs
        description: List of all jobs
        responses:
            200:
                'schema':
                    title: Jobs
                    type: array
                    items: {'$ref': '#/definitions/JobSummary'}
                description: Success. Return an array of jobs currently in job manager
    """
    response.content_type = 'application/json'
    return json.dumps(job_manager.get_job_summaries())


@app.get('/api/v1/jobs/<job_id:int>')
def specific_job_details(job_id, job_manager):
    """
    Endpoint to get details about a particular job
    ---
    get:
        tags:
          - jobs
        description: details about a particular job
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
        responses:
            200:
                'schema': JobSchema
                description: Job found. Return job.
            404:
                description: Job not found.
    """
    job = job_manager.load_job(job_id)

    if not job:
        response.status = 404
        return None

    response.status = 200
    return job_manager.dump_job()

@app.post('/api/v1/jobs')
@app.post('/api/v1/jobs/')
@use_args(get_job_schema)
def create_job(job_input, job_manager, side_effect):
    """
    Endpoint to create a new job
    ---
    post:
        tags:
          - jobs
        description: create a new job
        parameters:
            -   in: body
                name: Job
                schema:
                  $ref: "#/definitions/Job"
                required: true
        responses:
            201:
                'schema': JobSchema
                description: Job created
    """
    job_manager.load_job(1)

    output_job = job_manager.save_job(job_input)
    response.status = 201

    if job_manager.active_state:
        side_effect(output_job)

    return output_job


@app.post('/api/v1/active/testData')
@use_args(TestDataSchema(strict=True))
def add_test_data(test_data, job_manager):
    """
    Endpoint to add test data to the active job
    ---
    post:
        tags:
          - jobs
        description: adds test data to the active job
        parameters:
            -   in: body
                name: Test Data
                schema:
                  $ref: "#/definitions/TestData"
                required: true
        responses:
            201:
                'schema': {'$ref': '#/definitions/TestData'}
                description: Test data added to job
            202:
                description: No active job.  Test data not stored
    """
    #This will need to change when there is more than one job
    #likely by looking up the active job_id and getting that
    job = job_manager.load_job(1)

    if not job_manager.active_state or not job:
        response.status = 202
        return {}

    job_manager.add_test_data(test_data)
    response.status = 201
    return TestDataSchema().dump(test_data).data

@app.get('/api/v1/active')
def get_active_state(job_manager):
    """
    Endpiont for determining the active job
    ---
    get:
        tags:
          - jobs
        description: returns the active job
        responses:
            200:
                description: Success.
                schema:
                    title: Is Active
                    type: boolean
    """
    job_manager.load_job(1)
    if job_manager.active_state:
        return "1"
    return "0"

@app.put('/api/v1/active')
def set_active_state(job_manager, side_effect):
    """
    Endpiont for determining the active job
    ---
    put:
        tags:
          - jobs
        description: sets the active job
        responses:
            200:
                description: Success.
                schema:
                    title: Is Active
                    type: boolean
    """
    request_body = request.body.read() #pylint: disable=no-member
    try:
        active_job_eid = int(request_body)
    except ValueError:
        response.status = 400
        return

    job_manager.load_job(1)

    if active_job_eid == 1 or active_job_eid == 0:
        job_manager.set_active(active_job_eid)
        response.status = 200
        job_info = job_manager.dump_job()

        if active_job_eid == 1:
            side_effect(job_info)
        #reset job item to 0 when job is activated or de-activated
        #mylogger.debug("set_active_test_index(*1*) for activate/de-activate job, active_job_eid=" + str(active_job_eid) + ", index=-1")
        job_manager.set_active_test_plan_index(active_job_eid, -1)
    else:
        response.status = 400
    
    return{'Hello'}

@app.get('/api/v1/active/testDataIndex')
def get_active_test_index(job_manager):
    """
    Endpiont for determining the launched job item of active test plan
    ---
    get:
        tags:
          - jobs
        description: returns the index of the launched job item of active test plan
        responses:
            200:
                description: Success.
                schema:
                    title: Index of Launched Job Item in Active Test Plan
                    type: integer
    """
    active_job_eid=1
    job = job_manager.load_job(active_job_eid)

    if not job:
        response.status = 404
        return None
    response.status = 200
    indx = str(job_manager.get_active_test_plan_index(active_job_eid))
    #mylogger.debug("***get_active_test_index(), active_job_eid=" + str(active_job_eid) + ", index=" + indx)
    if (indx == 'None'):
       indx = "-1";
    return indx


@app.post('/api/v1/active/testDataIndex')
def set_active_test_index(job_manager):
    """
    Endpiont for record the launched job item of active test plan
    ---
    get:
        tags:
          - jobs
        description: set the index of the launched job item of active test plan
        responses:
            200:
                description: Success.
                schema:
                    title: Index of Launched Job Item in Active Test Plan
                    type: integer
    """
    request_body = request.body.read() #pylint: disable=no-member
    try:
        active_test_plan_eid = int(request_body)
    except ValueError:
        response.status = 400
        return

    active_job_eid=1
    job = job_manager.load_job(active_job_eid)

    #mylogger.debug("***set_active_test_index(), active_job_eid=" + str(active_job_eid) + ", index=" + str(active_test_plan_eid))
    job_manager.set_active_test_plan_index(active_job_eid, active_test_plan_eid)
    response.status = 200
    return


@app.put('/api/v1/jobs/<job_id:int>')
@use_args(get_job_schema)
def update_job_info(job_input, job_id, job_manager, side_effect):
    """
    Endpiont for determining the active job
    ---
    put:
        tags:
          - jobs
        description: updates the job information
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
            -   in: body
                name: Job
                schema:
                  $ref: "#/definitions/Job"
                required: true
        responses:
            200:
                description: Success.
                schema:
                'schema': JobSchema
                description: Job created
            404:
                description: Job not found

    """
    job_manager.load_job(job_id)
    if not job_manager.job:
        response.status = 404
        return {}
    output_job = job_manager.save_job(job_input)

    if job_manager.active_state:
        side_effect(output_job)

    response.status = 201
    return output_job


@app.delete('/api/v1/jobs/<job_id:int>')
def remove_job(job_id, job_manager):
    """
    Endpoint for deleting a job
    ---
    delete:
        tags:
          - jobs
        description: deletes the job
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
        responses:
            204:
                description: No Content
            404:
                description: Job Not Found
    """
    if job_manager.load_job(job_id):
        job_manager.delete_job()
        response.status = 204
    else:
        response.status = 404


@app.post('/api/v1/jobs/<job_id:int>/report')
@use_args(ArtifactSchema(strict=True))
def save_to_pdf(artifact_args, job_id, job_manager):
    """
    Endpoint for saving the contentes of a job to pdf
    ---
    post:
        tags:
          - jobs
        description: saves the contents of a job to a pdf report and optionally create a zip file
        parameters:
            -   in: path
                name: job_id
                required: true
                type: integer
            -   in: body
                name: Artifact
                schema: ArtifactSchema
                required: true
        responses:
            201:
                description: Summary report successfully created
                schema: FilePathSchema
            403:
                description: Filepath not found
            404:
                description: Job Not Found
    """
    job = job_manager.load_job(job_id)

    if not job:
        response.status = 404
        return {}

    file_path_return = create_artifacts(job=job, **artifact_args)
    if not file_path_return:
        response.status = 403
        return {}

    response.status = 201
    return {'filePath': file_path_return}

@app.post('/api/v1/jobs/jobManagerFile')
@use_args(FilePathSchema(strict=True))
def load_job_from_file(file_path_input, job_manager, side_effect):
    """
    Endpoint for loading a job to the datastore from a file located at a path supplied by the client
    ---
    post:
        tags:
          - jobs
        description: loads job to the datastore from a file located at a path supplied by the client
        parameters:
            -   in: body
                name: FilePathSchema
                schema: FilePathSchema
                required: true
        responses:
            201:
                description: Job loaded successfully
                schema: JobSchema
            422:
                description: Cannot process request for a reason supplied by the response
    """
    file_path = file_path_input['file_path']

    if not os.path.isfile(file_path):
        response.status = 422
        return {"filePath": ["Bad file path"]}

    active_job_eid=1
    job_manager.load_job(1)
    output_job = job_manager.load_job_from_file(file_path)

    if not output_job:
        response.status = 422
        return{file_path: ["Invalid job file"]}

    if job_manager.active_state:
        side_effect(output_job)
    #reset job item to 0 when job is loaded
    #mylogger.debug("set_active_test_index(*2*) for loading job, active_job_eid=" + str(active_job_eid) + ", index=-1")
    job_manager.set_active_test_plan_index(active_job_eid, -1)

    response.status = 201
    return output_job


@app.error(422)
def handle_unprocessable_entity(error):
    """
    Method to make error 422 error responses raised from webargs
    more consumer friendly

    The response body will now contain a representation of the schema
    parsing error in the request.  The message will contain information
    like:

    '{"customerName": ["Missing data for required field."]}'

    Args:
        error (HTTPError): the error raised by webargs due to parsing errors
    """
    return error.body
