try:
    from c_struct import *
except ImportError:
    from .c_struct import *


def crc16(input_data=None):
    crc = 0
    if len(input_data) % 2 > 0:
        print("ERROR 12554")

    for i in range(0, int(len(input_data)), 2):
        byte = input_data[i + 1] + (input_data[i] << 8)
        for bit in range(16):
            bit1 = byte & 0x1
            byte >>= 1
            bit1_crc = crc & 0x1
            crc >>= 1
            if (bit1_crc ^ bit1) == 0x0001:
                crc = (crc + 0x8000) ^ 0x408
    return crc


class FirstPart(Section):
    fields = [
        UNSIGNED_16("Crc"),
        UNSIGNED_16("Complement_Crc"),
        UNSIGNED_16("Section_Size"),
        STRING("Section_Name", 10),
        SIGNED_16("Section_Version"),
        BOOLEAN("Is_Last_Section"),
        Align()
    ]

    def is_valid(self):
        if self.data["Crc"].value() != ((~self.data["Complement_Crc"].value() ) & 0xFFFF):
            print("ERROR (%X != %X)" % (self.data["Crc"].value() , self.data["Complement_Crc"].value() ))
            return False
        else:
            return True


class FunctionInfo(Section):
    fields = [
        STRING("Name", 12, "no name"),
        SIGNED_16("Max_Power")
    ]

    def to_string(self):
        res = [self._name]
        for item in self._fields:
            res.append("\t%s"%item)
        return res

    def load(self, buffer, raw = None):
        if type(buffer) == bytes:
            self._bin_data = buffer
            buffer = unpack(self.pack(), self._bin_data)
        else:
            self._bin_data = raw

        offset = 0
        bin_offset = 0
        for v in self._fields:
            offset += v.load(buffer[offset:], self._bin_data[bin_offset : bin_offset + v.size()])
            bin_offset += v.size()
        return offset


class HeaderPart(Section):
    fields = [
        UNSIGNED_32("Header_Identity"),
        UNSIGNED_32("Module_Design"),
        STRING("Cfg_File",12),
        SIGNED_16("Jtag_Ir"),
        SIGNED_16("Jtag_Dr"),
        SIGNED_16("Hdw_Release"),
        SIGNED_16("Soft_Fab_Release"),
        DATETIME("Date_Fab"),
        STRING("Serial_Number", 11),
        STRING("Module_Name", 21),
        STRING("Bar_Code", 22),
        DATETIME("Date_Calib"),
        SIGNED_16("Soft_Cal_Release"),
        BOOLEAN("Calibrated"),
        Align(),
        SIGNED_16("Number_Of_Functions"),
        Array("Function", 10, FunctionInfo),
        STRING("Customer_Id", 5),
        STRING("Prefix", 5),
    ]


class CalibrationCommun(Section):
    fields = [
        FirstPart("First"),
        HeaderPart("Header"),
        Padding("Free", 512)
    ]

    def is_valid(self):
        return (self.data["First"].is_valid() and (crc16(self._bin_data[4:]) == self.data["First"].data["Crc"].value()))

    def update_crc(self):
        data = []
        for v in self._fields:
            data.extend(v.save())
        raw = pack(self.pack(), *data)
        self._bin_data = raw
        crc = crc16(raw[4:])
        self.data["First"].data["Crc"].value(crc)
        self.data["First"].data["Complement_Crc"].value(((~crc) & 0xFFFF))

    def init(self, platform):
        self.data["First"].data["Section_Size"].value(self.size())
        super(CalibrationCommun, self).init(platform)
        self.update_crc()

    def set(self, field, value):
        self.data["Header"].data[field].value(value)
        self.update_crc()

    def save(self):
        self.update_crc()
        data = []
        for v in self._fields:
            data.extend(v.save())
        return data

class ProdSection(Section):
    fields = [
        FirstPart("First"),
        Array("Prod_Parametres", 10, SIGNED_32),
        BLOB("Prod", 1024)
    ]

    def is_valid(self):
        return (self.data["First"].is_valid() and (crc16(self._bin_data[4:]) == self.data["First"].data["Crc"].value()))

    def update_crc(self):
        data = []
        for v in self._fields:
            data.extend(v.save())
        raw = pack(self.pack(), *data)
        self._bin_data = raw
        crc = crc16(raw[4:])
        self.data["First"].data["Crc"].value(crc)
        self.data["First"].data["Complement_Crc"].value(((~crc) & 0xFFFF))

    def init(self, platform):
        self.data["First"].data["Section_Size"].value(self.size())
        super(ProdSection, self).init(platform)
        self.update_crc()

    def save(self):
        self.update_crc()
        data = []
        for v in self._fields:
            data.extend(v.save())
        return data


class CommonHeader(Section):
    fields = [
        CalibrationCommun("Common"),
        ProdSection("Prod"),
        Padding("Free", 0x800)
    ]
