#! /usr/bin/python3
'''
Fiber version called by MTS/T-BERD's ISU service when notified across Dbus by USC.
(Poached & adapted from Germantown's solution for 5800v2)
One TODO: clear the /tmp/cdm/ file once it has been converted.
'''

import shutil
import os
import logging
import traceback

from rest_api.api.job_manager.cdm_schemas_viavi import make_cdm_schema
from rest_api.products.mts_services.instrument_jobs import notify_job_received, get_job_dir, get_job_extension, get_template_extension




CDM_JOB_NAME = '/tmp/cdm/workflow.cdm.json'

log = logging.getLogger(__name__)

def main():
    """Function to open the CDM workflow file sent from StrataSync 
    and save individual CDM files for each job it contains
    """
    with open(CDM_JOB_NAME) as workflow_file:
        workflow_json_string = workflow_file.read()
    parse_cdm_job(workflow_json_string)

def parse_cdm_job(cdm_json):
    """Function to parse the contents of a CDM workflow file from StrataSync
    and save individual jobs as separate files

    args:
        cdm_json (str): the json contents of the CDM file
    """
    try:
        top_schema = make_cdm_schema()
        cdm_job = top_schema().loads(cdm_json).data

        for cdm in cdm_job['cdm']:
            single_cdm = {'cdm_version': cdm_job['cdm_version'],
                          'cdm': [cdm]}

            workorder_id = cdm['workflow']['workorder_id']
            file_path = ('{}{}' + get_job_extension()).format(get_job_dir(), workorder_id)
            cdm_str = top_schema().dumps(single_cdm).data
            save_job_file(file_path, cdm_str)
            log.debug('parse_cdm_job: %s saved (Job Id %s)', file_path, workorder_id )

        return True

    except:
        print(traceback.format_exc())
        return False

def save_job_file(file_path, cdm_str):
    """Function to save a json string (in cdm format) to a file

    args:
        file_path (str): the path of the file to create
        cdm_str (str): the content to save
    """
    with open(file_path, 'wb') as job_file:
        job_file.write(cdm_str.encode("utf-8"))




if __name__ == "__main__":
    main()




