<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI
// Copyright JDSU 2018. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otdr;

use app\http\SMTSessionDirectoryUtil;

use app\serviceshelper\monitoring\SMTOtdrReferenceFiles;
use app\util\SMTIOException;

use app\util\SMTLogger;


/**
 * Utility class to handle OTDR trace files on OTU file system
 * 
 * @author Sylvain Desplat
 */
class SMTHighsensOtdrTraceFileUtil extends SMTBaseOtdrTraceFileUtil
{     
    /**
     * Copy the measure acquisition or the last acquisition from SmartOTU directory to OTU reference detection and localisation directories.
     * Don't remove the mesure acquisition from SmartOTU directory (can be used again in case of test addition failure)
     *
     * @param $portNumber string The port number where the acquisition was done.
     * @param $linkId string The link identifier.
     * @param $testId The test identifier if it exists.
     * @param $otdrShortTraceName string otdr measure trace name without path
     * @param $otdrTraceType string SMTOtdrTraceType::OTU_REFERENCE_TRACE, SMTOtdrTraceType::SMARTOTU_ACQUISITION_TRACE, SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE
     * @param $forceCopy     Whether the copy overrides an existing trace (FALSE to update an existing test, TRUE by default for a new test)   
     * 
     * @return SMTOtdrReferenceFiles The reference traces paths copied in OTU reference detection and localisation directories
     */
    public static function copyMeasurementAsReferenceInOTU( $portNumber, $linkId, $testId, $otdrShortTraceName, $otdrTraceType, $forceCopy = TRUE )
    {
    	$smartAcqExtension = self::OTDR_FILE_SOR_EXTENSION;
    	$otdrShortTraceName= self::replaceExtension($otdrShortTraceName, $smartAcqExtension);
    	
        //test if the new reference must be copied from SmartOTU last acquisition directory
        if ( $otdrTraceType == SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE )
        {
        	//search on SmartOtu last acquisition directory
        	$acquisitionTrace = self::getLastMonitoringTraceNameFromSmartOtu($testId, $portNumber);
        }
        else if ( $otdrShortTraceName != NULL)
        {
            //search acquisition trace on session directory
            $acquisitionTrace = SMTSessionDirectoryUtil::getOTDRTraceFileNameFromSession( $otdrShortTraceName );
        }
        else
        {
            //search acquisition trace on session directory
            $acquisitionTrace = SMTSessionDirectoryUtil::getLastOTDRTraceFileNameFromSession( $portNumber );  
        }
        SMTLogger::getInstance()->trace("acquisition trace in session: ".$acquisitionTrace, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    
        $smartAcqExtension = self::getSmartAcqFileExtension($acquisitionTrace);
        $smartAcqAcquisitionTrace = self::replaceExtension($acquisitionTrace, $smartAcqExtension);
        $traceNameOnOTU = sprintf( self::REF_TRACE_NAME, $portNumber);
        $newReferenceDetTraceNameOnOTU = self::OTU_MEASURE_REF_DET_DIR.$traceNameOnOTU ;
        $smartAcqNewReferenceDetTraceNameOnOTU = self::replaceExtension($newReferenceDetTraceNameOnOTU, $smartAcqExtension);
        
		// sor
        if ( !$forceCopy && file_exists( $newReferenceDetTraceNameOnOTU ) )
        {
        	$traceWithoutDuplicateSuffix = self::removeExtension($newReferenceDetTraceNameOnOTU);
        	//si le fichier memorise existe deja, on l'ecrase
        	$newReferenceDetTraceNameOnOTU = $traceWithoutDuplicateSuffix.self::OTDR_FILE_MEMORIZED_SUFFIX.self::OTDR_FILE_SOR_EXTENSION;
        }
    
        SMTLogger::getInstance()->trace( sprintf("Start copying detection acquisition trace %s to %s: ", $acquisitionTrace, $newReferenceDetTraceNameOnOTU), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        if ( !copy( $acquisitionTrace, $newReferenceDetTraceNameOnOTU) )
        {
        	throw new SMTIOException( SMTIOException::COPY_FAILURE, $newReferenceDetTraceNameOnOTU);
        }
        
        $otdrReferenceFiles = new SMTOtdrReferenceFiles();
        $otdrReferenceFiles->setOtdrFileName( $traceNameOnOTU );
        $otdrReferenceFiles->setOtdrDetFilePath($newReferenceDetTraceNameOnOTU);
                
        //smartAcq
        if (self::isSmartAcqFileExtension($smartAcqExtension) )
        {        	        
	        //if smartAcq exists
	        if ( !$forceCopy && file_exists( $smartAcqNewReferenceDetTraceNameOnOTU ) )
	        {
	        	$traceWithoutDuplicateSuffix = self::removeExtension( $smartAcqNewReferenceDetTraceNameOnOTU);
	        	//si le fichier memorise existe deja, on l'ecrase
	        	$smartAcqNewReferenceDetTraceNameOnOTU = $traceWithoutDuplicateSuffix.self::OTDR_FILE_MEMORIZED_SUFFIX.$smartAcqExtension;
	        }
	        
	        SMTLogger::getInstance()->trace( sprintf("Start copying detection short acq acquisition trace %s to %s: ", $smartAcqAcquisitionTrace, $smartAcqNewReferenceDetTraceNameOnOTU), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
	        
	        if ( !copy( $smartAcqAcquisitionTrace, $smartAcqNewReferenceDetTraceNameOnOTU) )
	        {
	        	//first remove first file copied
	        	unlink( $newReferenceDetTraceNameOnOTU);
	        	throw new SMTIOException( SMTIOException::COPY_FAILURE, $smartAcqNewReferenceDetTraceNameOnOTU);
	        }
	        
	        $otdrReferenceFiles->setOtdrLocFilePath($smartAcqNewReferenceDetTraceNameOnOTU);
        }               
        
        return $otdrReferenceFiles;
    }
}
?>