<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otau;

/**
 * Decode otu switch configuration string.
 * Used when retrieving the switch config.
 * 
 * @author Sylvain Desplat
 */
class SMTSwitchDecoder
{
    const TOPOLOGY=1;
    const CUSTOMISATION=2;
    const IN_POSITION=3;
    const OUT_POSITION=5;
    const OUT_POSITION_V=6;
    const OSU_COUNT=9;
    
    /**
     * switch string to decode
     * @var string
     */
    private $switch_string = NULL;    

    /**
     * OSU count per topologies excluding topologies M,N,O,P,Q,V,X,Y where the number of OSU is given directly in the configuration string
     */
    private $osu_Topologies = array("A"=>0,"B"=>1,"C"=>2,"D"=>1,"E"=>0,"F"=>0,"G"=>0,"H"=>0,"I"=>1,"J"=>1,"K"=>0,"L"=>4,"R"=>1, "T"=>2,"U"=>1);  
    
    /**
     * Number of distributor ports at the end of the internal switch
     * @var integer
     */
    private $nbPortsDistributeurAtEnd = 0;
    
    /**
     * Construtor to set the given switch configuration string
	 * @param $switch_string The switch configuration string to decode
     */
	function __construct( $switchString ) 
    {
		$this->switch_string =$switchString;
	}
    
    /**
     * Returns the switch encoded string
     * 
     * @return the switch encoded string.
     */
    function getSwitchEncodingString()       
    {
        return $this->switch_string;    
    }
        
	/**
	 * Returns the number of entries of the switch
	 * 
	 * @return int
	 */ 
    function getIn()
    {
        $type  = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::TOPOLOGY, 1);
        if ( $type == 'V')
        {
            return 1;
        }
        else 
        {
            return ( strlen( $this->getSwitchEncodingString() ) > 0)? intval( substr($this->getSwitchEncodingString(), SMTSwitchDecoder::IN_POSITION, 2) ) : 0;  
        }
    }
    
	/**
	 * Returns the number of global outputs of the switch
	 * 
	 * @return number
	 */ 
    function getOut()
    {
    	$switchOutPut='0';
    	if ( strlen( $this->getSwitchEncodingString() ) > 0)
    	{
        	$type  = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::TOPOLOGY, 1);
        	
        	//for M,N,P,Q topologies the switch output count is given with 4 digits
        	if ( $type == 'M' || $type == 'N' || $type == 'O' || $type == 'P' || $type == 'Q' || $type == 'X'|| $type == 'Y')
        	{
        		$switchOutPut = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::OUT_POSITION, 4);
        	}
        	if ( $type == 'V' )
        	{
        	    $switchOutPut = intval( substr($this->getSwitchEncodingString(), SMTSwitchDecoder::IN_POSITION, 3) ) -$this->getPortDistributorNumber() + intval(substr($this->getSwitchEncodingString(), SMTSwitchDecoder::OUT_POSITION_V, 3) ) * $this->getOSUCount() ; 
        	}
        	//for other topologies, the switch output count is given with 3 digits
        	else
        	{
        		$switchOutPut = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::OUT_POSITION, 3);
        	}
    	}
        return intval( $switchOutPut );
    }     
    
    /**
     * 
     * @return number
     */
    function getOSUCount()
    {
    	$osuCount = 0;
    	if ( strlen( $this->getSwitchEncodingString() ) > 0)
    	{
        	$type  = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::TOPOLOGY, 1);
        	
        	//for M,N,O,P,Q topologies the count of osus is set with 2 digits in the switch string
        	if ( $type == 'M' || $type == 'N' || $type == 'O' || $type == 'P' || $type == 'Q' || $type == 'V')
        	{
        		$osuCount = intval( substr($this->getSwitchEncodingString(), SMTSwitchDecoder::OSU_COUNT, 2) );
        	}
        	//for X topologies the count of SKC osus is set with 2 digits in the switch string
        	//The total count of osus = SKB osus + SKC osus with SKB = (NNNN – (SKCx36))/48
        	else if ( $type == 'X')
        	{
        		$osuCountSKC = intval( substr($this->getSwitchEncodingString(), SMTSwitchDecoder::OSU_COUNT, 2) );
        		$osuCountSKB = ($this->getOut() - ( $osuCountSKC * 36 ) ) / 48;
        		$osuCount = $osuCountSKB + $osuCountSKC;
        	}   
        	//for Y topologies the count of SKC osus is set with 2 digits in the switch string
        	//The total count of osus = SKB osus + SKC osus with SKC = (NNNN – (SKCx36))/64
        	else if ( $type == 'Y')
        	{
        		$osuCountSKC = intval( substr($this->getSwitchEncodingString(), SMTSwitchDecoder::OSU_COUNT, 2) );
        		$osuCountSKB = ($this->getOut() - ( $osuCountSKC * 36 ) ) / 64;
        		$osuCount = $osuCountSKB + $osuCountSKC;
        	} 
        	//in other topologies, find the number of OSU by the topology
        	else
        	{	    		
        		$osuCount = $this->osu_Topologies[$type];
        	}
    	}

        return intval($osuCount);
    }  
        
    /**
     * 
     * @return string The distributor OSU name.
     */
    function getOSUDistributor()
    {
    	$osuDistributorPos = '';

    	$type  = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::TOPOLOGY, 1);
    	
    	//for P,Q topologies there is a distributor OSU
    	if ( $type == 'P' || $type == 'Q')
    	{
    		$osuDistributorPos='OSU31';
    	}

        return $osuDistributorPos;    
    }
    
    /**
     * Whether There is an OSU distributor.
     * 
     * @return boolean Whether There is an OSU distributor.
     */
    function hasOSUDistributor()
    {
    	$osuDistributor = FALSE;
    	
    	$type  = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::TOPOLOGY, 1);
    	
    	//for P,Q topologies there is a distributor OSU
    	if ( $type == 'P' || $type == 'Q')
    	{
    		$osuDistributor = TRUE;
    	}
    	
    	return $osuDistributor;
    }
    
    /**
     * Whether the topology uses OTAMs
     * 
     * @return boolean
     */
    function hasMolexOSU()
    {
    	$molex = FALSE;
    	$customization = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::CUSTOMISATION, 1);
    	if ( $customization == 'A' || $customization == 'B' || $customization == 'D' || $customization == 'F' || $customization == 'H')
    	{
    		$molex = TRUE;
    	}
    	return $molex;
    }
    
    /**
     * Whether the topology uses JDSU OSU
     *
     * @return boolean
     */
    function hasJdsuOSU()
    {
    	$jdsu = FALSE;
    	$customization = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::CUSTOMISATION, 1);
    	if ( $customization == '3' || $customization == '4' ||$customization == '6' || $customization == '7' || $customization == '8' || $customization == '9' || $customization == 'C'  || $customization == 'E' || $customization == 'G' )
    	{
    		$jdsu= TRUE;
    	}
    	return $jdsu;
    }
    
    /**
     * Number of ports at the end used for distribution.
     *
     * @return int  Number of ports at the end used for distribution.
     */
    function getPortDistributorNumber()
    {        
        return intval($this->nbPortsDistributeurAtEnd);
    }

    /**
     * Number of ports at the end used for distribution.
     *
     * @param int  Number of ports at the end used for distribution.
     */
    function setPortDistributorNumber( $nbPortsDistributeurAtEnd )
    {
        $this->nbPortsDistributeurAtEnd = $nbPortsDistributeurAtEnd;
    }
    
    /**
     * Whether the topology uses ports distributor at the end
     *
     * @return boolean
     */
    function hasPortsDistributorAtEnd()
    {
        $type  = substr($this->getSwitchEncodingString(), SMTSwitchDecoder::TOPOLOGY, 1);
        return ( $type == 'V' );
    }
}
?>