<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otau;

/**
 * Switch buffer encoder/decoder used for switch buffer in OTDR acquisitions, tests...
 * 
 * @author Sylvain Desplat
 */
class SMTSwitchBufferEncoderDecoder
{
    //default settings for SmartOTU
    const DEFAULT_SWITCH_INPUT = 1;
    const DEFAULT_SWITCH_COUNT = 1;
    
    const NB_DIGIT = 8;
    const NB_OUT_DIGIT = 4;
    /**
     * Macro switch id 0
     */
    const MACRO_SWITCH_ID_0 = 0;
    /**
     * @var string
     */
    private $switchBuffer="";
    /** 
     * @var array
     */
    private $arrayIn = array();
    /**
     * @var array
     */
    private $arrayOut = array();
    
    /**
     * Decode the given switch buffer string
     * 
     * @param string $switchBuffer
     */
    function __construct( $switchBuffer )
    {
        $this->switchBuffer = $switchBuffer;
    	$this->decodeSwitches();
    }
    
    /**
     * Build a switch buffer string from the given in ,out switches of each macro_switch, and from the array with the selected switches
     *  
     * @param array $arrayIn
     * @param array $arrayOut
     * @param array $array_otu_used
     * @return string
     */
    static function encodeSwitches($arrayIn, $arrayOut, $array_otu_used)
    {    
    	// OTAU buffer string
    	$buffer_tmp = '';
    
    	foreach ($arrayIn as $macroSwitchId => $otau_in)
    	{
    		//if OTAU is used in that configuration, add it to the encoded buffer
    		//
    		if($array_otu_used[$macroSwitchId] == 1)
    		{    
    			$otau_out = $arrayOut[$macroSwitchId];        		
        		$buffer_tmp .= self::encode($otau_in, $otau_out, $macroSwitchId);
    		}
    	}
        
		$buf_len = strlen($buffer_tmp);
		$info_len = strlen($buf_len);
		$switchBuffer = "#".$info_len.$buf_len.$buffer_tmp;    
		
		return $switchBuffer;
    }
    
    /**
     * Build a switch buffer string from the given in ,out switches of the macroswitch of id 0.
     * Used for SmartOTU
     *
     * @param number $otauIn
     * @param number $otauOut
     * 
     * @return string
     */
    static function encodeSwitch( $otauIn, $otauOut )
    {
		$buffer_tmp = self::encode($otauIn, $otauOut, self::MACRO_SWITCH_ID_0);

		$buf_len = strlen($buffer_tmp);
		$info_len = strlen($buf_len);
		$switchBuffer = "#".$info_len.$buf_len.$buffer_tmp;

		return $switchBuffer;
    }    
    
    /**
     * Build a switch buffer string from the given in ,out switches of each macro_switch
     * 
     * @param number $otau_in
     * @param number $otau_out
     * @param number $macroSwitchId (macroswitch id 0 by default)
     * @return string
     */
    private static function encode($otau_in, $otau_out, $macroSwitchId = self::MACRO_SWITCH_ID_0)
    {    
		$addr_otau_hex_tmp = strtoupper(base_convert($macroSwitchId, 10,16));
		$addr_otau_hex = substr('00', 0, (2 - strlen($addr_otau_hex_tmp))).$addr_otau_hex_tmp;

		$otau_out_hex_tmp = strtoupper(base_convert($otau_out , 10,16));
		$otau_out_hex = substr('000', 0, (self::NB_OUT_DIGIT - strlen($otau_out_hex_tmp))).$otau_out_hex_tmp;
		$otau_in_hex_tmp = strtoupper(base_convert($otau_in , 10,16));
		$otau_in_hex = substr('00', 0, (2 - strlen($otau_in_hex_tmp))).$otau_in_hex_tmp;
    
   		return $addr_otau_hex.$otau_in_hex.$otau_out_hex;
    }    
    
    /**
     * Retrieve in and out switches for the given switch buffer string
     * 
     */
    private function decodeSwitches()
    {            
        $info_len = substr($this->switchBuffer, 1, 1);
        //if there is a switch
        if ( $info_len > 0 )
        {
            $buf_len = substr( $this->switchBuffer, 2, $info_len);
            $nbBlocks = round($buf_len / self::NB_DIGIT);
            $buffer_tmp = substr($this->switchBuffer, -($buf_len), $buf_len);    
        
            for ($index=0; $index<$nbBlocks; $index++)
            {
                $otau_in_hex = substr($buffer_tmp,($index*self::NB_DIGIT)+2, 2);
                $otau_out_hex = substr($buffer_tmp,($index*self::NB_DIGIT)+4, self::NB_OUT_DIGIT);
                $addr_otau_hex = substr($buffer_tmp,$index*self::NB_DIGIT, 2);
            
                $addr_otau = base_convert($addr_otau_hex, 16, 10);
                $otau_in = base_convert($otau_in_hex, 16, 10);
                $otau_out = base_convert($otau_out_hex, 16, 10);
                $this->arrayIn[$addr_otau] = $otau_in;
                $this->arrayOut[$addr_otau] = $otau_out;
            }
        }
    }
    
    function getSwitchBuffer()
    {
        return $this->switchBuffer;
    }

    /**
     * The switches input of all macroswitchs
     * 
     * @return array
     */
    function getSwitchesIn()
    {
        return $this->arrayIn;
    }

    /**
     * The switches output of all macroswitchs
     *
     * @return array
     */    
    function getSwitchesOut()
    {
        return $this->arrayOut;
    }
    
    /**
     * The switch input index of macroswitch id 0 or NULL if no switch is available
     *
     * @return number
     */    
    function getSwitchIn()
    {
        //if there is no switch, switch-in is NULL
    	return ( count( $this->arrayIn ) > 0 )? $this->arrayIn[self::MACRO_SWITCH_ID_0] : NULL;
    }
    /**
     * The switch output count of macroswitch id 0 or NULL if no switch is available
     * 
     * @return number
     */
    function getSwitchOut()
    {
        //if there is no switch, switch-output count is NULL
    	return ( count( $this->arrayOut ) > 0 )? $this->arrayOut[self::MACRO_SWITCH_ID_0] : NULL;
    }    
}


?>