<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\setup;

use app\serviceshelper\maintenance\SMTLicence;

use app\util\SMTLogger;

use app\serviceshelper\ethernet\SMTFirewall;

use app\serviceshelper\ethernet\SMTEthernet;

use app\services\SMTIService;
use app\services\SMTService;

use app\parser\SMTOtuApi;
use app\admin\SMTOtuType;
use app\admin\SMTOtuMode;
use app\util\SMTUtil;

/**
 * Service to setup the OTU features (ethernet, modules, switches, Email, SNMP) and display its configuration.
 * 
 * @author Sylvain Desplat
 */
class SMTEthernetService extends SMTService implements SMTIService
{        
    /**
     * 
     * @url GET /localmode
     * 
	 * @access protected 
	 * @class SMTAccessControl {@requires user}
     */
    function isLocalMode()
    {
        $localMode = FALSE; 
        
        try
        {
            $localMode = SMTEthernet::isLocalMode( $this->getContext()->getOtuSocket() );
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        
    	return $localMode;
    }
            
    /**
     *
     * @url POST /hostname/{hostname}
     * 
	 * @access protected 
	 * @class SMTAccessControl {@requires install}
     */
    function setHostName( $hostname )
    {        
        try 
        {
            SMTEthernet::checkHostName( $hostname );

            //update OTU host name:
            $this->send( sprintf( SMTOtuApi::CMD_set_host_name, $hostname ) );
            
            //update also OTU name:
            $this->send( sprintf( SMTOtuApi::CMD_set_otu_name, $hostname ) );
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }        
        
        return $hostname;
    }
    
    /**
     * 
     * @url GET /hostname
     * 
	 * @access protected 
	 * @class SMTAccessControl {@requires user}
     */
    function getHostName()
    {
        $hostName = '';
        
        try
        {
        	$hostName = $this->sendReceive( SMTOtuApi::CMD_get_host_name );
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }
        
        return $hostName;        
    }
    
    /**
     *
     * @url GET /name
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getName()
    {
    	$name = '';
    	
    	try
    	{
    		$name= $this->sendReceive( SMTOtuApi::CMD_get_otu_name);
    		if ( !SMTOtuMode::isSmartOtuFamily() )
    		{
    		    $name = SMTUtil::convertToAscii($name );
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return $name;
    }
    
    /**
     *
     * @url GET /ipv4
     * 
	 * @access protected 
	 * @class SMTAccessControl {@requires user}
     */
    function getIpv4Config()
    {
    	try
    	{
    	    $ethernet = new SMTEthernet();
    	    $ethernet->setContext( $this->getContext() );
    	    $ipv4 = $ethernet->getIpv4Config();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $ipv4->getJsonData();
    }    
    
    /**
     *
     * @url GET /ipv6
     * 
	 * @access protected 
	 * @class SMTAccessControl {@requires user}
     */
    function getIpv6Config()
    {
    	try
    	{
    	    $ethernet = new SMTEthernet();
    	    $ethernet->setContext( $this->getContext() );
    	    $ipv6 = $ethernet->getIpv6Config();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $ipv6->getJsonData();
    }      
    
    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'  
     * 
     * @url POST /ipv4
	 * @access protected 
	 * @class SMTAccessControl {@requires install}
     */
    function setIpv4Config( $request_data ) 
    {    	    
    	try
    	{
    	    $ipv4 = SMTIpv4Dto::getInstance( $request_data );
    	    
    	    $ethernet = new SMTEthernet();
    	    $ethernet->setContext( $this->getContext() );
    	    $ethernet->setIpv4Config($ipv4); 
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $ipv4->getJsonData();
    }    
    
    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'
     *
     * @url POST /ipv6
	 * @access protected 
	 * @class SMTAccessControl {@requires install}
     */
    function setIpv6Config( $request_data )
    {
    	try
    	{
    	    $ipv6 = SMTIpv4Dto::getInstance( $request_data );
    	    
    	    $ethernet = new SMTEthernet();
    	    $ethernet->setContext( $this->getContext() );
    	    $ethernet->setIpv6Config($ipv6); 
    
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $ipv6->getJsonData();
    }
    
    /**
     *
     * @url GET /
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getEthernetConfig()
    {
        $ethernet = new SMTEthernet();
        $ethernet->setContext( $this->getContext() );
        
        try
        {	            
            $firewallMode = $this->getFirewallMode();
        }
        catch( \Exception $e )
        {
            $firewallMode = SMTFirewall::FIREWALL_OFF;
            $this->getContext()->traceException($e);
        }
        
        try
        {
        	$enhancedSecurity = $this->getEnhancedSecurity();
        }
        catch( \Exception $e )
        {
        	$enhancedSecurity = SMTLicence::ENHANCED_SECURITY_OFF;
        	$this->getContext()->traceException($e);
        }
        
    	try
    	{
    	    $hostName = $this->getHostName();
    	    $ipv4 = $ethernet->getIpv4Config();   
    	    $ipv6 = $ethernet->getIpv6Config();
    	    $ethernetConfig = new SMTEthernetDto();
    	    $ethernetConfig->setFirewallMode($firewallMode);
    	    $ethernetConfig->setEnhancedSecurity($enhancedSecurity);
    	    $ethernetConfig->setHostName($hostName);
    	    $ethernetConfig->setIpv4($ipv4);
    	    $ethernetConfig->setIpv6($ipv6);
    	    if ( SMTOtuType::isOtu8KV2() && SMTOtuMode::isONMSi() )
    	    {
    	    	try
    	    	{
    	    		$ethernet->getBackupRoute($ethernetConfig);
    	    	}
    	    	catch(\Exception $e)
    	    	{
    	    		$this->getContext()->traceException($e);
    	    	}
    	    }
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $ethernetConfig->getJsonData();
    }

    /**
     * Full json request data are by default stored in an array in parameter Default::$fullRequestDataName='request_Data'
     *
     * @url POST /
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setEthernetConfig( $request_data )
    {
        $ethernet = new SMTEthernet();
        $ethernet->setContext( $this->getContext() );
        
        //if firewall config update fails, continue to set the ip config
    	try
    	{    	   
    	    $ethernetConfig = SMTEthernetDto::getInstance( $request_data );
    	    
    	    //pas de changement du firewall en mode ONMSi
    	    //if ( !SMTOtuMode::isONMSi() )
    	    	//Dans le cas d'une config de firewall custom
    	    if ( strcasecmp( $ethernetConfig->getFirewallMode(), SMTFirewall::FIREWALL_CUSTOM) == 0 )
    	    {
    	        $customFirewallConfig = $ethernetConfig->getCustomFirewall();
    	        //if firewall config is set
    	        if ( isset ($customFirewallConfig) && $customFirewallConfig != NULL )
    	        {
    	            $firewall = new SMTFirewall();
    	            $firewall->setContext( $this->getContext() );
    	            //si la config du firewall n'est pas vide
    	            if ( count ($customFirewallConfig->getCustomFirewall() ) > 0 )
    	            {
    	                $firewall->saveConfig($customFirewallConfig->getCustomFirewall());
    	                //if config save fails, don't change to custom firewall (only change when everything got right)
    	                $this->setFirewallMode( SMTFirewall::FIREWALL_CUSTOM );
    	            }
    	            else
    	            {
    	                $this->getContext()->getLogger()->trace("customFirewallConfig empty", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__ );
    	            }
    	        }
    	        else
    	        {
    	            //check that custom firewall config file exists, before switching to custom firewall
    	            $firewall = new SMTFirewall();
    	            $firewall->setContext( $this->getContext() );
    	            $firewall->readCustomFirewallConfig();
    	            $this->setFirewallMode( SMTFirewall::FIREWALL_CUSTOM );
    	        }
    	    }
    	    else
    	    {
    	        $this->setFirewallMode( $ethernetConfig->getFirewallMode() );
    	    }
	    }
	    catch ( \Exception $e )
	    {
	        $this->getContext()->traceException($e);
	    }
	    //backup route uniquement en mode ONMSi et sur OTU8kv2
	    if ( SMTOtuType::isOtu8KV2() && SMTOtuMode::isONMSi() )
	    {
	    	try
	    	{
	    		$ethernet->setBackupRoute($ethernetConfig->getBackupRouteActive());
	    	}
	    	catch(\Exception $e)
	    	{
	    		$this->getContext()->traceException($e);
	    	}
	    }

	    try 
	    {	    
    	    //Only change ip settings if they have changed
    	    if ( $ethernetConfig->isReboot() )
    	    {
    	        $this->setHostName( $ethernetConfig->getHostName() );
    	        $ethernet->setIpv4Config( $ethernetConfig->getIpv4() );
    	        $ethernet->setIpv6Config( $ethernetConfig->getIpv6() );    	   

        	    //Reboot the OTU to apply the new configuration
        	    $this->send( sprintf( SMTOtuApi::CMD_system_reboot ) );  

    	        $this->getContext()->setOTUReboot();
    	    }
    	    else
    	    {
    	        try
    	        {
    	        	$enhancedSecurity = $this->getEnhancedSecurity();
    	        }
    	        catch( \Exception $e )
    	        {
    	        	$enhancedSecurity = SMTLicence::ENHANCED_SECURITY_OFF;
    	        	$this->getContext()->traceException($e);
    	        }
    	        
        	    $firewallMode = $this->getFirewallMode();        	    
        	    $hostName = $this->getHostName();
        	    $ipv4 = $ethernet->getIpv4Config();   
        	    $ipv6 = $ethernet->getIpv6Config();
        	    $ethernetConfig->setHostName($hostName);
        	    $ethernetConfig->setFirewallMode($firewallMode);
        	    $ethernetConfig->setEnhancedSecurity($enhancedSecurity);
        	    $ethernetConfig->setIpv4($ipv4);
        	    $ethernetConfig->setIpv6($ipv6);
        	    if ( SMTOtuType::isOtu8KV2() && SMTOtuMode::isONMSi() )
        	    {
        	    	try
        	    	{
        	    		$ethernet->getBackupRoute($ethernetConfig);
        	    	}
        	    	catch(\Exception $e)
        	    	{
        	    		$this->getContext()->traceException($e);
        	    	}
        	    }
    	    }
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $ethernetConfig->getJsonData();
    }    
    
    /**
     * Change firewall status
     *
     * @url POST /firewall/mode/{firewallStatus}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setFirewallMode( $firewallStatus )
    {
    	try
    	{
    	    $firewall = new SMTFirewall();
    	    $firewall->setContext( $this->getContext() );
    	    $firewall->setFirewallMode( $firewallStatus );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return self::RESULT_OK;
    }
    
    /**
     * return the firewall status
     *
     * @url GET /firewall/mode
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getFirewallMode()
    {
        $firewallMode = SMTFirewall::FIREWALL_OFF;
    	try
    	{
    	    $firewall = new SMTFirewall();
    	    $firewall->setContext( $this->getContext() );
    	    $firewallMode = $firewall->getFirewallMode();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $firewallMode;
    }
    
    /**
     *
     * @url POST /firewall/config
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setCustomFirewallConfig( $request_data )
    {
    	try
    	{
    	    $customFirewallConfig = SMTCustomFirewallDto::getInstance( $request_data );
    	    $firewall = new SMTFirewall();
    	    $firewall->setContext( $this->getContext() );
    	    $firewall->saveConfig($customFirewallConfig->getCustomFirewall());
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return self::RESULT_OK;
    }    
    
    /**
     * return the firewall status
     *
     * @url GET /firewall/config
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getCustomFirewallConfig()
    {
    	try
    	{
    		$firewall = new SMTFirewall();
    		$firewall->setContext( $this->getContext() );
    		$lines = $firewall->readCustomFirewallConfig();
    		$customFirewallConfig = new SMTCustomFirewallDto();
    		$customFirewallConfig->setCustomFirewall($lines);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $customFirewallConfig->getJsonData();
    }
    
    /**
     * Reset custom firewall content file with default content
     *
     * @url POST /firewall/resetcustom
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function resetCustomFileContent()
    {
    	try
    	{
    		$firewall = new SMTFirewall();
    		$firewall->setContext( $this->getContext() );
    		$firewall->resetCustomFileContent();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	return self::RESULT_OK;
    }
    
    /**
     * return whether the enhanced security mode is active
     *
     * @url GET /enhancedsecurity
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getEnhancedSecurity()
    {
        $highSecurityMode = SMTLicence::ENHANCED_SECURITY_OFF;
    	try
    	{
    	    $context = $this->getContext();
    		$highSecurityMode = SMTLicence::isHighSecurityMode($context)? SMTLicence::ENHANCED_SECURITY_ON : SMTLicence::ENHANCED_SECURITY_OFF;
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $highSecurityMode;
    }
    
    /**
     * @url GET /test/updateipv4
     */
    function testIpv4()
    {        
//         $ipv4 = new SMTIpv4Dto();
//         $ipv4->setMode( "DHCP" );
//         $ipv4->setIp( '10.33.16.230' );
//         $ipv4->setSubnet( '255.255.252.0' );
//         $ipv4->setGateway( '10.33.19.254' );
//         $ipv4->setDns( '10.49.2.132' );
//         $ipv4->setDomain( 'ds.jdsu.net' );
        
//         $url = '/ethernet/ipv4';
//         $json_string = json_encode( $ipv4->getJsonData() );
//         $page = new SMTTestRestApiPage($url, $json_string,'POST');
//         $this->getContext()->getHttpResponse()->send( $page );        
    }
    
    /**
     * @url GET /test/ethernetconfig
     */
    function testEthernetConfig()
    {    
//     	$url = '/ethernet/';
//     	$json_string = json_encode( $this->getEthernetConfig() );
//     	$page = new SMTTestRestApiPage($url, $json_string,'POST');
//     	$this->getContext()->getHttpResponse()->send( $page );
    }
    
    /**
     * @url GET /test/updatehostname
     */
    function testUpdateHostname()
    {
//         $url = '/ethernet/hostname/newHostName';
//         $page = new SMTTestRestApiPage($url, "", 'POST');
//         $this->getContext()->getHttpResponse()->send( $page );
    }
    
}
?>