#!/bin/bash
#set -u
#set -x

unset LANG
unset LANGUAGE
unset LC_ALL
unset LC_CTYPE
unset LC_NUMERIC

DEV_SDSCARD="/dev/mmcblk0"
DEV_USER="${DEV_SDSCARD}p3"
DEV_SYST="${DEV_SDSCARD}p2"
DEV_LOG="${DEV_SDSCARD}p1"


PGNAME=`basename $0`

#the following size are in MB
HD_SYST_SIZE=600M
HD_LOG_SIZE=200M

HD_DEV_USER_LABEL="JDHDUSER"
HD_DEV_SYST_LABEL="JDHDSYST"
HD_DEV_LOG_LABEL="JDHDLOG"
DISK_DEV_USER_LABEL="FLASHDISK"

DISK_DEV_USER="/dev/mmcblk1p3"

####################################################################
function Umount_Device()
   {
   DEV_PART=$1

   grep "$DEV_PART" /etc/mtab > /dev/null
   [ "$?" != "0" ] && return 0

   umount $DEV_PART 2>&1 | grep -q "device is busy"
   [ $? == 0 ] && return 1
   
   [ ! -e $DEV_PART ] && return 0
   
   umount $DEV_PART 2>&1 | grep -q "not mounted"
   [ $? == 0 ] && return 0
   
   return 1
   }

###########################################################
function Free_Devices()
   {
   ATTEMPT_TO_FREE=10

   fuser -SIGTERM -mk $*

   while fuser -m $*
      do
      ATTEMPT_TO_FREE=$(( ATTEMPT_TO_FREE - 1 ))
      if [ $ATTEMPT_TO_FREE -lt 0 ]
         then
         fuser -SIGKILL -mk $*
         sleep 2
         break
         fi
      usleep 200
      done
   echo "Before umount"
   fuser -m $*
   for DEV in $*
      do
      Umount_Device $DEV || return 1
      done
   }

####################################################################
function LOG()
   {
   echo $*
   logger -t $PGNAME $*
   }
####################################################################
function Format_DISK()
   {
    if ! Umount_Device $DISK_DEV_USER
      then
      LOG "$DISK_DEV_USER is in use"
      return 1
	fi   
   
   Format_Partition $DISK_DEV_USER $DISK_DEV_USER_LABEL   
   
   if [ $? != 0 ]
      then
      LOG "Failure while formatting $DISK_DEV_USER"
      return 1
   else
      return 0
      fi
   }
####################################################################
function Check_DISK()
   {
   DISK_PART="$(cat /etc/fstab | grep /user/disk | cut -f1 -d ' ')"
   if ! [ -b $DISK_PART ]    
      then
      LOG "Unable to find partition for /user/disk"
      return 1
      fi
   
   if [ "$(e2label $DISK_PART)" != "User" ]
      then
      LOG "Bad label for /user/disk"
      return 1
      fi

   return 0
   }
####################################################################
function Not_Used()
   {
   echo | sfdisk $DEV_SDSCARD 2>&1 | grep BLKRRPART
   if [ $? == 0 ]
      then
      LOG $DEV_SDSCARD" is in use"
      return 1
      fi
   
   LOG $DEV_SDSCARD" is not used"
   return 0
   }
####################################################################
function Make_HD_Not_Busy()
   {
   Free_Devices $DEV_SYST $DEV_USER $DEV_LOG

   if ! Umount_Device $DEV_SYST
      then
      LOG "$DEV_SYST is in use"
      return 1
      fi

   if ! Umount_Device $DEV_USER
      then
      LOG "$DEV_USER is in use"
      return 1
      fi

   if ! Umount_Device $DEV_LOG
      then
      LOG "$DEV_LOG is in use"
      return 1
      fi
      
   return 0
   }
####################################################################
function Make_Part()
   { 
   if [ ! -b $DEV_SDSCARD ]
      then
      LOG "No DEVICE "$DEV_SDSCARD
      return 1
      fi
  
   dd if=/dev/zero of=$DEV_SDSCARD bs=100k count=1
   sync
   sfdisk $DEV_SDSCARD << EOF
EOF
   sync
   ATTEMPT=5
   while [ -b $DEV_LOG  -o -b $DEV_SYST -o -b $DEV_USER ]
      do      
      sleep 1
      ATTEMPT=`expr $ATTEMPT - 1`
      if [ $ATTEMPT == 0 ]
         then
         LOG "Timeout while removing HD_DEVICEs"
         return 1
         fi
      done
   
   sfdisk $DEV_SDSCARD << EOF
,$HD_SYST_SIZE
,$HD_LOG_SIZE
,,0
EOF
   
   if [ $? != 0 ]
      then
      LOG "Cannot create partitions table"
      return 1
      fi
   
   ATTEMPT=30
   while [ 1 ]
      do
      [ -b $DEV_USER ] && [ -b $DEV_SYST ] && [ -b $DEV_LOG ] && break
      sleep 1
      ATTEMPT=`expr $ATTEMPT - 1`
      if [ $ATTEMPT == 0 ]
         then
         LOG "Timeout while waiting for DEVICEs"
         return 1
         fi
      done

   return 0
   }
####################################################################
function Format_Partition()
   {
   DEVICE_TO_FORMAT=$1   
   DEVICE_LABEL=$2
   if ! Umount_Device $DEVICE_TO_FORMAT
      then
      LOG "$DEVICE_TO_FORMAT is in use"
      return 1
      fi
   
   mkfs.ext4 -F $DEVICE_TO_FORMAT -L ${DEVICE_LABEL}
   if [ $? != 0 ]
      then
      LOG "Failure while formatting $DEVICE_TO_FORMAT"
      return 1
      fi
   
   tune2fs -i 0 -c 0 $DEVICE_TO_FORMAT   
	
   return 0
   }

   
####################################################################
function CHECKHD()
   {
   
   if [ ! -b $DEV_SDSCARD ]
      then
      LOG "No "$DEV_SDSCARD
      return 1
      fi
   
   if [ ! -b $DEV_USER ]
      then
      LOG "No "$DEV_USER
      return 1
      fi
   
   if [ ! -b $DEV_SYST ]
      then
      LOG "No "$DEV_SYST
      return 1
      fi

   if [ ! -b $DEV_LOG ]
      then
      LOG "No "$DEV_LOG
      return 1
      fi
   
#   if [ "`CFG_Get_Dosfslabel $HD_DEV_USER`" != $HD_DEV_USER_LABEL ]
   if [ "`e2label $DEV_USER`" != $HD_DEV_USER_LABEL ]
      then
      LOG "Bad label for "$DEV_USER
      return 1
      fi
   
   if [ "`e2label $DEV_SYST`" != $HD_DEV_SYST_LABEL ]
      then
      LOG "Bad label for $DEV_SYST"
      return 1
      fi

   if [ "`e2label $DEV_LOG`" != $HD_DEV_LOG_LABEL ]
      then
      LOG "Bad label for $DEV_LOG"
      return 1
      fi
   
   LOG "Ok"
   return 0
   }
####################################################################
function Force_Udev_Full_Restart()
    {
    rm -f /etc/udev/dev.tar
    }
####################################################################

ACTION=$1

if [ "$ACTION" == "FORMATDISK" ]
   then
   Format_DISK || exit 1   
   exit 0
elif [ "$ACTION" == "CHECKDISK" ]
   then
   Check_DISK || exit 1   
   exit 0
elif [ "$ACTION" == "FORMATHD" ]
   then
   Force_Udev_Full_Restart
   Make_HD_Not_Busy || exit 1
   Not_Used    || exit 1
   Make_Part   || exit 1
   Format_Partition $DEV_USER $HD_DEV_USER_LABEL || exit 1
   Format_Partition $DEV_SYST $HD_DEV_SYST_LABEL || exit 1
   Format_Partition $DEV_LOG  $HD_DEV_LOG_LABEL || exit 1
   udevadm trigger --name=$(basename ${DEV_SDSCARD})
   exit 0
elif [ "$ACTION" == "FREEHD" ]
   then
   Make_HD_Not_Busy || exit 1
   Not_Used    || exit 1
   exit 0
elif [ "$ACTION" == "FORMATSYST" ]
   then
   Force_Udev_Full_Restart
   Format_Partition $DEV_SYST $HD_DEV_SYST_LABEL || exit 1
   exit 0
elif [ "$ACTION" == "FORMATUSER" ]
   then
   Force_Udev_Full_Restart
   Format_Partition $DEV_USER $HD_DEV_USER_LABEL || exit 1
   exit 0
elif [ "$ACTION" == "FORMATLOG" ]
   then
   Force_Udev_Full_Restart   
   Format_Partition $DEV_LOG $HD_DEV_LOG_LABEL || exit 1
   exit $?
elif [ "$ACTION" == "CHECKHD" ]
   then
   CHECKHD
   exit $?
else
   LOG "Usage: "$0" FORMATDISK | CHECKDISK | FORMATHD | FREEHD | CHECKHD | FORMATLOG | FORMATSYST | FORMATUSER"
   exit 1
   fi

exit 1
