

import gzip
import struct
from json import dumps


class CommonSpec:
    @staticmethod
    def encode(data: str) -> str:
        data = data.encode('utf8')
        raw = gzip.compress(data)

        return "".join(["%02X" % ch for ch in raw])

    @staticmethod
    def make_header(fmt, *args) -> str:
        raw_header = [bytes([arg]) for arg in args]
        header = struct.pack(fmt, *raw_header)
        return "".join(["%02X" % ch for ch in header])


class OltsSpec:
    @staticmethod
    def format():
        return 0

    @staticmethod
    def abridgments():
        return (
            ("GlobalVerdict", "GlobalVD"),
            ("Hardware", "HD"),
            ("UnitA", "UA"),
            ("UnitB", "UB"),
            ("LastCalibrationDate", "LCAD"),
            ("ModelName", "MD"),
            ("SerialNumber", "SN"),
            ("JSONVersion", "JV"),
            ("LastChangeDate", "LCD"),
            ("Measurement", "MEA"),
            ("Identifiers", "IDS"),
            ("Length", "LEN"),
            ("Results", "RES"),
            ("LossAB", "LAB"),
            ("LossBA", "LBA"),
            ("Wavelength", "WA"),
            ("TestDate", "TD"),
            ("ReferenceDate", "REFDate"),
            ("Unknown", "UNK")  # << values
        )

    @staticmethod
    def acronyms():
        return (
            ('""', '*'),
            ("''", '*'),
            ('},{', '#'),
            (':{', '~'),
            ('"},"', '$'),
            ("'},'", '$'),
            ('":"', '|'),
            ("':'", '|'),
            ('","', '?'),
            ("','", '?'),
            (':[{', '!'),
            ('}]}', '/')
        )

    @staticmethod
    def schema():
        return {
            "GlobalVerdict": "$.tests.0.results.status",
            "Hardware": {
                "UnitA": {
                    "LastCalibrationDate": "$.assetInfo.calibrationDate",
                    "ModelName": "$.assetInfo.model",
                    "SerialNumber": "$.assetInfo.uniqueId"
                },
                "UnitB": {
                    "LastCalibrationDate": "",  # TODO
                    "ModelName": "",  # TODO
                    "SerialNumber": ""  # TODO
                }
            },
            "Identification": {
                "JobId": "$.workflow.workOrderId"
            },
            "JSONVersion": "1.0",
            "LastChangeDate": "2020-10-20",
            "TestDate": "$.tests.0.results.testTime",
            "ReferenceDate": "$.tests.0.results.data.measuredResults.0.referenceTime",
            "Measurement": {
                "Identifiers": "",
                "Length": "$.tests.{}.results.data.fiberLengthM",
                "Results": [
                    {
                        "LossAB": "$.tests.{}.results.data.measuredResults.{}.linkLossAtoBdB",
                        "LossBA": "$.tests.{}.results.data.measuredResults.{}.linkLossBtoAdB",
                        "OrlA": "$.tests.{}.results.data.measuredResults.{}.linkOrlAdB",
                        "OrlB": "$.tests.{}.results.data.measuredResults.{}.linkOrlBdB",
                        "Wavelength": "$.tests.{}.results.data.measuredResults.{}.wavelengthNm"
                    }
                ]
            }
    }


class OtdrSpec:
    @staticmethod
    def format():
        return 1

    @staticmethod
    def abridgments():
        return (
            ("GlobalVerdict", "GlobalVD"),
            ("Hardware", "HD"),
            ("UnitA", "UA"),
            ("UnitB", "UB"),
            ("LastCalibrationDate", "LCAD"),
            ("ModelName", "MD"),
            ("SerialNumber", "SN"),
            ("JSONVersion", "JV"),
            ("LastChangeDate", "LCD"),
            ("Measurement", "MEA"),
            ("Identifiers", "IDS"),
            ("Level", '"LEV"'),
            ("Value", "VAL"),
            ("LocationDirection", "LDIR"),
            ("LinkResults", "LinkRES"),
            ("CompletionStatus", "CompletionST"),
            ("Length", "LEN"),
            ("Results", "RES"),
            ("Loss", '"L"'),
            ("Wavelength", "WA"),
            ("Elements", "ELM"),
            ("Position", "PS"),
            ("PreviousFiberSection", "PRFS"),
            ("Attenuation", "ATT"),
            ("Reflectance", "REFL"),
            ("Status", "ST"),
            ("Type", "TY"),
            ("TestDate", "TD"),
            ("Connector", "CNT"),  # << values
            ("Splice", "SP"),
            ("Macrobend", "MCD"),
            ("Group", "GRP")
        )

    @staticmethod
    def acronyms():
        return (
            ('""', '*'),
            ("''", '*'),
            ('},{', '#'),
            (':{', '~'),
            ('"},"', '$'),
            ("'},'", '$'),
            (':[{', '!'),
            ('":"', '|'),
            ("':'", '|'),
            ('","', '?'),
            ("','", '?'),
            ('}]}', '/')
        )

    @staticmethod
    def schema():
        return {
            "Hardware": {
                "UnitA": {
                    "LastCalibrationDate": "$.assetInfo.calibrationDate",
                    "ModelName": "$.assetInfo.model",
                    "SerialNumber": "$.assetInfo.uniqueId"
                }
            },
            "Identification": {
                "JobId": "$.workflow.workOrderId"
            },
            "Identifiers": [
                {
                    "Level": "NaN",
                    "Name": "Cable_ID",
                    "Value": "$.tests.0.configuration.otdrSettings.fiber.cableId"
                },
                {
                    "Level": "$.tests.0.configuration.otdrSettings.fiber.number",
                    "Name": "Fiber_ID",
                    "Value": "$.tests.0.configuration.otdrSettings.fiber.fiberId"
                }
            ],
            "JSONVersion": "1.0",
            "LastChangeDate": "",
            "LocationDirection": "$.tests.0.configuration.otdrSettings.fiber.direction",
            "TestDate": "$.tests.0.results.testTime",
            "LinkResults": {
                "CompletionStatus": "$.tests.0.results.status",
                "Length": "$.tests.0.results.data.otdrResults.measuredResults.0.fiberLength",  # TODO Laser/Units ?
                "Results": [
                    {
                        "Loss": "$.tests.{}.results.data.otdrResults.measuredResults.{}.linkLossdB",
                        "Orl": "$.tests.{}.results.data.otdrResults.measuredResults.{}.linkOrldB",
                        "Wavelength": "$.tests.{}.results.data.otdrResults.measuredResults.{}.wavelengthNm"
                    }
                ]
            },
            "Measurement": {
                "Elements": [
                    {
                        "Position": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.distance",
                        "Status": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.eventTestStatus",
                        "Type": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.eventType",
                        "Results": [
                            {
                                "Loss": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.lossdB",
                                "Reflectance": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.reflectancedB",
                                "Wavelength": "$.tests.{}.results.data.otdrResults.measuredResults.{}.wavelengthNm"
                            }
                        ],
                        "PreviousFiberSection" : {
                            "Length": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.sectionLengthM",
                            "Results": [
                                {
                                    "Attenuation": "NaN",
                                    "Loss": "$.tests.{}.results.data.otdrResults.measuredResults.{}.events.{}.slopedB",
                                    "Wavelength": "$.tests.{}.results.data.otdrResults.measuredResults.{}.wavelengthNm"
                                }
                            ]
                        }
                    }
                ]
            }
        }

