"""Module containing functions to query the instrument for information
specific to the unit
"""
from datetime import datetime
from rest_api.products.usc import dbus
from rest_api.products.mts_services.instrument_upgrade import getDeployTimeValue

def make_asset_info():
    """Function to get the instruments information for use in connecting
    to StrataSync
        hwOptions:
          type: array of strings
          description: hardware options on the mainframe
    """
    instrument_attributes = get_all_attributes()

    options_list = instrument_attributes.get('swOptionsInstalled', [])
    formatted_sw_options = format_options_for_stratasync(options_list)

    asset_info = {
        'serialNo': instrument_attributes.get('serialNo', 'unknown'),
        'assetType': instrument_attributes.get('family', 'unknown'),
        'uniqueId': instrument_attributes.get('uniqueId', 'unknown'),
        'model': instrument_attributes.get('model', 'unknown'),
        'hwVersion': instrument_attributes.get('hwVersion', 'unknown'),
        'swVersion': instrument_attributes.get('swVersion', 'unknown'),
        'calibrationDateIso': _convert_date_format(instrument_attributes.get('calDate')),
        'macAddress': instrument_attributes.get('macAddress', 'unknown'),
        'mfgDateIso': _convert_date_format(instrument_attributes.get('mfgDate')),
        'modulesInfo': instrument_attributes.get('modulesInfo', []),
        'swOptions': formatted_sw_options,
        'hwOptions': instrument_attributes.get('hwOptions', []),
        'techId': instrument_attributes.get('techId', 'unknown'),
    }

    custom_fields = make_custom_fields(instrument_attributes)
    asset_info.update(custom_fields)

    if (getDeployTimeValue() != 0):
        asset_info['fwDeployTime'] = getDeployTimeValue()

    return asset_info

def _convert_date_format(unformatted_date):
    """Function to format a date string from 'dd/mm/yyyy' to 'yyyy-mm-dd'

    args:
        date string formmatted as 'dd/mm/yyyy'

    returns:
        date string formatted as 'yyyy-mm-dd'
    """
    if not unformatted_date:
        return unformatted_date
    try:
        date = datetime.strptime(unformatted_date, "%d/%m/%Y")
    except ValueError:
        date = datetime.strptime('01/01/1970', "%d/%m/%Y")
    return date.strftime('%Y-%m-%d')

def format_options_for_stratasync(options_list):
    """Function to format the list of options into the format StrataSync wants
    Switches from a list of lists to a list of dictionaries

    See this url for documentation:
    https://conf1.ds.jdsu.net/wiki/pages/viewpage.action?spaceKey=UNIV&title=Assets+REST+API#AssetsRESTAPI-AddAsset

    args:
        option_list (list): a list of options as gathered from usc (only really valid for software  options?)
    """
    formatted_options = [
        format_sw_option(option)
        for option in options_list
    ]
    return formatted_options


def format_sw_option(option):
    """Function to tranlate from the list format from usc to the dictionary
    format requried by StrataSync

    args:
        option (list): the list format fromt usc.  Will have three elements
            for permanent and timed (demo) entries and 4 elelments for
            floating entries
    returns:
        formatted_option (dict): the dictionary in the StrataSync format
    """
    formatted_option = {
        'description': option[0],
        'name': option[1],
        'optionLicenseType': option[2]
    }

    if formatted_option['optionLicenseType'] == 'floating':
        formatted_option['expirationDate'] = option[3]

    return formatted_option


def make_custom_fields(instrument_attributes):
    """Function to create a custom fields dictionary for fields not used on all
    instruments

    Currently this function is hard coded to supply custom fields used on
    the 5800 becuase that is the only insturment that currently uses them

    args:
        instrument_attributes (dict): the dictionary of instrument attributes
        returned by `get_all_attributes`

    returns:
        custom_fields (dict): dictionary with custom fields if they are present
        else an empty dictionary
    """
    custom_fields = {}
    custom_field_keys = ['Board Info', 'Challenge ID']

    for key in custom_field_keys:
        value = instrument_attributes.get(key)
        if value:
            custom_fields[key] = value

    if not custom_fields:
        return {}
    return {'customFields': custom_fields}


def get_all_attributes():
    """Function to get all attributes needed for StrataSync

    returns:
        attributes (dict): StrataSync test
    """
    if not dbus:
        return {}

    system_bus = dbus.SystemBus()

    services = get_services_from_usc(system_bus)

    attributes = {}
    modules_info = []
    sw_options = []

    for service in services:
        mainframe_attributes = get_mainframe_attributes(system_bus, service)

        sw_options.extend(mainframe_attributes.pop('swOptionsInstalled', []))
        attributes.update(mainframe_attributes)

        modules_info.extend(get_modules_info(system_bus, service))

    attributes['modulesInfo'] = modules_info
    attributes['swOptionsInstalled'] = sw_options

    return attributes


def get_services_from_usc(system_bus):
    """Function to get the list of service endpoints known to the
    Univeral Sync Client (USC)

    args:
        system_bus (dbus.SystemBus) initialized instance of a system bus

    returns:
        services (list): list of strings each representing a service
    """

    usc = system_bus.get_object('com.jdsu.usc', '/Usc')
    services = usc.Services(dbus_interface='com.jdsu.usc')

    return services


def get_mainframe_attributes(system_bus, service):
    """Function to get attributes from the mainframe

    args:
        system_bus (dbus.SystemBus) initialized instance of a system bus
        service (str): the name of the service to query for attributes
    """
    try:
        mainframe = system_bus.get_object(service, '/Mainframe')
        attributes = mainframe.getAllAttributes(
            dbus_interface='com.jdsu.CommonAttributes'
        )
    except dbus.exceptions.DBusException:
        return {}
    except ValueError:
        return {}

    sw_options_str = (
        attributes.pop('swOptionsInstalled', '')
        or
        attributes.pop('swOptionsInstalledv2', '') #ISU on 5800 family
    )

    if sw_options_str:
        sw_options_list = split_options(sw_options_str)
        attributes['swOptionsInstalled'] = sw_options_list

    if 'hwOptionsInstalled' in attributes:
        hw_options = split_hw_options(attributes.pop('hwOptionsInstalled'))
        attributes['hwOptions'] = hw_options

    return attributes


def get_modules_info(system_bus, service):
    """Function to get the module information for the service

    args:
        system_bus (dbus.SystemBus) initialized instance of a system bus
        service (str): the name of the service to querry for attributes
    """
    try:
        mainframe = system_bus.get_object(service, '/Mainframe')
    except dbus.exceptions.DBusException:
        return []
    except ValueError: # 5800 reports .com.jdsu.Microscope which errors on the leading .
        return []

    try:
        modules = mainframe.listModules(
            dbus_interface='com.jdsu.Asset'
        )
    except dbus.exceptions.DBusException:
        # advisor com.viavisolutions.platform.Options doesn't have com.jdsu.Asset
        modules = []

    modules_info = []
    for module_path in modules:
        module = system_bus.get_object(service, module_path)
        try:
            attributes = module.getAllAttributes(
                dbus_interface='com.jdsu.CommonAttributes'
            )
        except dbus.exceptions.DBusException:
            continue

        if 'hwOptionsInstalled' in attributes:
            hw_options = split_hw_options(attributes.pop('hwOptionsInstalled'))
            attributes['hwOptions'] = hw_options

        reformat(attributes)

        modules_info.append(attributes)

    return modules_info


def split_options(options):
    """Function to split options from a string to a list
    """
    grouped_options = options.split(';;')
    separated_options = [option_group.split(';') for option_group in grouped_options]
    if [''] in separated_options:
        separated_options.remove([''])

    return separated_options

def split_hw_options(options):
    """Function to split options from a string to a list
    """
    separated_options = options.split(';')
    if [''] in separated_options:
        separated_options.remove([''])

    return separated_options


def reformat(attributes):
    """Function to reformat attributes to match what StrataSync expects
    This function will modify the dictionary passed as an argument

    Args:
        attributes (dict): dictionary of attributes formatted by the instrument
    """
    family = attributes.pop('family', '')
    if family:
        attributes['assetType'] = family

    calibration_date = attributes.pop('calDate', '')
    if calibration_date:
        attributes['calibrationDate'] = _convert_date_format(calibration_date)

    manufacturing_date = attributes.pop('mfgDate', '')
    if manufacturing_date:
        attributes['mfgDate'] = _convert_date_format(manufacturing_date)
