"""
Module that contains schemas for the config information of different tests to
launch on the T-BERD/MTS-4000 product
"""
from marshmallow import Schema, fields
from marshmallow.validate import OneOf

from .config_schema_common import ConfigBase, test_type_factory

class FiberInspectionSchema(Schema):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC-61300-3-35 Ed. 2.0)",
        "SM APC (IEC-61300-3-35 Ed. 2.0)",
        "MM  (IEC-61300-3-35 Ed. 2.0)",
        "SM PC & Fiber Stub (IEC-61300-3-35 Ed. 2.0)",
    )

    test_type = test_type_factory('Fiber Inspection')
    profile = fields.Str(
        load_from='profile',
        dump_to='profile',
        description='The profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )


class OTDR(Schema):
    """Schema for the configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    alarm_choices = (
        'Default',
        'TIA-568.3',
        'TIA-568.3 RL35',
        'ISO/IEC 11801 (2002)',
        'ISO/IEC 11801 (2010)',
        'ISO/IEC 14763-3 (2006)',
        'ISO/IEC 14763-3 (2014)',
        'G.697/G.98x PON',
        'G.697/IEEE PON',
        'User',
    )
    test_type = test_type_factory('OTDR')
    otdr_topology = fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        description='The distance of the launch cable used'
    )
    receive_cable = fields.Str(
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The distance of the receive cable used'
    )
    loop_cable = fields.Str(
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The distance of the loop cable used'
    )
    alarms = fields.Str(
        load_from='alarms',
        dump_to='alarms',
        validate=OneOf(alarm_choices),
        description='The path to a config file with alarm configurations'
    )
    labelFile = fields.Str(
        load_from='labelFile',
        dump_to='labelFile',
        x_uiPathHint='/user/disk/',
        description='The path to the .pro file containing labels'
    )
    config_file = fields.Str(
        required=False,
        load_from='configFile',
        dump_to='configFile',
        x_uiPathHint='/user/disk/',
        description=('the config file that needs to be ''loaded for the test at launch')
    )


CONFIG_SCHEMAS = {'OTDR': OTDR, 'Fiber Inspection': FiberInspectionSchema}


class ConfigSchema(ConfigBase):
    """
    Polymorphic mapping for different ConfigSchemas
    """
    type_schemas = CONFIG_SCHEMAS
