"""
Module that contains schemas for the rest api
"""
from marshmallow import Schema, fields
from marshmallow.validate import OneOf, Length
from .schemas_viavi import FilePathSchema, ArtifactSchema, ReferenceInfoSchema


UI_STRING_VALIDATOR = Length(max=60)


# class FilePathSchema(Schema):
#     """
#     Schema for for creating summary reports
#     """
#     file_path = fields.Str(required=True,
#                            load_from='filePath',
#                            dump_to='filePath',
#                            description=('The path to the file on the local'
#                                         ' file system'))
#
#
# class ArtifactSchema(FilePathSchema):
#     """
#     Schema for creating job artifacts such as summary reports, consolidated
#     reports, and zip files containing test reports and files
#     """
#     report_type = fields.Str(
#         load_from='reportType', dump_to='reportType', missing='summary', validate=OneOf(
#             ('summary', 'consolidated')), description=(
#                 'The desired type of report to create '
#                 'either a brief summary report or a '
#                 'consolidated report with all PDF files '
#                 'in the job appended to a summary report'))
#     create_zip = fields.Bool(
#         load_from='createZip', dump_to='createZip', missing=False, description=(
#             'Whether or not to create a zip file containing the '
#             'summary or consolidated report and all of the test '
#             'data files in the job'))

def make_job_schema(config_schema=None):
    """Job Schema factory function to create a job schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    class JobSchema(Schema):
        """
        Schema for the job object

        """
        job_number = fields.Str(
            load_from='jobNumber',
            dump_to='jobNumber',
            validate=UI_STRING_VALIDATOR,
            missing='--',
            description='string representation of the job number')
        customer_name = fields.Str(
            required=True,
            load_from='customerName',
            dump_to='customerName',
            validate=UI_STRING_VALIDATOR,
            description='the name of the end customer for the job')
        technician_id = fields.Str(
            load_from='technicianId',
            dump_to='technicianId',
            validate=UI_STRING_VALIDATOR,
            missing='--',
            description="the technician's identification number")
        test_location = fields.Str(
            load_from='testLocation',
            dump_to='testLocation',
            missing='--',
            validate=UI_STRING_VALIDATOR,
            description='the location where the test is being performed')
        logo = fields.Str(
            required=False,
            load_from='logo',
            dump_to='logo',
            missing="",
            description='the path to the logo that will be added to a report')
        test_data = fields.Nested(
            'TestDataSchemaB',
            load_from='testData',
            dump_to='testData',
            many=True,
            description='list containing zero or more test_data dictionaires')
        test_plan = fields.Nested(
            make_planned_test_schema(config_schema),
            load_from='testPlan',
            dump_to='testPlan',
            many=True,
            description='list containing zero or more planned tests for the job')
        workflow_id = fields.Integer(
            required=False,
            missing=0,
            load_from="workflowId",
            dump_to="workflowId",
            description='integer, id of the workflow')
        cdm_string = fields.Str(
            required=False,
            load_from='cdmString',
            dump_to='cdmString',
            description='the cdm string')

    return JobSchema

class TestDataSchemaB(Schema):
    """
    Schema for test data entries

    """
    test_type = fields.Str(
        required=True,
        load_from='testType',
        dump_to='testType',
        description='the type of test that created this test data')
    file_path = fields.Str(
        required=True,
        load_from='filePath',
        dump_to='filePath',
        description='the path to the underlying file for this test data entry')
    verdict = fields.Str(required=True,
                         load_from='verdict',
                         dump_to='verdict',
                         validate=OneOf(
                             ('Pass', 'Fail', 'Marginal', 'NotApplicable', 'Skipped')
                         ),
                         description='the verdict of the test')
    comments = fields.Str(
        required=False,
        load_from='comments',
        dump_to='comments',
        description='any comments on the particular test item')
    reference_info = fields.Nested(
        'ReferenceInfoSchema',
        required=False,
        many=True,
        load_from='referenceInfo',
        dump_to='referenceInfo',
        description=(
            'list of zero or more dictionaries containing '
            'reference info differentiationg this test data '
            'from others like it'))

def make_planned_test_schema(config_schema):
    """Planned test Schema factory function to create a planned test schema class
    based on the config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    def make_config(config_schema):
        if not config_schema:
            return fields.Dict(allow_none=True)
        return fields.Nested(
            config_schema,
            allow_none=True,
            required=False,
            load_from='config',
            dump_to='config',
            description='the configuration of the particular test to be launched'
        )
    class PlannedTestOldSchema(Schema):
        """
        Schema for tests that are planned to be performed as part of a
        test plan

        """
        test_type = fields.Str(required=True,
                               load_from='testType',
                               dump_to='testType',
                               description='the type of test required')
        status = fields.Str(
            missing='To Do',
            load_from='status',
            dump_to='status',
            validate=OneOf(
                ('To Do',
                 'Pass',
                 'Fail',
                 'Marginal',
                 'Complete',
                 'Skipped')),
            description='the status of the test')
        reference_info = fields.Nested(
            'ReferenceInfoSchema',
            required=False,
            many=True,
            load_from='referenceInfo',
            dump_to='referenceInfo',
            missing=[],
            description=(
                'list of zero or more dictionaries containin'
                'reference info differentiationg this test data '
                'from others like it'))
        reference_info_str = fields.Method(
            serialize='make_reference_info_str',
            dump_to='referenceInfoStr',
            description=(
                'the string representation'
                ' of the reference information '
                'can be used to display on user interfaces '
                'ignored on POST and PUT requests'))
        procedures = fields.Str(
            required=False,
            missing='',
            load_from='procedures',
            dump_to='procedures',
            description='path to method and procedures for this planned test')
        test_data = fields.Nested(
            TestDataSchemaB,
            missing=[],
            many=True,
            load_from='testData',
            dump_to='testData',
            description='list containing zero or more test_data dictionaires')
        config = make_config(config_schema)

        def make_reference_info_str(self, obj):  # pylint: disable=no-self-use
            """Method to create the reference information string based on the
            reference_info list in the PlannedTest object

            Args:
                obj (PlannedTest object): the required object input for method
                    fields

            Returns:
                reference_info_str (str): the string representation of the
                    reference info
            """
            reference_info_strings = ('{}: {}'.format(item['key'], item['value'])
                                      for item in obj.reference_info)
            reference_info_str = ' , '.join(reference_info_strings)
            return reference_info_str

    return PlannedTestOldSchema


# class ReferenceInfoSchema(Schema):
#     """
#     Schema for reference information for the test
#     each test may have zero or more reference information
#     entries to differentiate it from other tests of the same type
#
#     """
#     key = fields.Str(required=True,
#                      load_from='key',
#                      dump_to='key',
#                      validate=UI_STRING_VALIDATOR,
#                      description='the field name for this reference')
#     value = fields.Str(
#         required=True,
#         load_from='value',
#         dump_to='value',
#         validate=UI_STRING_VALIDATOR,
#         description='the entered value for this particular field')
